/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.computer.core.sender;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hugegraph.computer.core.common.exception.ComputerException;
import org.apache.hugegraph.computer.core.common.exception.TransportException;
import org.apache.hugegraph.computer.core.config.ComputerOptions;
import org.apache.hugegraph.computer.core.config.Config;
import org.apache.hugegraph.computer.core.network.TransportClient;
import org.apache.hugegraph.computer.core.network.message.MessageType;
import org.apache.hugegraph.computer.core.sender.MessageQueue;
import org.apache.hugegraph.computer.core.sender.MessageSender;
import org.apache.hugegraph.computer.core.sender.QueuedMessage;
import org.apache.hugegraph.concurrent.BarrierEvent;
import org.apache.hugegraph.config.TypedOption;
import org.apache.hugegraph.util.E;
import org.apache.hugegraph.util.Log;
import org.slf4j.Logger;

public class QueuedMessageSender
implements MessageSender {
    public static final Logger LOG = Log.logger(QueuedMessageSender.class);
    private static final String NAME = "send-executor";
    private final WorkerChannel[] channels;
    private final Thread sendExecutor;
    private final BarrierEvent anyQueueNotEmptyEvent;
    private final BarrierEvent anyClientNotBusyEvent;

    public QueuedMessageSender(Config config) {
        int workerCount = (Integer)config.get((TypedOption)ComputerOptions.JOB_WORKERS_COUNT);
        this.channels = new WorkerChannel[workerCount];
        this.sendExecutor = new Thread((Runnable)new Sender(), NAME);
        this.anyQueueNotEmptyEvent = new BarrierEvent();
        this.anyClientNotBusyEvent = new BarrierEvent();
    }

    public void init() {
        for (WorkerChannel channel : this.channels) {
            E.checkNotNull((Object)channel, (String)"channel");
        }
        this.sendExecutor.start();
    }

    public void close() {
        this.sendExecutor.interrupt();
        try {
            this.sendExecutor.join();
        }
        catch (InterruptedException e) {
            throw new ComputerException("Interrupted when waiting for send-executor to stop", (Throwable)e);
        }
    }

    public void addWorkerClient(int workerId, TransportClient client) {
        WorkerChannel channel;
        MessageQueue queue = new MessageQueue(() -> ((BarrierEvent)this.anyQueueNotEmptyEvent).signal());
        this.channels[QueuedMessageSender.channelId((int)workerId)] = channel = new WorkerChannel(workerId, queue, client);
        LOG.info("Add client {} for worker {}", (Object)client.connectionId(), (Object)workerId);
    }

    @Override
    public CompletableFuture<Void> send(int workerId, MessageType type) throws InterruptedException {
        WorkerChannel channel = this.channels[QueuedMessageSender.channelId(workerId)];
        CompletableFuture<Void> future = channel.newFuture();
        future.whenComplete((r, e) -> channel.resetFuture(future));
        channel.queue.put(new QueuedMessage(-1, type, null));
        return future;
    }

    @Override
    public void send(int workerId, QueuedMessage message) throws InterruptedException {
        WorkerChannel channel = this.channels[QueuedMessageSender.channelId(workerId)];
        channel.queue.put(message);
    }

    public Runnable notBusyNotifier() {
        return () -> ((BarrierEvent)this.anyClientNotBusyEvent).signal();
    }

    private void waitAnyQueueNotEmpty() {
        try {
            this.anyQueueNotEmptyEvent.await();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        finally {
            this.anyQueueNotEmptyEvent.reset();
        }
    }

    private void waitAnyClientNotBusy() {
        try {
            this.anyClientNotBusyEvent.await();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ComputerException("Interrupted when waiting any client not busy");
        }
        finally {
            this.anyClientNotBusyEvent.reset();
        }
    }

    private int activeClientCount() {
        int count = 0;
        for (WorkerChannel channel : this.channels) {
            if (!channel.client.sessionActive()) continue;
            ++count;
        }
        return count;
    }

    private static int channelId(int workerId) {
        assert (workerId > 0);
        return workerId - 1;
    }

    private static class WorkerChannel {
        private final int workerId;
        private final MessageQueue queue;
        private final TransportClient client;
        private final AtomicReference<CompletableFuture<Void>> futureRef;

        public WorkerChannel(int workerId, MessageQueue queue, TransportClient client) {
            this.workerId = workerId;
            this.queue = queue;
            this.client = client;
            this.futureRef = new AtomicReference();
        }

        public CompletableFuture<Void> newFuture() {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            if (!this.futureRef.compareAndSet(null, future)) {
                throw new ComputerException("The origin future must be null");
            }
            return future;
        }

        public void resetFuture(CompletableFuture<Void> future) {
            if (!this.futureRef.compareAndSet(future, null)) {
                throw new ComputerException("Failed to reset futureRef, expect future object is %s, but some thread modified it", new Object[]{future});
            }
        }

        public boolean doSend(QueuedMessage message) throws TransportException, InterruptedException {
            switch (message.type()) {
                case START: {
                    this.sendStartMessage();
                    return true;
                }
                case FINISH: {
                    this.sendFinishMessage();
                    return true;
                }
            }
            return this.sendDataMessage(message);
        }

        public void sendStartMessage() throws TransportException {
            this.client.startSessionAsync().whenComplete((r, e) -> {
                CompletableFuture<Void> future = this.futureRef.get();
                assert (future != null);
                if (e != null) {
                    LOG.info("Failed to start session connected to {}", (Object)this);
                    future.completeExceptionally((Throwable)e);
                } else {
                    LOG.info("Start session connected to {}", (Object)this);
                    future.complete(null);
                }
            });
        }

        public void sendFinishMessage() throws TransportException {
            this.client.finishSessionAsync().whenComplete((r, e) -> {
                CompletableFuture<Void> future = this.futureRef.get();
                assert (future != null);
                if (e != null) {
                    LOG.info("Failed to finish session connected to {}", (Object)this);
                    future.completeExceptionally((Throwable)e);
                } else {
                    LOG.info("Finish session connected to {}", (Object)this);
                    future.complete(null);
                }
            });
        }

        public boolean sendDataMessage(QueuedMessage message) throws TransportException {
            return this.client.send(message.type(), message.partitionId(), message.buffer());
        }

        public String toString() {
            return String.format("workerId=%s(remoteAddress=%s)", this.workerId, this.client.remoteAddress());
        }
    }

    private class Sender
    implements Runnable {
        private Sender() {
        }

        @Override
        public void run() {
            LOG.info("The send-executor is running");
            Thread thread = Thread.currentThread();
            while (!thread.isInterrupted()) {
                try {
                    int emptyQueueCount = 0;
                    int busyClientCount = 0;
                    for (WorkerChannel channel : QueuedMessageSender.this.channels) {
                        QueuedMessage message = channel.queue.peek();
                        if (message == null) {
                            ++emptyQueueCount;
                            continue;
                        }
                        if (channel.doSend(message)) {
                            channel.queue.take();
                            continue;
                        }
                        ++busyClientCount;
                    }
                    int channelCount = QueuedMessageSender.this.channels.length;
                    if (emptyQueueCount >= channelCount) {
                        LOG.debug("The send executor was blocked to wait any queue not empty");
                        QueuedMessageSender.this.waitAnyQueueNotEmpty();
                    }
                    if (busyClientCount < channelCount) continue;
                    LOG.debug("The send executor was blocked to wait any client not busy");
                    QueuedMessageSender.this.waitAnyClientNotBusy();
                }
                catch (InterruptedException e) {
                    thread.interrupt();
                    if (QueuedMessageSender.this.activeClientCount() <= 0) continue;
                    throw new ComputerException("Interrupted when waiting for message queue not empty");
                }
                catch (TransportException e) {
                    throw new ComputerException("Failed to send message", (Throwable)e);
                }
            }
            LOG.info("The send-executor is terminated");
        }
    }
}

