/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.kylin.rest.controller;

import static org.apache.kylin.common.constant.HttpConstant.HTTP_VND_APACHE_KYLIN_JSON;
import static org.apache.kylin.common.constant.HttpConstant.HTTP_VND_APACHE_KYLIN_V4_PUBLIC_JSON;

import java.nio.ByteBuffer;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.kylin.common.exception.KylinException;
import org.apache.kylin.metadata.streaming.KafkaConfig;
import org.apache.kylin.rest.request.StreamingRequest;
import org.apache.kylin.rest.response.EnvelopeResponse;
import org.apache.kylin.rest.service.KafkaService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

@Controller
@RequestMapping(value = "/api/kafka", produces = { HTTP_VND_APACHE_KYLIN_JSON, HTTP_VND_APACHE_KYLIN_V4_PUBLIC_JSON })
public class KafkaController extends NBasicController {

    @Autowired(required = false)
    @Qualifier("kafkaService")
    private KafkaService kafkaService;

    @PostMapping(value = "topics", produces = { HTTP_VND_APACHE_KYLIN_JSON, HTTP_VND_APACHE_KYLIN_V4_PUBLIC_JSON })
    @ResponseBody
    public EnvelopeResponse<Map<String, List<String>>> getTopics(@RequestBody StreamingRequest streamingRequest) {
        checkStreamingEnabled();
        KafkaConfig kafkaConfig = streamingRequest.getKafkaConfig();
        return new EnvelopeResponse<>(KylinException.CODE_SUCCESS,
                kafkaService.getTopics(kafkaConfig, streamingRequest.getProject(), streamingRequest.getFuzzyKey()), "");
    }

    @PostMapping(value = "messages", produces = { HTTP_VND_APACHE_KYLIN_JSON, HTTP_VND_APACHE_KYLIN_V4_PUBLIC_JSON })
    @ResponseBody
    public EnvelopeResponse getMessages(@RequestBody StreamingRequest streamingRequest) {
        checkStreamingEnabled();
        String projectName = checkProjectName(streamingRequest.getProject());
        List<ByteBuffer> messages = kafkaService.getMessages(streamingRequest.getKafkaConfig(), projectName);
        if (CollectionUtils.isEmpty(messages)) {
            return new EnvelopeResponse<>(KylinException.CODE_SUCCESS, "", "There is no message in this topic");
        }
        Map<String, Object> resp = kafkaService.decodeMessage(messages);
        return new EnvelopeResponse<>(KylinException.CODE_SUCCESS, resp, "");
    }

    @PostMapping(value = "convert", produces = { HTTP_VND_APACHE_KYLIN_JSON, HTTP_VND_APACHE_KYLIN_V4_PUBLIC_JSON })
    @ResponseBody
    public EnvelopeResponse<Map<String, Object>> convertMessage(@RequestBody StreamingRequest streamingRequest) {
        checkStreamingEnabled();
        String projectName = checkProjectName(streamingRequest.getProject());
        String message = streamingRequest.getMessage();
        KafkaConfig kafkaConfig = streamingRequest.getKafkaConfig();
        Map<String, Object> result = kafkaService.parserMessage(projectName, kafkaConfig, message);
        return new EnvelopeResponse<>(KylinException.CODE_SUCCESS, result, "");
    }

    @GetMapping(value = "parsers", produces = { HTTP_VND_APACHE_KYLIN_JSON, HTTP_VND_APACHE_KYLIN_V4_PUBLIC_JSON })
    @ResponseBody
    public EnvelopeResponse<List<String>> getParser(@RequestParam("project") String project) {
        checkStreamingEnabled();
        String projectName = checkProjectName(project);
        List<String> classList = kafkaService.getParsers(projectName);
        return new EnvelopeResponse<>(KylinException.CODE_SUCCESS, classList, "");
    }

    @DeleteMapping(value = "parser", produces = { HTTP_VND_APACHE_KYLIN_JSON, HTTP_VND_APACHE_KYLIN_V4_PUBLIC_JSON })
    @ResponseBody
    public EnvelopeResponse<String> removeParser(@RequestParam("project") String project,
            @RequestParam("class_name") String className) {
        checkStreamingEnabled();
        String projectName = checkProjectName(project);
        String removedClassName = kafkaService.removeParser(projectName, className);
        return new EnvelopeResponse<>(KylinException.CODE_SUCCESS, removedClassName, "");
    }
}
