/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.conversation;

import java.util.HashMap;
import java.util.Map;

/**
 * Maintains a list of aspects the system might attach to a conversation instance.
 * <p>
 * Aspects are a way of extending the functionality of a class without modifying it;
 * it acts something like the "decorator" pattern, but with many different decorator
 * classes being supported at once.
 * <p>
 * In the simplest form, an Aspect can be attached to a Conversation simply as a "marker"
 * to indicate whether the conversation should be treated in a certain way or not.
 * <p>
 * In more sophisticated form, an Aspect can provide an API. Code that wants to
 * manipulate some "conversation-related" property can query the conversation for the
 * appropriate Aspect object, then invoke the aspect api to read or write the
 * Conversation instance via an implementation that is isolated from both the caller
 * and the Conversation.
 * <p>
 * Examples of aspects that can be attached to a conversation are:
 * <ul>
 *     <li>Timeout handling</li>
 *     <li>Access-scope handling</li>
 * </ul>
 */
public class ConversationAspects
{
    private Map conversationAspects = new HashMap();

    public void addAspect(ConversationAspect aspect)
    {
        conversationAspects.put(aspect.getClass().getName(), aspect);
    }

    public ConversationAspect getAspect(Class aspectClass)
    {
        return (ConversationAspect) conversationAspects.get(aspectClass.getName());
    }
}
