/****************************************************************************
 * lib/lib_alarm.c
 *
 *   Copyright (C) 2019 Xiaomi Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name NuttX nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <unistd.h>
#include <sys/time.h>

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Name: alarm
 *
 * Description:
 *   The alarm() function will arranges for a SIGALRM signal to be delivered
 *   to the calling process in seconds seconds.
 *
 *   Interactions between alarm() and any of setitimer() are unspecified.
 *
 * Input Parameters:
 *   seconds - The number of seconds to alarm
 *
 * Returned Value:
 *   alarm() returns the number of seconds remaining until any previously
 *   scheduled alarm was due to be delivered, or zero if there was no
 *   previously scheduled alarm.
 *
 * Assumptions:
 *
 ****************************************************************************/

unsigned int alarm(unsigned int seconds)
{
  struct itimerval value =
  {
    {0, 0},
    {0, 0}
  };

  struct itimerval ovalue =
  {
    {0, 0},
    {0, 0}
  };

  value.it_value.tv_sec = seconds;
  setitimer(ITIMER_REAL, &value, &ovalue);

  return ovalue.it_value.tv_sec;
}
