/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.client.ECReplicationConfig;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.ContainerReplicaNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerStateManager;
import org.apache.hadoop.hdds.scm.container.ContainerStateManagerImpl;
import org.apache.hadoop.hdds.scm.container.metrics.SCMContainerManagerMetrics;
import org.apache.hadoop.hdds.scm.container.replication.ContainerReplicaPendingOps;
import org.apache.hadoop.hdds.scm.ha.SCMHAManager;
import org.apache.hadoop.hdds.scm.ha.SequenceIdGenerator;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineManager;
import org.apache.hadoop.hdds.utils.CollectionUtils;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.ozone.common.statemachine.InvalidStateTransitionException;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerManagerImpl
implements ContainerManager {
    private static final Logger LOG = LoggerFactory.getLogger(ContainerManagerImpl.class);
    private final Lock lock;
    private final PipelineManager pipelineManager;
    private final ContainerStateManager containerStateManager;
    private final SCMHAManager haManager;
    private final SequenceIdGenerator sequenceIdGen;
    private final SCMContainerManagerMetrics scmContainerManagerMetrics;
    private final int numContainerPerVolume;
    private final Random random = new Random();

    public ContainerManagerImpl(Configuration conf, SCMHAManager scmHaManager, SequenceIdGenerator sequenceIdGen, PipelineManager pipelineManager, Table<ContainerID, ContainerInfo> containerStore, ContainerReplicaPendingOps containerReplicaPendingOps) throws IOException {
        this.lock = new ReentrantLock();
        this.pipelineManager = pipelineManager;
        this.haManager = scmHaManager;
        this.sequenceIdGen = sequenceIdGen;
        this.containerStateManager = ContainerStateManagerImpl.newBuilder().setConfiguration(conf).setPipelineManager(pipelineManager).setRatisServer(scmHaManager.getRatisServer()).setContainerStore(containerStore).setSCMDBTransactionBuffer(scmHaManager.getDBTransactionBuffer()).setContainerReplicaPendingOps(containerReplicaPendingOps).build();
        this.numContainerPerVolume = conf.getInt("ozone.scm.pipeline.owner.container.count", 3);
        this.scmContainerManagerMetrics = SCMContainerManagerMetrics.create();
    }

    @Override
    public void reinitialize(Table<ContainerID, ContainerInfo> containerStore) throws IOException {
        this.lock.lock();
        try {
            this.containerStateManager.reinitialize(containerStore);
        }
        catch (IOException ioe) {
            LOG.error("Failed to reinitialize containerManager", (Throwable)ioe);
            throw ioe;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public ContainerInfo getContainer(ContainerID id) throws ContainerNotFoundException {
        return Optional.ofNullable(this.containerStateManager.getContainer(id)).orElseThrow(() -> new ContainerNotFoundException("ID " + id));
    }

    @Override
    public List<ContainerInfo> getContainers(ContainerID startID, int count) {
        this.scmContainerManagerMetrics.incNumListContainersOps();
        return this.toContainers(ContainerManagerImpl.filterSortAndLimit(startID, count, this.containerStateManager.getContainerIDs()));
    }

    @Override
    public List<ContainerInfo> getContainers(HddsProtos.LifeCycleState state) {
        this.scmContainerManagerMetrics.incNumListContainersOps();
        return this.toContainers(this.containerStateManager.getContainerIDs(state));
    }

    @Override
    public List<ContainerInfo> getContainers(ContainerID startID, int count, HddsProtos.LifeCycleState state) {
        this.scmContainerManagerMetrics.incNumListContainersOps();
        return this.toContainers(ContainerManagerImpl.filterSortAndLimit(startID, count, this.containerStateManager.getContainerIDs(state)));
    }

    @Override
    public int getContainerStateCount(HddsProtos.LifeCycleState state) {
        return this.containerStateManager.getContainerIDs(state).size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ContainerInfo allocateContainer(ReplicationConfig replicationConfig, String owner) throws IOException {
        ContainerInfo containerInfo;
        block12: {
            Pipeline pipeline;
            List<Pipeline> pipelines;
            this.pipelineManager.acquireReadLock();
            this.lock.lock();
            containerInfo = null;
            try {
                pipelines = this.pipelineManager.getPipelines(replicationConfig, Pipeline.PipelineState.OPEN);
                if (!pipelines.isEmpty()) {
                    pipeline = pipelines.get(this.random.nextInt(pipelines.size()));
                    containerInfo = this.createContainer(pipeline, owner);
                }
            }
            finally {
                this.lock.unlock();
                this.pipelineManager.releaseReadLock();
            }
            if (pipelines.isEmpty()) {
                try {
                    pipeline = this.pipelineManager.createPipeline(replicationConfig);
                    this.pipelineManager.waitPipelineReady(pipeline.getId(), 0L);
                }
                catch (IOException e) {
                    this.scmContainerManagerMetrics.incNumFailureCreateContainers();
                    throw new IOException("Could not allocate container. Cannot get any matching pipeline for replicationConfig: " + replicationConfig + ", State:PipelineState.OPEN", e);
                }
                this.pipelineManager.acquireReadLock();
                this.lock.lock();
                try {
                    pipelines = this.pipelineManager.getPipelines(replicationConfig, Pipeline.PipelineState.OPEN);
                    if (!pipelines.isEmpty()) {
                        pipeline = pipelines.get(this.random.nextInt(pipelines.size()));
                        containerInfo = this.createContainer(pipeline, owner);
                        break block12;
                    }
                    throw new IOException("Could not allocate container. Cannot get any matching pipeline for replicationConfig: " + replicationConfig + ", State:PipelineState.OPEN");
                }
                finally {
                    this.lock.unlock();
                    this.pipelineManager.releaseReadLock();
                }
            }
        }
        return containerInfo;
    }

    private ContainerInfo createContainer(Pipeline pipeline, String owner) throws IOException {
        ContainerInfo containerInfo = this.allocateContainer(pipeline, owner);
        if (LOG.isTraceEnabled()) {
            LOG.trace("New container allocated: {}", (Object)containerInfo);
        }
        return containerInfo;
    }

    private ContainerInfo allocateContainer(Pipeline pipeline, String owner) throws IOException {
        long uniqueId = this.sequenceIdGen.getNextId("containerId");
        Preconditions.checkState((uniqueId > 0L ? 1 : 0) != 0, (String)"Cannot allocate container, negative container id generated. %s.", (long)uniqueId);
        ContainerID containerID = ContainerID.valueOf((long)uniqueId);
        HddsProtos.ContainerInfoProto.Builder containerInfoBuilder = HddsProtos.ContainerInfoProto.newBuilder().setState(HddsProtos.LifeCycleState.OPEN).setPipelineID(pipeline.getId().getProtobuf()).setUsedBytes(0L).setNumberOfKeys(0L).setStateEnterTime(Time.now()).setOwner(owner).setContainerID(containerID.getId()).setDeleteTransactionId(0L).setReplicationType(pipeline.getType());
        if (pipeline.getReplicationConfig() instanceof ECReplicationConfig) {
            containerInfoBuilder.setEcReplicationConfig(((ECReplicationConfig)pipeline.getReplicationConfig()).toProto());
        } else {
            containerInfoBuilder.setReplicationFactor(ReplicationConfig.getLegacyFactor((ReplicationConfig)pipeline.getReplicationConfig()));
        }
        this.containerStateManager.addContainer(containerInfoBuilder.build());
        this.scmContainerManagerMetrics.incNumSuccessfulCreateContainers();
        return this.containerStateManager.getContainer(containerID);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void updateContainerState(ContainerID cid, HddsProtos.LifeCycleEvent event) throws IOException, InvalidStateTransitionException {
        HddsProtos.ContainerID protoId = cid.getProtobuf();
        this.lock.lock();
        try {
            if (this.containerExist(cid)) {
                this.containerStateManager.updateContainerState(protoId, event);
            } else {
                this.throwContainerNotFoundException(cid);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public Set<ContainerReplica> getContainerReplicas(ContainerID id) throws ContainerNotFoundException {
        return Optional.ofNullable(this.containerStateManager.getContainerReplicas(id)).orElseThrow(() -> new ContainerNotFoundException("ID " + id));
    }

    @Override
    public void updateContainerReplica(ContainerID cid, ContainerReplica replica) throws ContainerNotFoundException {
        if (this.containerExist(cid)) {
            this.containerStateManager.updateContainerReplica(cid, replica);
        } else {
            this.throwContainerNotFoundException(cid);
        }
    }

    @Override
    public void removeContainerReplica(ContainerID cid, ContainerReplica replica) throws ContainerNotFoundException, ContainerReplicaNotFoundException {
        if (this.containerExist(cid)) {
            this.containerStateManager.removeContainerReplica(cid, replica);
        } else {
            this.throwContainerNotFoundException(cid);
        }
    }

    @Override
    public void updateDeleteTransactionId(Map<ContainerID, Long> deleteTransactionMap) throws IOException {
        this.containerStateManager.updateDeleteTransactionId(deleteTransactionMap);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ContainerInfo getMatchingContainer(long size, String owner, Pipeline pipeline, Set<ContainerID> excludedContainerIDs) {
        try {
            PipelineID pipelineID = pipeline.getId();
            synchronized (pipelineID) {
                NavigableSet<ContainerID> containerIDs = this.getContainersForOwner(pipeline, owner);
                if (containerIDs.size() < this.getOpenContainerCountPerPipeline(pipeline)) {
                    this.allocateContainer(pipeline, owner);
                    containerIDs = this.getContainersForOwner(pipeline, owner);
                }
                containerIDs.removeAll(excludedContainerIDs);
                ContainerInfo containerInfo = this.containerStateManager.getMatchingContainer(size, owner, pipeline.getId(), containerIDs);
                if (containerInfo == null) {
                    containerInfo = this.allocateContainer(pipeline, owner);
                }
                return containerInfo;
            }
        }
        catch (Exception e) {
            LOG.warn("Container allocation failed on pipeline={}", (Object)pipeline, (Object)e);
            return null;
        }
    }

    private int getOpenContainerCountPerPipeline(Pipeline pipeline) {
        int minContainerCountPerDn = this.numContainerPerVolume * this.pipelineManager.minHealthyVolumeNum(pipeline);
        int minPipelineCountPerDn = this.pipelineManager.minPipelineLimit(pipeline);
        return (int)Math.ceil((double)minContainerCountPerDn / (double)minPipelineCountPerDn);
    }

    private NavigableSet<ContainerID> getContainersForOwner(Pipeline pipeline, String owner) throws IOException {
        NavigableSet<ContainerID> containerIDs = this.pipelineManager.getContainersInPipeline(pipeline.getId());
        Iterator<ContainerID> containerIDIterator = containerIDs.iterator();
        while (containerIDIterator.hasNext()) {
            ContainerID cid = containerIDIterator.next();
            try {
                if (this.getContainer(cid).getOwner().equals(owner)) continue;
                containerIDIterator.remove();
            }
            catch (ContainerNotFoundException e) {
                LOG.error("Could not find container info for container {}", (Object)cid, (Object)e);
                containerIDIterator.remove();
            }
        }
        return containerIDs;
    }

    @Override
    public void notifyContainerReportProcessing(boolean isFullReport, boolean success) {
        if (isFullReport) {
            if (success) {
                this.scmContainerManagerMetrics.incNumContainerReportsProcessedSuccessful();
            } else {
                this.scmContainerManagerMetrics.incNumContainerReportsProcessedFailed();
            }
        } else if (success) {
            this.scmContainerManagerMetrics.incNumICRReportsProcessedSuccessful();
        } else {
            this.scmContainerManagerMetrics.incNumICRReportsProcessedFailed();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void deleteContainer(ContainerID cid) throws IOException {
        boolean found;
        HddsProtos.ContainerID protoId = cid.getProtobuf();
        this.lock.lock();
        try {
            found = this.containerExist(cid);
            if (found) {
                this.containerStateManager.removeContainer(protoId);
            }
        }
        finally {
            this.lock.unlock();
        }
        if (found) {
            this.scmContainerManagerMetrics.incNumSuccessfulDeleteContainers();
        } else {
            this.scmContainerManagerMetrics.incNumFailureDeleteContainers();
            this.throwContainerNotFoundException(cid);
        }
    }

    @Override
    public boolean containerExist(ContainerID id) {
        return this.containerStateManager.contains(id);
    }

    private void throwContainerNotFoundException(ContainerID id) throws ContainerNotFoundException {
        throw new ContainerNotFoundException("Container with id " + id + " not found.");
    }

    @Override
    public void close() throws IOException {
        this.containerStateManager.close();
    }

    @Override
    @VisibleForTesting
    public ContainerStateManager getContainerStateManager() {
        return this.containerStateManager;
    }

    @VisibleForTesting
    public SCMHAManager getSCMHAManager() {
        return this.haManager;
    }

    private static List<ContainerID> filterSortAndLimit(ContainerID startID, int count, Set<ContainerID> set) {
        if (ContainerID.MIN.equals((Object)startID) && count >= set.size()) {
            ArrayList<ContainerID> list = new ArrayList<ContainerID>(set);
            Collections.sort(list);
            return list;
        }
        return CollectionUtils.findTopN(set, (int)count, Comparator.reverseOrder(), id -> id.compareTo(startID) >= 0);
    }

    private List<ContainerInfo> toContainers(Collection<ContainerID> ids) {
        ArrayList<ContainerInfo> containers = new ArrayList<ContainerInfo>(ids.size());
        for (ContainerID id : ids) {
            ContainerInfo container = this.containerStateManager.getContainer(id);
            if (container == null) continue;
            containers.add(container);
        }
        return containers;
    }
}

