/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.pipeline;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.pipeline.DuplicatedPipelineIdException;
import org.apache.hadoop.hdds.scm.pipeline.InvalidPipelineStateException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class PipelineStateMap {
    private static final Logger LOG = LoggerFactory.getLogger(PipelineStateMap.class);
    private final Map<PipelineID, Pipeline> pipelineMap = new ConcurrentHashMap<PipelineID, Pipeline>();
    private final Map<PipelineID, NavigableSet<ContainerID>> pipeline2container = new ConcurrentHashMap<PipelineID, NavigableSet<ContainerID>>();
    private final Map<ReplicationConfig, List<Pipeline>> query2OpenPipelines = new HashMap<ReplicationConfig, List<Pipeline>>();

    PipelineStateMap() {
    }

    void addPipeline(Pipeline pipeline) throws IOException {
        Preconditions.checkNotNull((Object)pipeline, (Object)"Pipeline cannot be null");
        Preconditions.checkArgument((pipeline.getNodes().size() == pipeline.getReplicationConfig().getRequiredNodes() ? 1 : 0) != 0, (String)"Nodes size=%s, replication factor=%s do not match ", (int)pipeline.getNodes().size(), (int)pipeline.getReplicationConfig().getRequiredNodes());
        if (this.pipelineMap.putIfAbsent(pipeline.getId(), pipeline) != null) {
            LOG.warn("Duplicate pipeline ID detected. {}", (Object)pipeline.getId());
            throw new DuplicatedPipelineIdException(String.format("Duplicate pipeline ID %s detected.", pipeline.getId()));
        }
        this.pipeline2container.put(pipeline.getId(), new TreeSet());
        if (pipeline.getPipelineState() == Pipeline.PipelineState.OPEN) {
            this.query2OpenPipelines.computeIfAbsent(pipeline.getReplicationConfig(), any -> new CopyOnWriteArrayList()).add(pipeline);
        }
    }

    void addContainerToPipeline(PipelineID pipelineID, ContainerID containerID) throws IOException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Preconditions.checkNotNull((Object)containerID, (Object)"Container Id cannot be null");
        Pipeline pipeline = this.getPipeline(pipelineID);
        if (pipeline.isClosed()) {
            throw new InvalidPipelineStateException(String.format("Cannot add container to pipeline=%s in closed state", pipelineID));
        }
        this.pipeline2container.get(pipelineID).add(containerID);
    }

    void addContainerToPipelineSCMStart(PipelineID pipelineID, ContainerID containerID) throws IOException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Preconditions.checkNotNull((Object)containerID, (Object)"Container Id cannot be null");
        Pipeline pipeline = this.getPipeline(pipelineID);
        if (pipeline.isClosed()) {
            LOG.info("Container {} in open state for pipeline={} in closed state", (Object)containerID, (Object)pipelineID);
        }
        this.pipeline2container.get(pipelineID).add(containerID);
    }

    Pipeline getPipeline(PipelineID pipelineID) throws PipelineNotFoundException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Pipeline pipeline = this.pipelineMap.get(pipelineID);
        if (pipeline == null) {
            throw new PipelineNotFoundException(String.format("%s not found", pipelineID));
        }
        return pipeline;
    }

    public List<Pipeline> getPipelines() {
        return new ArrayList<Pipeline>(this.pipelineMap.values());
    }

    List<Pipeline> getPipelines(ReplicationConfig replicationConfig) {
        Preconditions.checkNotNull((Object)replicationConfig, (Object)"ReplicationConfig cannot be null");
        ArrayList<Pipeline> pipelines = new ArrayList<Pipeline>();
        for (Pipeline pipeline : this.pipelineMap.values()) {
            if (!pipeline.getReplicationConfig().equals(replicationConfig)) continue;
            pipelines.add(pipeline);
        }
        return pipelines;
    }

    List<Pipeline> getPipelines(ReplicationConfig replicationConfig, Pipeline.PipelineState state) {
        Preconditions.checkNotNull((Object)replicationConfig, (Object)"ReplicationConfig cannot be null");
        Preconditions.checkNotNull((Object)state, (Object)"Pipeline state cannot be null");
        if (state == Pipeline.PipelineState.OPEN) {
            return new ArrayList<Pipeline>(this.query2OpenPipelines.getOrDefault(replicationConfig, Collections.emptyList()));
        }
        ArrayList<Pipeline> pipelines = new ArrayList<Pipeline>();
        for (Pipeline pipeline : this.pipelineMap.values()) {
            if (!pipeline.getReplicationConfig().equals(replicationConfig) || pipeline.getPipelineState() != state) continue;
            pipelines.add(pipeline);
        }
        return pipelines;
    }

    int getPipelineCount(ReplicationConfig replicationConfig, Pipeline.PipelineState state) {
        Preconditions.checkNotNull((Object)replicationConfig, (Object)"ReplicationConfig cannot be null");
        Preconditions.checkNotNull((Object)state, (Object)"Pipeline state cannot be null");
        if (state == Pipeline.PipelineState.OPEN) {
            return this.query2OpenPipelines.getOrDefault(replicationConfig, Collections.emptyList()).size();
        }
        int count = 0;
        for (Pipeline pipeline : this.pipelineMap.values()) {
            if (!pipeline.getReplicationConfig().equals(replicationConfig) || pipeline.getPipelineState() != state) continue;
            ++count;
        }
        return count;
    }

    List<Pipeline> getPipelines(ReplicationConfig replicationConfig, Pipeline.PipelineState state, Collection<DatanodeDetails> excludeDns, Collection<PipelineID> excludePipelines) {
        Preconditions.checkNotNull((Object)replicationConfig, (Object)"ReplicationConfig cannot be null");
        Preconditions.checkNotNull((Object)state, (Object)"Pipeline state cannot be null");
        Preconditions.checkNotNull(excludeDns, (Object)"Datanode exclude list cannot be null");
        Preconditions.checkNotNull(excludePipelines, (Object)"Pipeline exclude list cannot be null");
        ArrayList<Object> pipelines = null;
        if (state == Pipeline.PipelineState.OPEN) {
            pipelines = new ArrayList(this.query2OpenPipelines.getOrDefault(replicationConfig, Collections.emptyList()));
            if (excludeDns.isEmpty() && excludePipelines.isEmpty()) {
                return pipelines;
            }
        } else {
            pipelines = new ArrayList<Pipeline>(this.pipelineMap.values());
        }
        Iterator iter = pipelines.iterator();
        block0: while (iter.hasNext()) {
            Pipeline pipeline = (Pipeline)iter.next();
            if (!pipeline.getReplicationConfig().equals(replicationConfig) || pipeline.getPipelineState() != state || excludePipelines.contains(pipeline.getId())) {
                iter.remove();
                continue;
            }
            for (DatanodeDetails dn : pipeline.getNodes()) {
                if (!excludeDns.contains(dn)) continue;
                iter.remove();
                continue block0;
            }
        }
        return pipelines;
    }

    NavigableSet<ContainerID> getContainers(PipelineID pipelineID) throws PipelineNotFoundException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        NavigableSet<ContainerID> containerIDs = this.pipeline2container.get(pipelineID);
        if (containerIDs == null) {
            throw new PipelineNotFoundException(String.format("%s not found", pipelineID));
        }
        return new TreeSet<ContainerID>((SortedSet<ContainerID>)containerIDs);
    }

    int getNumberOfContainers(PipelineID pipelineID) throws PipelineNotFoundException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Set containerIDs = this.pipeline2container.get(pipelineID);
        if (containerIDs == null) {
            throw new PipelineNotFoundException(String.format("%s not found", pipelineID));
        }
        return containerIDs.size();
    }

    Pipeline removePipeline(PipelineID pipelineID) throws IOException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Pipeline pipeline = this.getPipeline(pipelineID);
        if (!pipeline.isClosed()) {
            throw new InvalidPipelineStateException(String.format("Pipeline with %s is not yet closed", pipelineID));
        }
        this.pipelineMap.remove(pipelineID);
        this.pipeline2container.remove(pipelineID);
        return pipeline;
    }

    void removeContainerFromPipeline(PipelineID pipelineID, ContainerID containerID) throws IOException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Preconditions.checkNotNull((Object)containerID, (Object)"container Id cannot be null");
        Set containerIDs = this.pipeline2container.get(pipelineID);
        if (containerIDs == null) {
            throw new PipelineNotFoundException(String.format("%s not found", pipelineID));
        }
        containerIDs.remove(containerID);
    }

    Pipeline updatePipelineState(PipelineID pipelineID, Pipeline.PipelineState state) throws PipelineNotFoundException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Preconditions.checkNotNull((Object)state, (Object)"Pipeline LifeCycleState cannot be null");
        Pipeline pipeline = this.getPipeline(pipelineID);
        if (pipeline.getPipelineState() == state) {
            LOG.debug("CurrentState and NewState are the same, return from updatePipelineState directly.");
            return pipeline;
        }
        Pipeline updatedPipeline = this.pipelineMap.compute(pipelineID, (id, p) -> Pipeline.newBuilder((Pipeline)pipeline).setState(state).build());
        List<Pipeline> pipelineList = this.query2OpenPipelines.get(pipeline.getReplicationConfig());
        if (updatedPipeline.getPipelineState() == Pipeline.PipelineState.OPEN) {
            if (pipelineList == null) {
                pipelineList = new CopyOnWriteArrayList<Pipeline>();
                this.query2OpenPipelines.put(pipeline.getReplicationConfig(), pipelineList);
            }
            pipelineList.add(updatedPipeline);
        } else if (pipelineList != null) {
            pipelineList.remove(pipeline);
        }
        return updatedPipeline;
    }
}

