/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om.snapshot;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheLoader;
import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hadoop.ozone.om.IOmMetadataReader;
import org.apache.hadoop.ozone.om.OmSnapshot;
import org.apache.hadoop.ozone.om.OmSnapshotManager;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.helpers.SnapshotInfo;
import org.apache.hadoop.ozone.om.snapshot.ReferenceCounted;
import org.apache.hadoop.ozone.om.snapshot.ReferenceCountedCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotCache
implements ReferenceCountedCallback {
    static final Logger LOG = LoggerFactory.getLogger(SnapshotCache.class);
    private final ConcurrentHashMap<String, ReferenceCounted<IOmMetadataReader, SnapshotCache>> dbMap = new ConcurrentHashMap();
    private final Set<ReferenceCounted<IOmMetadataReader, SnapshotCache>> pendingEvictionList = Collections.synchronizedSet(new LinkedHashSet());
    private final OmSnapshotManager omSnapshotManager;
    private final CacheLoader<String, OmSnapshot> cacheLoader;
    private final int cacheSizeLimit;

    public SnapshotCache(OmSnapshotManager omSnapshotManager, CacheLoader<String, OmSnapshot> cacheLoader, int cacheSizeLimit) {
        this.omSnapshotManager = omSnapshotManager;
        this.cacheLoader = cacheLoader;
        this.cacheSizeLimit = cacheSizeLimit;
    }

    @VisibleForTesting
    ConcurrentHashMap<String, ReferenceCounted<IOmMetadataReader, SnapshotCache>> getDbMap() {
        return this.dbMap;
    }

    @VisibleForTesting
    Set<ReferenceCounted<IOmMetadataReader, SnapshotCache>> getPendingEvictionList() {
        return this.pendingEvictionList;
    }

    public int size() {
        return this.dbMap.size();
    }

    public int getPendingEvictionListSize() {
        return this.pendingEvictionList.size();
    }

    public void invalidate(String key) throws IOException {
        ReferenceCounted<IOmMetadataReader, SnapshotCache> rcOmSnapshot = this.dbMap.get(key);
        if (rcOmSnapshot != null) {
            this.pendingEvictionList.remove(rcOmSnapshot);
            try {
                ((OmSnapshot)rcOmSnapshot.get()).close();
            }
            catch (IOException e) {
                throw new IllegalStateException("Failed to close snapshot: " + key, e);
            }
            this.dbMap.remove(key);
        }
    }

    public void invalidateAll() {
        Iterator<Map.Entry<String, ReferenceCounted<IOmMetadataReader, SnapshotCache>>> it = this.dbMap.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, ReferenceCounted<IOmMetadataReader, SnapshotCache>> entry = it.next();
            this.pendingEvictionList.remove(entry.getValue());
            OmSnapshot omSnapshot = (OmSnapshot)entry.getValue().get();
            try {
                omSnapshot.close();
            }
            catch (IOException e) {
                throw new IllegalStateException("Failed to close snapshot", e);
            }
            it.remove();
        }
    }

    public ReferenceCounted<IOmMetadataReader, SnapshotCache> get(String key) throws IOException {
        return this.get(key, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ReferenceCounted<IOmMetadataReader, SnapshotCache> get(String key, boolean skipActiveCheck) throws IOException {
        ReferenceCounted rcOmSnapshot = this.dbMap.compute(key, (k, v) -> {
            LOG.info("Loading snapshot. Table key: {}", k);
            if (v == null) {
                try {
                    v = new ReferenceCounted<IOmMetadataReader, SnapshotCache>((IOmMetadataReader)this.cacheLoader.load(k), false, this);
                }
                catch (OMException omEx) {
                    if (!omEx.getResult().equals((Object)OMException.ResultCodes.FILE_NOT_FOUND)) {
                        throw new IllegalStateException(omEx);
                    }
                }
                catch (IOException ioEx) {
                    throw new IllegalStateException(ioEx);
                }
                catch (Exception ex) {
                    throw new IllegalStateException(ex);
                }
            }
            return v;
        });
        if (rcOmSnapshot == null) {
            throw new OMException("Snapshot table key '" + key + "' not found, " + "or the snapshot is no longer active", OMException.ResultCodes.FILE_NOT_FOUND);
        }
        rcOmSnapshot.incrementRefCount();
        if (!skipActiveCheck && !this.omSnapshotManager.isSnapshotStatus(key, SnapshotInfo.SnapshotStatus.SNAPSHOT_ACTIVE)) {
            rcOmSnapshot.decrementRefCount();
            throw new OMException("Unable to load snapshot. Snapshot with table key '" + key + "' is no longer active", OMException.ResultCodes.FILE_NOT_FOUND);
        }
        Set<ReferenceCounted<IOmMetadataReader, SnapshotCache>> set = this.pendingEvictionList;
        synchronized (set) {
            this.pendingEvictionList.remove(rcOmSnapshot);
        }
        this.cleanup();
        return rcOmSnapshot;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void release(String key) {
        ReferenceCounted<IOmMetadataReader, SnapshotCache> rcOmSnapshot = this.dbMap.get(key);
        if (rcOmSnapshot == null) {
            throw new IllegalArgumentException("Key '" + key + "' does not exist in cache");
        }
        if (rcOmSnapshot.decrementRefCount() == 0L) {
            Set<ReferenceCounted<IOmMetadataReader, SnapshotCache>> set = this.pendingEvictionList;
            synchronized (set) {
                this.pendingEvictionList.add(rcOmSnapshot);
            }
        }
        this.cleanup();
    }

    public void release(OmSnapshot omSnapshot) {
        String snapshotTableKey = omSnapshot.getSnapshotTableKey();
        this.release(snapshotTableKey);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void callback(ReferenceCounted referenceCounted) {
        Set<ReferenceCounted<IOmMetadataReader, SnapshotCache>> set = this.pendingEvictionList;
        synchronized (set) {
            if (referenceCounted.getTotalRefCount() == 0L) {
                Preconditions.checkState((!this.pendingEvictionList.contains(referenceCounted) ? 1 : 0) != 0, (Object)"SnapshotCache is inconsistent. Entry should not be in the pendingEvictionList when ref count just reached zero.");
                this.pendingEvictionList.add(referenceCounted);
            } else if (referenceCounted.getTotalRefCount() == 1L) {
                this.pendingEvictionList.remove(referenceCounted);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void cleanup() {
        Set<ReferenceCounted<IOmMetadataReader, SnapshotCache>> set = this.pendingEvictionList;
        synchronized (set) {
            this.cleanupInternal();
        }
    }

    private void cleanupInternal() {
        long numEntriesToEvict = (long)this.dbMap.size() - (long)this.cacheSizeLimit;
        while (numEntriesToEvict > 0L && this.pendingEvictionList.size() > 0) {
            ReferenceCounted<IOmMetadataReader, SnapshotCache> rcOmSnapshot = this.pendingEvictionList.iterator().next();
            OmSnapshot omSnapshot = (OmSnapshot)rcOmSnapshot.get();
            LOG.debug("Evicting OmSnapshot instance {} with table key {}", rcOmSnapshot, (Object)omSnapshot.getSnapshotTableKey());
            Preconditions.checkState((rcOmSnapshot.getTotalRefCount() == 0L ? 1 : 0) != 0, (Object)("Illegal state: OmSnapshot reference count non-zero (" + rcOmSnapshot.getTotalRefCount() + ") but shows up in the " + "clean up list"));
            String key = omSnapshot.getSnapshotTableKey();
            ReferenceCounted<IOmMetadataReader, SnapshotCache> result = this.dbMap.remove(key);
            Preconditions.checkState((rcOmSnapshot == result ? 1 : 0) != 0, (Object)("Cache map entry removal failure. The cache is in an inconsistent state. Expected OmSnapshot instance: " + rcOmSnapshot + ", actual: " + result));
            this.pendingEvictionList.remove(result);
            try {
                ((OmSnapshot)rcOmSnapshot.get()).close();
            }
            catch (IOException ex) {
                throw new IllegalStateException("Error while closing snapshot DB", ex);
            }
            --numEntriesToEvict;
        }
        if ((long)this.dbMap.size() > (long)this.cacheSizeLimit) {
            LOG.warn("Current snapshot cache size ({}) is exceeding configured soft-limit ({}) after possible evictions.", (Object)this.dbMap.size(), (Object)this.cacheSizeLimit);
            Preconditions.checkState((this.pendingEvictionList.size() == 0 ? 1 : 0) != 0);
        }
    }

    @VisibleForTesting
    public boolean isConsistent() {
        LOG.info("dbMap has {} entries", (Object)this.dbMap.size());
        LOG.info("pendingEvictionList has {} entries", (Object)this.pendingEvictionList.size());
        if (this.dbMap.size() > this.cacheSizeLimit && this.pendingEvictionList.size() != 0) {
            LOG.error("pendingEvictionList is not empty even when cache sizeexceeds limit");
        }
        this.dbMap.forEach((k, v) -> {
            if (v.getTotalRefCount() == 0L) {
                long threadRefCount = v.getCurrentThreadRefCount();
                if (threadRefCount != 0L) {
                    LOG.error("snapshotTableKey='{}' instance has inconsistent ref count. Total ref count is 0 but thread ref count is {}", k, (Object)threadRefCount);
                }
                if (!this.pendingEvictionList.contains(v)) {
                    LOG.error("snapshotTableKey='{}' instance has zero ref count but not in pendingEvictionList", k);
                }
            }
        });
        this.pendingEvictionList.forEach(v -> {
            if (!this.dbMap.contains(v)) {
                LOG.error("Instance '{}' is in pendingEvictionList but not in dbMap", v);
            }
            if (v.getTotalRefCount() != 0L) {
                LOG.error("Instance '{}' is in pendingEvictionList but ref count is not zero", v);
            }
        });
        return true;
    }

    public static enum Reason {
        FS_API_READ,
        SNAPDIFF_READ,
        DEEP_CLEAN_WRITE,
        GARBAGE_COLLECTION_WRITE;

    }
}

