/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.transport.server.ratis;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.StorageUnit;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.ratis.ContainerCommandRequestMessage;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerNotOpenException;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.utils.Cache;
import org.apache.hadoop.hdds.utils.ResourceCache;
import org.apache.hadoop.ozone.common.utils.BufferUtils;
import org.apache.hadoop.ozone.container.common.interfaces.ContainerDispatcher;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.CSMMetrics;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.DispatcherContext;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.LocalStream;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.XceiverServerRatis;
import org.apache.hadoop.ozone.container.keyvalue.impl.KeyValueStreamDataChannel;
import org.apache.hadoop.ozone.container.ozoneimpl.ContainerController;
import org.apache.hadoop.util.Time;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.RaftClientRequest;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftGroupMemberId;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.exceptions.StateMachineException;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.statemachine.StateMachineStorage;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.statemachine.impl.BaseStateMachine;
import org.apache.ratis.statemachine.impl.SimpleStateMachineStorage;
import org.apache.ratis.statemachine.impl.SingleFileSnapshotInfo;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.thirdparty.com.google.protobuf.InvalidProtocolBufferException;
import org.apache.ratis.thirdparty.com.google.protobuf.MessageOrBuilder;
import org.apache.ratis.thirdparty.com.google.protobuf.TextFormat;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.TaskQueue;
import org.apache.ratis.util.function.CheckedSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerStateMachine
extends BaseStateMachine {
    static final Logger LOG = LoggerFactory.getLogger(ContainerStateMachine.class);
    private final SimpleStateMachineStorage storage = new SimpleStateMachineStorage();
    private final RaftGroupId gid;
    private final ContainerDispatcher dispatcher;
    private final ContainerController containerController;
    private final XceiverServerRatis ratisServer;
    private final ConcurrentHashMap<Long, CompletableFuture<ContainerProtos.ContainerCommandResponseProto>> writeChunkFutureMap;
    private final Map<Long, Long> container2BCSIDMap;
    private final TaskQueueMap containerTaskQueues = new TaskQueueMap();
    private final ExecutorService executor;
    private final List<ThreadPoolExecutor> chunkExecutors;
    private final Map<Long, Long> applyTransactionCompletionMap;
    private final Cache<Long, ByteString> stateMachineDataCache;
    private final AtomicBoolean stateMachineHealthy;
    private final Semaphore applyTransactionSemaphore;
    private final boolean waitOnBothFollowers;
    private final CSMMetrics metrics;

    public ContainerStateMachine(RaftGroupId gid, ContainerDispatcher dispatcher, ContainerController containerController, List<ThreadPoolExecutor> chunkExecutors, XceiverServerRatis ratisServer, ConfigurationSource conf, String threadNamePrefix) {
        this.gid = gid;
        this.dispatcher = dispatcher;
        this.containerController = containerController;
        this.ratisServer = ratisServer;
        this.metrics = CSMMetrics.create(gid);
        this.writeChunkFutureMap = new ConcurrentHashMap();
        this.applyTransactionCompletionMap = new ConcurrentHashMap<Long, Long>();
        long pendingRequestsBytesLimit = (long)conf.getStorageSize("dfs.container.ratis.leader.pending.bytes.limit", "1GB", StorageUnit.BYTES);
        this.stateMachineDataCache = new ResourceCache((index, data) -> data.size(), pendingRequestsBytesLimit, p -> {
            if (p.wasEvicted()) {
                this.metrics.incNumEvictedCacheCount();
            }
        });
        this.chunkExecutors = chunkExecutors;
        this.container2BCSIDMap = new ConcurrentHashMap<Long, Long>();
        int numContainerOpExecutors = conf.getInt("dfs.container.ratis.num.container.op.executors", 10);
        int maxPendingApplyTransactions = conf.getInt("dfs.container.ratis.statemachine.max.pending.apply-transactions", 100000);
        this.applyTransactionSemaphore = new Semaphore(maxPendingApplyTransactions);
        this.stateMachineHealthy = new AtomicBoolean(true);
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat(threadNamePrefix + "ContainerOp-" + gid.getUuid() + "-%d").build();
        this.executor = Executors.newFixedThreadPool(numContainerOpExecutors, threadFactory);
        this.waitOnBothFollowers = ((DatanodeConfiguration)((Object)conf.getObject(DatanodeConfiguration.class))).waitOnAllFollowers();
    }

    public StateMachineStorage getStateMachineStorage() {
        return this.storage;
    }

    public CSMMetrics getMetrics() {
        return this.metrics;
    }

    public void initialize(RaftServer server, RaftGroupId id, RaftStorage raftStorage) throws IOException {
        super.initialize(server, id, raftStorage);
        this.storage.init(raftStorage);
        this.ratisServer.notifyGroupAdd(this.gid);
        this.loadSnapshot(this.storage.getLatestSnapshot());
    }

    private long loadSnapshot(SingleFileSnapshotInfo snapshot) throws IOException {
        if (snapshot == null) {
            TermIndex empty = TermIndex.valueOf((long)0L, (long)-1L);
            LOG.info("{}: The snapshot info is null. Setting the last applied indexto:{}", (Object)this.gid, (Object)empty);
            this.setLastAppliedTermIndex(empty);
            return empty.getIndex();
        }
        File snapshotFile = snapshot.getFile().getPath().toFile();
        TermIndex last = SimpleStateMachineStorage.getTermIndexFromSnapshotFile((File)snapshotFile);
        LOG.info("{}: Setting the last applied index to {}", (Object)this.gid, (Object)last);
        this.setLastAppliedTermIndex(last);
        this.buildMissingContainerSet(snapshotFile);
        return last.getIndex();
    }

    @VisibleForTesting
    public void buildMissingContainerSet(File snapshotFile) throws IOException {
        try (FileInputStream fin = new FileInputStream(snapshotFile);){
            ContainerProtos.Container2BCSIDMapProto proto = ContainerProtos.Container2BCSIDMapProto.parseFrom((InputStream)fin);
            this.container2BCSIDMap.putAll(proto.getContainer2BCSIDMap());
            this.dispatcher.buildMissingContainerSetAndValidate(this.container2BCSIDMap);
        }
    }

    public void persistContainerSet(OutputStream out) throws IOException {
        ContainerProtos.Container2BCSIDMapProto.Builder builder = ContainerProtos.Container2BCSIDMapProto.newBuilder();
        builder.putAllContainer2BCSID(this.container2BCSIDMap);
        builder.build().writeTo(out);
    }

    public boolean isStateMachineHealthy() {
        return this.stateMachineHealthy.get();
    }

    public long takeSnapshot() throws IOException {
        TermIndex ti = this.getLastAppliedTermIndex();
        long startTime = Time.monotonicNow();
        if (!this.isStateMachineHealthy()) {
            String msg = "Failed to take snapshot  for " + this.gid + " as the stateMachine is unhealthy. The last applied index is at " + ti;
            StateMachineException sme = new StateMachineException(msg);
            LOG.error(msg);
            throw sme;
        }
        if (ti != null && ti.getIndex() != -1L) {
            File snapshotFile = this.storage.getSnapshotFile(ti.getTerm(), ti.getIndex());
            LOG.info("{}: Taking a snapshot at:{} file {}", new Object[]{this.gid, ti, snapshotFile});
            try (FileOutputStream fos = new FileOutputStream(snapshotFile);){
                this.persistContainerSet(fos);
                fos.flush();
                fos.getFD().sync();
            }
            catch (IOException ioe) {
                LOG.error("{}: Failed to write snapshot at:{} file {}", new Object[]{this.gid, ti, snapshotFile});
                throw ioe;
            }
            LOG.info("{}: Finished taking a snapshot at:{} file:{} took: {} ms", new Object[]{this.gid, ti, snapshotFile, Time.monotonicNow() - startTime});
            return ti.getIndex();
        }
        return -1L;
    }

    public TransactionContext startTransaction(RaftProtos.LogEntryProto entry, RaftProtos.RaftPeerRole role) {
        ContainerProtos.ContainerCommandRequestProto logProto;
        TransactionContext trx = super.startTransaction(entry, role);
        RaftProtos.StateMachineLogEntryProto stateMachineLogEntry = entry.getStateMachineLogEntry();
        try {
            logProto = ContainerStateMachine.getContainerCommandRequestProto(this.gid, stateMachineLogEntry.getLogData());
        }
        catch (InvalidProtocolBufferException e) {
            trx.setException((Exception)((Object)e));
            return trx;
        }
        ContainerProtos.ContainerCommandRequestProto requestProto = logProto.getCmdType() == ContainerProtos.Type.WriteChunk ? ContainerProtos.ContainerCommandRequestProto.newBuilder((ContainerProtos.ContainerCommandRequestProto)logProto).setWriteChunk(ContainerProtos.WriteChunkRequestProto.newBuilder((ContainerProtos.WriteChunkRequestProto)logProto.getWriteChunk()).setData(stateMachineLogEntry.getStateMachineEntry().getStateMachineData())).build() : logProto;
        return trx.setStateMachineContext((Object)new Context(requestProto, logProto));
    }

    public TransactionContext startTransaction(RaftClientRequest request) throws IOException {
        ContainerProtos.ContainerCommandRequestProto proto = this.message2ContainerCommandRequestProto(request.getMessage());
        Preconditions.checkArgument((boolean)request.getRaftGroupId().equals((Object)this.gid));
        TransactionContext.Builder builder = TransactionContext.newBuilder().setClientRequest(request).setStateMachine((StateMachine)this).setServerRole(RaftProtos.RaftPeerRole.LEADER);
        try {
            this.dispatcher.validateContainerCommand(proto);
        }
        catch (IOException ioe) {
            if (ioe instanceof ContainerNotOpenException) {
                this.metrics.incNumContainerNotOpenVerifyFailures();
            } else {
                this.metrics.incNumStartTransactionVerifyFailures();
                LOG.error("startTransaction validation failed on leader", (Throwable)ioe);
            }
            return builder.build().setException((Exception)ioe);
        }
        if (proto.getCmdType() == ContainerProtos.Type.WriteChunk) {
            ContainerProtos.WriteChunkRequestProto write = proto.getWriteChunk();
            ContainerProtos.WriteChunkRequestProto commitWriteChunkProto = ContainerProtos.WriteChunkRequestProto.newBuilder().setBlockID(write.getBlockID()).setChunkData(write.getChunkData()).build();
            ContainerProtos.ContainerCommandRequestProto commitContainerCommandProto = ContainerProtos.ContainerCommandRequestProto.newBuilder((ContainerProtos.ContainerCommandRequestProto)proto).setPipelineID(this.gid.getUuid().toString()).setWriteChunk(commitWriteChunkProto).setTraceID(proto.getTraceID()).build();
            Preconditions.checkArgument((boolean)write.hasData());
            Preconditions.checkArgument((!write.getData().isEmpty() ? 1 : 0) != 0);
            Context context = new Context(proto, commitContainerCommandProto);
            return builder.setStateMachineContext((Object)context).setStateMachineData(write.getData()).setLogData(commitContainerCommandProto.toByteString()).build();
        }
        Context context = new Context(proto, proto);
        return builder.setStateMachineContext((Object)context).setLogData(proto.toByteString()).build();
    }

    private static ContainerProtos.ContainerCommandRequestProto getContainerCommandRequestProto(RaftGroupId id, ByteString request) throws InvalidProtocolBufferException {
        return ContainerProtos.ContainerCommandRequestProto.newBuilder((ContainerProtos.ContainerCommandRequestProto)ContainerProtos.ContainerCommandRequestProto.parseFrom((ByteString)request)).setPipelineID(id.getUuid().toString()).build();
    }

    private ContainerProtos.ContainerCommandRequestProto message2ContainerCommandRequestProto(Message message) throws InvalidProtocolBufferException {
        return ContainerCommandRequestMessage.toProto((ByteString)message.getContent(), (RaftGroupId)this.gid);
    }

    private ContainerProtos.ContainerCommandResponseProto dispatchCommand(ContainerProtos.ContainerCommandRequestProto requestProto, DispatcherContext context) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("{}: dispatch {} containerID={} pipelineID={} traceID={}", new Object[]{this.gid, requestProto.getCmdType(), requestProto.getContainerID(), requestProto.getPipelineID(), requestProto.getTraceID()});
        }
        ContainerProtos.ContainerCommandResponseProto response = this.dispatcher.dispatch(requestProto, context);
        if (LOG.isTraceEnabled()) {
            LOG.trace("{}: response {}", (Object)this.gid, (Object)response);
        }
        return response;
    }

    private CompletableFuture<ContainerProtos.ContainerCommandResponseProto> link(ContainerProtos.ContainerCommandRequestProto requestProto, RaftProtos.LogEntryProto entry) {
        return CompletableFuture.supplyAsync(() -> {
            DispatcherContext context = DispatcherContext.newBuilder(DispatcherContext.Op.STREAM_LINK).setTerm(entry.getTerm()).setLogIndex(entry.getIndex()).setStage(DispatcherContext.WriteChunkStage.COMMIT_DATA).setContainer2BCSIDMap(this.container2BCSIDMap).build();
            return this.dispatchCommand(requestProto, context);
        }, this.executor);
    }

    private CompletableFuture<Message> writeStateMachineData(ContainerProtos.ContainerCommandRequestProto requestProto, long entryIndex, long term, long startTime) {
        ContainerProtos.WriteChunkRequestProto write = requestProto.getWriteChunk();
        RaftServer server = this.ratisServer.getServer();
        Preconditions.checkArgument((!write.getData().isEmpty() ? 1 : 0) != 0);
        try {
            if (server.getDivision(this.gid).getInfo().isLeader()) {
                this.stateMachineDataCache.put((Object)entryIndex, (Object)write.getData());
            }
        }
        catch (InterruptedException ioe) {
            Thread.currentThread().interrupt();
            return ContainerStateMachine.completeExceptionally(ioe);
        }
        catch (IOException ioe) {
            return ContainerStateMachine.completeExceptionally(ioe);
        }
        DispatcherContext context = DispatcherContext.newBuilder(DispatcherContext.Op.WRITE_STATE_MACHINE_DATA).setTerm(term).setLogIndex(entryIndex).setStage(DispatcherContext.WriteChunkStage.WRITE_DATA).setContainer2BCSIDMap(this.container2BCSIDMap).build();
        CompletableFuture<Message> raftFuture = new CompletableFuture<Message>();
        CompletableFuture<ContainerProtos.ContainerCommandResponseProto> writeChunkFuture = CompletableFuture.supplyAsync(() -> {
            try {
                return this.dispatchCommand(requestProto, context);
            }
            catch (Exception e) {
                LOG.error("{}: writeChunk writeStateMachineData failed: blockId{} logIndex {} chunkName {}", new Object[]{this.gid, write.getBlockID(), entryIndex, write.getChunkData().getChunkName(), e});
                this.metrics.incNumWriteDataFails();
                this.stateMachineHealthy.set(false);
                raftFuture.completeExceptionally(e);
                throw e;
            }
        }, this.getChunkExecutor(requestProto.getWriteChunk()));
        this.writeChunkFutureMap.put(entryIndex, writeChunkFuture);
        if (LOG.isDebugEnabled()) {
            LOG.debug("{}: writeChunk writeStateMachineData : blockId{} logIndex {} chunkName {}", new Object[]{this.gid, write.getBlockID(), entryIndex, write.getChunkData().getChunkName()});
        }
        writeChunkFuture.thenApply(r -> {
            if (r.getResult() != ContainerProtos.Result.SUCCESS && r.getResult() != ContainerProtos.Result.CONTAINER_NOT_OPEN && r.getResult() != ContainerProtos.Result.CLOSED_CONTAINER_IO) {
                StorageContainerException sce = new StorageContainerException(r.getMessage(), r.getResult());
                LOG.error(this.gid + ": writeChunk writeStateMachineData failed: blockId" + write.getBlockID() + " logIndex " + entryIndex + " chunkName " + write.getChunkData().getChunkName() + " Error message: " + r.getMessage() + " Container Result: " + r.getResult());
                this.metrics.incNumWriteDataFails();
                this.stateMachineHealthy.set(false);
                raftFuture.completeExceptionally((Throwable)sce);
            } else {
                this.metrics.incNumBytesWrittenCount(requestProto.getWriteChunk().getChunkData().getLen());
                if (LOG.isDebugEnabled()) {
                    LOG.debug(this.gid + ": writeChunk writeStateMachineData  completed: blockId" + write.getBlockID() + " logIndex " + entryIndex + " chunkName " + write.getChunkData().getChunkName());
                }
                raftFuture.complete(() -> ((ContainerProtos.ContainerCommandResponseProto)r).toByteString());
                this.metrics.recordWriteStateMachineCompletionNs(Time.monotonicNowNanos() - startTime);
            }
            this.writeChunkFutureMap.remove(entryIndex);
            return r;
        });
        return raftFuture;
    }

    private StateMachine.DataChannel getStreamDataChannel(ContainerProtos.ContainerCommandRequestProto requestProto, DispatcherContext context) throws StorageContainerException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("{}: getStreamDataChannel {} containerID={} pipelineID={} traceID={}", new Object[]{this.gid, requestProto.getCmdType(), requestProto.getContainerID(), requestProto.getPipelineID(), requestProto.getTraceID()});
        }
        this.dispatchCommand(requestProto, context);
        return this.dispatcher.getStreamDataChannel(requestProto);
    }

    public CompletableFuture<StateMachine.DataStream> stream(RaftClientRequest request) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                ContainerProtos.ContainerCommandRequestProto requestProto = this.message2ContainerCommandRequestProto(request.getMessage());
                DispatcherContext context = DispatcherContext.newBuilder(DispatcherContext.Op.STREAM_INIT).setStage(DispatcherContext.WriteChunkStage.WRITE_DATA).setContainer2BCSIDMap(this.container2BCSIDMap).build();
                StateMachine.DataChannel channel = this.getStreamDataChannel(requestProto, context);
                ExecutorService chunkExecutor = requestProto.hasWriteChunk() ? this.getChunkExecutor(requestProto.getWriteChunk()) : null;
                return new LocalStream(channel, chunkExecutor);
            }
            catch (IOException e) {
                throw new CompletionException("Failed to create data stream", e);
            }
        }, this.executor);
    }

    public CompletableFuture<?> link(StateMachine.DataStream stream, RaftProtos.LogEntryProto entry) {
        if (stream == null) {
            return JavaUtils.completeExceptionally((Throwable)new IllegalStateException("DataStream is null"));
        }
        if (!(stream instanceof LocalStream)) {
            return JavaUtils.completeExceptionally((Throwable)new IllegalStateException("Unexpected DataStream " + stream.getClass()));
        }
        StateMachine.DataChannel dataChannel = stream.getDataChannel();
        if (dataChannel.isOpen()) {
            return JavaUtils.completeExceptionally((Throwable)new IllegalStateException("DataStream: " + stream + " is not closed properly"));
        }
        if (!(dataChannel instanceof KeyValueStreamDataChannel)) {
            return JavaUtils.completeExceptionally((Throwable)new IllegalStateException("Unexpected DataChannel " + dataChannel.getClass()));
        }
        KeyValueStreamDataChannel kvStreamDataChannel = (KeyValueStreamDataChannel)dataChannel;
        ContainerProtos.ContainerCommandRequestProto request = kvStreamDataChannel.getPutBlockRequest();
        return this.link(request, entry).whenComplete((response, e) -> {
            if (e != null) {
                LOG.warn("Failed to link logEntry {} for request {}", new Object[]{TermIndex.valueOf((RaftProtos.LogEntryProto)entry), request, e});
            }
            if (response != null) {
                ContainerProtos.Result result = response.getResult();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{} to link logEntry {} for request {}, response: {}", new Object[]{result, TermIndex.valueOf((RaftProtos.LogEntryProto)entry), request, response});
                }
                if (result == ContainerProtos.Result.SUCCESS) {
                    kvStreamDataChannel.setLinked();
                    return;
                }
            }
            kvStreamDataChannel.cleanUp();
        });
    }

    private ExecutorService getChunkExecutor(ContainerProtos.WriteChunkRequestProto req) {
        int i = (int)(req.getBlockID().getLocalID() % (long)this.chunkExecutors.size());
        return this.chunkExecutors.get(i);
    }

    public CompletableFuture<Message> write(RaftProtos.LogEntryProto entry, TransactionContext trx) {
        try {
            this.metrics.incNumWriteStateMachineOps();
            long writeStateMachineStartTime = Time.monotonicNowNanos();
            Context context = (Context)trx.getStateMachineContext();
            Objects.requireNonNull(context, "context == null");
            ContainerProtos.ContainerCommandRequestProto requestProto = context.getRequestProto();
            ContainerProtos.Type cmdType = requestProto.getCmdType();
            switch (cmdType) {
                case WriteChunk: {
                    return this.writeStateMachineData(requestProto, entry.getIndex(), entry.getTerm(), writeStateMachineStartTime);
                }
            }
            throw new IllegalStateException("Cmd Type:" + cmdType + " should not have state machine data");
        }
        catch (Exception e) {
            this.metrics.incNumWriteStateMachineFails();
            return ContainerStateMachine.completeExceptionally(e);
        }
    }

    public CompletableFuture<Message> query(Message request) {
        try {
            this.metrics.incNumQueryStateMachineOps();
            ContainerProtos.ContainerCommandRequestProto requestProto = this.message2ContainerCommandRequestProto(request);
            return CompletableFuture.completedFuture(() -> ((ContainerProtos.ContainerCommandResponseProto)this.dispatchCommand(requestProto, null)).toByteString());
        }
        catch (IOException e) {
            this.metrics.incNumQueryStateMachineFails();
            return ContainerStateMachine.completeExceptionally(e);
        }
    }

    private ByteString readStateMachineData(ContainerProtos.ContainerCommandRequestProto requestProto, long term, long index) throws IOException {
        this.metrics.incNumReadStateMachineMissCount();
        ContainerProtos.WriteChunkRequestProto writeChunkRequestProto = requestProto.getWriteChunk();
        ContainerProtos.ChunkInfo chunkInfo = writeChunkRequestProto.getChunkData();
        ContainerProtos.ReadChunkRequestProto.Builder readChunkRequestProto = ContainerProtos.ReadChunkRequestProto.newBuilder().setBlockID(writeChunkRequestProto.getBlockID()).setChunkData(chunkInfo).setReadChunkVersion(ContainerProtos.ReadChunkVersion.V1);
        ContainerProtos.ContainerCommandRequestProto dataContainerCommandProto = ContainerProtos.ContainerCommandRequestProto.newBuilder((ContainerProtos.ContainerCommandRequestProto)requestProto).setCmdType(ContainerProtos.Type.ReadChunk).setReadChunk(readChunkRequestProto).build();
        DispatcherContext context = DispatcherContext.newBuilder(DispatcherContext.Op.READ_STATE_MACHINE_DATA).setTerm(term).setLogIndex(index).build();
        ContainerProtos.ContainerCommandResponseProto response = this.dispatchCommand(dataContainerCommandProto, context);
        if (response.getResult() != ContainerProtos.Result.SUCCESS) {
            StorageContainerException sce = new StorageContainerException(response.getMessage(), response.getResult());
            LOG.error("gid {} : ReadStateMachine failed. cmd {} logIndex {} msg : {} Container Result: {}", new Object[]{this.gid, response.getCmdType(), index, response.getMessage(), response.getResult()});
            this.stateMachineHealthy.set(false);
            throw sce;
        }
        ContainerProtos.ReadChunkResponseProto responseProto = response.getReadChunk();
        ByteString data = responseProto.hasData() ? responseProto.getData() : BufferUtils.concatByteStrings((List)responseProto.getDataBuffers().getBuffersList());
        Preconditions.checkNotNull((Object)data, (String)"read chunk data is null for chunk: %s", (Object)chunkInfo);
        Preconditions.checkState(((long)data.size() == chunkInfo.getLen() ? 1 : 0) != 0, (String)"read chunk len=%s does not match chunk expected len=%s for chunk:%s", (Object)data.size(), (Object)chunkInfo.getLen(), (Object)chunkInfo);
        return data;
    }

    public CompletableFuture<Void> flush(long index) {
        List<CompletableFuture> futureList = this.writeChunkFutureMap.entrySet().stream().filter(x -> (Long)x.getKey() <= index).map(Map.Entry::getValue).collect(Collectors.toList());
        return CompletableFuture.allOf(futureList.toArray(new CompletableFuture[futureList.size()]));
    }

    public CompletableFuture<ByteString> read(RaftProtos.LogEntryProto entry, TransactionContext trx) {
        this.metrics.incNumReadStateMachineOps();
        ByteString dataInContext = Optional.ofNullable(trx).map(TransactionContext::getStateMachineLogEntry).map(RaftProtos.StateMachineLogEntryProto::getStateMachineEntry).map(RaftProtos.StateMachineEntryProto::getStateMachineData).orElse(null);
        if (dataInContext != null && !dataInContext.isEmpty()) {
            return CompletableFuture.completedFuture(dataInContext);
        }
        ByteString dataInCache = (ByteString)this.stateMachineDataCache.get((Object)entry.getIndex());
        if (dataInCache != null) {
            Preconditions.checkArgument((!dataInCache.isEmpty() ? 1 : 0) != 0);
            this.metrics.incNumDataCacheHit();
            return CompletableFuture.completedFuture(dataInCache);
        }
        this.metrics.incNumDataCacheMiss();
        try {
            ContainerProtos.ContainerCommandRequestProto requestProto;
            Context context = Optional.ofNullable(trx).map(TransactionContext::getStateMachineContext).orElse(null);
            ContainerProtos.ContainerCommandRequestProto containerCommandRequestProto = requestProto = context != null ? context.getLogProto() : ContainerStateMachine.getContainerCommandRequestProto(this.gid, entry.getStateMachineLogEntry().getLogData());
            if (requestProto.getCmdType() != ContainerProtos.Type.WriteChunk) {
                throw new IllegalStateException("Cmd type:" + requestProto.getCmdType() + " cannot have state machine data");
            }
            CompletableFuture<ByteString> future = new CompletableFuture<ByteString>();
            CompletableFuture.runAsync(() -> {
                try {
                    future.complete(this.readStateMachineData(requestProto, entry.getTerm(), entry.getIndex()));
                }
                catch (IOException e) {
                    this.metrics.incNumReadStateMachineFails();
                    future.completeExceptionally(e);
                }
            }, this.getChunkExecutor(requestProto.getWriteChunk()));
            return future;
        }
        catch (Exception e) {
            this.metrics.incNumReadStateMachineFails();
            LOG.error("{} unable to read stateMachineData:", (Object)this.gid, (Object)e);
            return ContainerStateMachine.completeExceptionally(e);
        }
    }

    private synchronized void updateLastApplied() {
        Long removed;
        Long appliedTerm = null;
        long appliedIndex = -1L;
        long i = this.getLastAppliedTermIndex().getIndex() + 1L;
        while ((removed = this.applyTransactionCompletionMap.remove(i)) != null) {
            appliedTerm = removed;
            appliedIndex = i++;
        }
        if (appliedTerm != null) {
            this.updateLastAppliedTermIndex(appliedTerm, appliedIndex);
        }
    }

    public void notifyTermIndexUpdated(long term, long index) {
        this.applyTransactionCompletionMap.put(index, term);
        this.updateLastApplied();
        this.removeStateMachineDataIfNeeded(index);
    }

    private CompletableFuture<ContainerProtos.ContainerCommandResponseProto> applyTransaction(ContainerProtos.ContainerCommandRequestProto request, DispatcherContext context, Consumer<Throwable> exceptionHandler) {
        long containerId = request.getContainerID();
        CheckedSupplier task = () -> {
            try {
                return this.dispatchCommand(request, context);
            }
            catch (Exception e) {
                exceptionHandler.accept(e);
                throw e;
            }
        };
        return this.containerTaskQueues.submit(containerId, (CheckedSupplier<ContainerProtos.ContainerCommandResponseProto, Exception>)task, this.executor);
    }

    private void removeStateMachineDataIfNeeded(long index) {
        if (this.waitOnBothFollowers) {
            try {
                RaftServer.Division division = this.ratisServer.getServer().getDivision(this.gid);
                if (division.getInfo().isLeader()) {
                    long minIndex = Arrays.stream(division.getInfo().getFollowerNextIndices()).min().getAsLong();
                    LOG.debug("Removing data corresponding to log index {} min index {} from cache", (Object)index, (Object)minIndex);
                    this.removeCacheDataUpTo(Math.min(minIndex, index));
                }
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
    }

    public CompletableFuture<Message> applyTransaction(TransactionContext trx) {
        long index = trx.getLogEntry().getIndex();
        try {
            this.removeStateMachineDataIfNeeded(index);
            this.removeStateMachineDataIfMajorityFollowSync(index);
            DispatcherContext.Builder builder = DispatcherContext.newBuilder(DispatcherContext.Op.APPLY_TRANSACTION).setTerm(trx.getLogEntry().getTerm()).setLogIndex(index);
            long applyTxnStartTime = Time.monotonicNowNanos();
            this.applyTransactionSemaphore.acquire();
            this.metrics.incNumApplyTransactionsOps();
            Context context = (Context)trx.getStateMachineContext();
            Objects.requireNonNull(context, "context == null");
            ContainerProtos.ContainerCommandRequestProto requestProto = context.getLogProto();
            ContainerProtos.Type cmdType = requestProto.getCmdType();
            if (cmdType == ContainerProtos.Type.WriteChunk) {
                Preconditions.checkArgument((boolean)requestProto.getWriteChunk().getData().isEmpty());
                builder.setStage(DispatcherContext.WriteChunkStage.COMMIT_DATA);
            }
            if (cmdType == ContainerProtos.Type.WriteChunk || cmdType == ContainerProtos.Type.PutSmallFile || cmdType == ContainerProtos.Type.PutBlock || cmdType == ContainerProtos.Type.CreateContainer || cmdType == ContainerProtos.Type.StreamInit) {
                builder.setContainer2BCSIDMap(this.container2BCSIDMap);
            }
            CompletableFuture<Message> applyTransactionFuture = new CompletableFuture<Message>();
            Consumer<Throwable> exceptionHandler = e -> {
                LOG.error(this.gid + ": failed to applyTransaction at logIndex " + index + " for " + requestProto.getCmdType(), e);
                this.stateMachineHealthy.compareAndSet(true, false);
                this.metrics.incNumApplyTransactionsFails();
                applyTransactionFuture.completeExceptionally((Throwable)e);
            };
            CompletableFuture<ContainerProtos.ContainerCommandResponseProto> future = this.applyTransaction(requestProto, builder.build(), exceptionHandler);
            ((CompletableFuture)future.thenApply(r -> {
                if (trx.getServerRole() == RaftProtos.RaftPeerRole.LEADER) {
                    long startTime = context.getStartTime();
                    this.metrics.incPipelineLatencyMs(cmdType, (Time.monotonicNowNanos() - startTime) / 1000000L);
                }
                if (r.getResult() != ContainerProtos.Result.SUCCESS && r.getResult() != ContainerProtos.Result.CONTAINER_NOT_OPEN && r.getResult() != ContainerProtos.Result.CLOSED_CONTAINER_IO) {
                    StorageContainerException sce = new StorageContainerException(r.getMessage(), r.getResult());
                    LOG.error("gid {} : ApplyTransaction failed. cmd {} logIndex {} msg : {} Container Result: {}", new Object[]{this.gid, r.getCmdType(), index, r.getMessage(), r.getResult()});
                    this.metrics.incNumApplyTransactionsFails();
                    applyTransactionFuture.completeExceptionally((Throwable)sce);
                    this.stateMachineHealthy.compareAndSet(true, false);
                    this.ratisServer.handleApplyTransactionFailure(this.gid, trx.getServerRole());
                } else {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("gid {} : ApplyTransaction completed. cmd {} logIndex {} msg : {} Container Result: {}", new Object[]{this.gid, r.getCmdType(), index, r.getMessage(), r.getResult()});
                    }
                    if (cmdType == ContainerProtos.Type.WriteChunk || cmdType == ContainerProtos.Type.PutSmallFile) {
                        this.metrics.incNumBytesCommittedCount(requestProto.getWriteChunk().getChunkData().getLen());
                    }
                    applyTransactionFuture.complete(() -> ((ContainerProtos.ContainerCommandResponseProto)r).toByteString());
                    if (this.isStateMachineHealthy()) {
                        Long previous = this.applyTransactionCompletionMap.put(index, trx.getLogEntry().getTerm());
                        Preconditions.checkState((previous == null ? 1 : 0) != 0);
                        this.updateLastApplied();
                    }
                }
                return applyTransactionFuture;
            })).whenComplete((r, t) -> {
                if (t != null) {
                    exceptionHandler.accept((Throwable)t);
                }
                this.applyTransactionSemaphore.release();
                this.metrics.recordApplyTransactionCompletionNs(Time.monotonicNowNanos() - applyTxnStartTime);
            });
            return applyTransactionFuture;
        }
        catch (InterruptedException e2) {
            this.metrics.incNumApplyTransactionsFails();
            Thread.currentThread().interrupt();
            return ContainerStateMachine.completeExceptionally(e2);
        }
        catch (Exception e3) {
            this.metrics.incNumApplyTransactionsFails();
            return ContainerStateMachine.completeExceptionally(e3);
        }
    }

    private void removeStateMachineDataIfMajorityFollowSync(long index) {
        if (!this.waitOnBothFollowers) {
            this.removeCacheDataUpTo(index);
        }
    }

    private void removeCacheDataUpTo(long index) {
        this.stateMachineDataCache.removeIf(k -> k <= index);
    }

    private static <T> CompletableFuture<T> completeExceptionally(Exception e) {
        CompletableFuture future = new CompletableFuture();
        future.completeExceptionally(e);
        return future;
    }

    public void notifyNotLeader(Collection<TransactionContext> pendingEntries) {
        this.evictStateMachineCache();
    }

    public CompletableFuture<Void> truncate(long index) {
        this.stateMachineDataCache.removeIf(k -> k > index);
        return CompletableFuture.completedFuture(null);
    }

    @VisibleForTesting
    public void evictStateMachineCache() {
        this.stateMachineDataCache.clear();
    }

    public void notifyFollowerSlowness(RaftProtos.RoleInfoProto roleInfoProto) {
        this.ratisServer.handleNodeSlowness(this.gid, roleInfoProto);
    }

    public void notifyExtendedNoLeader(RaftProtos.RoleInfoProto roleInfoProto) {
        this.ratisServer.handleNoLeader(this.gid, roleInfoProto);
    }

    public void notifyLogFailed(Throwable t, RaftProtos.LogEntryProto failedEntry) {
        LOG.error("{}: {} {}", new Object[]{this.gid, TermIndex.valueOf((RaftProtos.LogEntryProto)failedEntry), this.toStateMachineLogEntryString(failedEntry.getStateMachineLogEntry()), t});
        this.ratisServer.handleNodeLogFailure(this.gid, t);
    }

    public CompletableFuture<TermIndex> notifyInstallSnapshotFromLeader(RaftProtos.RoleInfoProto roleInfoProto, TermIndex firstTermIndexInLog) {
        this.ratisServer.handleInstallSnapshotFromLeader(this.gid, roleInfoProto, firstTermIndexInLog);
        CompletableFuture<TermIndex> future = new CompletableFuture<TermIndex>();
        future.complete(firstTermIndexInLog);
        return future;
    }

    public void notifyGroupRemove() {
        this.ratisServer.notifyGroupRemove(this.gid);
        for (Long cid : this.container2BCSIDMap.keySet()) {
            try {
                this.containerController.markContainerForClose(cid);
                this.containerController.quasiCloseContainer(cid, "Ratis group removed");
            }
            catch (IOException e) {
                LOG.debug("Failed to quasi-close container {}", (Object)cid);
            }
        }
    }

    public void close() {
        this.evictStateMachineCache();
        this.executor.shutdown();
        this.metrics.unRegister();
    }

    public void notifyLeaderChanged(RaftGroupMemberId groupMemberId, RaftPeerId raftPeerId) {
        this.ratisServer.handleLeaderChangedNotification(groupMemberId, raftPeerId);
    }

    public String toStateMachineLogEntryString(RaftProtos.StateMachineLogEntryProto proto) {
        return ContainerStateMachine.smProtoToString(this.gid, this.containerController, proto);
    }

    public static String smProtoToString(RaftGroupId gid, ContainerController containerController, RaftProtos.StateMachineLogEntryProto proto) {
        StringBuilder builder = new StringBuilder();
        try {
            ContainerProtos.ContainerCommandRequestProto requestProto = ContainerStateMachine.getContainerCommandRequestProto(gid, proto.getLogData());
            long contId = requestProto.getContainerID();
            builder.append(TextFormat.shortDebugString((MessageOrBuilder)requestProto));
            if (containerController != null) {
                String location = containerController.getContainerLocation(contId);
                builder.append(", container path=");
                builder.append(location);
            }
        }
        catch (Exception t) {
            LOG.info("smProtoToString failed", (Throwable)t);
            builder.append("smProtoToString failed with ");
            builder.append(t.getMessage());
        }
        return builder.toString();
    }

    static class Context {
        private final ContainerProtos.ContainerCommandRequestProto requestProto;
        private final ContainerProtos.ContainerCommandRequestProto logProto;
        private final long startTime = Time.monotonicNowNanos();

        Context(ContainerProtos.ContainerCommandRequestProto requestProto, ContainerProtos.ContainerCommandRequestProto logProto) {
            this.requestProto = requestProto;
            this.logProto = logProto;
        }

        ContainerProtos.ContainerCommandRequestProto getRequestProto() {
            return this.requestProto;
        }

        ContainerProtos.ContainerCommandRequestProto getLogProto() {
            return this.logProto;
        }

        long getStartTime() {
            return this.startTime;
        }
    }

    static class TaskQueueMap {
        private final Map<Long, TaskQueue> map = new HashMap<Long, TaskQueue>();

        TaskQueueMap() {
        }

        synchronized CompletableFuture<ContainerProtos.ContainerCommandResponseProto> submit(long containerId, CheckedSupplier<ContainerProtos.ContainerCommandResponseProto, Exception> task, ExecutorService executor) {
            TaskQueue queue = this.map.computeIfAbsent(containerId, id -> new TaskQueue("container" + id));
            CompletableFuture f = queue.submit(task, executor);
            f.thenAccept(dummy -> this.removeIfEmpty(containerId));
            return f;
        }

        synchronized void removeIfEmpty(long containerId) {
            this.map.computeIfPresent(containerId, (id, q) -> q.isEmpty() ? null : q);
        }
    }
}

