/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.volume;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nonnull;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdfs.server.datanode.checker.VolumeCheckResult;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.common.volume.AsyncChecker;
import org.apache.hadoop.ozone.container.common.volume.StorageVolume;
import org.apache.hadoop.ozone.container.common.volume.ThrottledAsyncChecker;
import org.apache.hadoop.ozone.container.common.volume.VolumeSet;
import org.apache.hadoop.util.Timer;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageVolumeChecker {
    public static final int MAX_VOLUME_FAILURE_TOLERATED_LIMIT = -1;
    public static final Logger LOG = LoggerFactory.getLogger(StorageVolumeChecker.class);
    private AsyncChecker<Boolean, VolumeCheckResult> delegateChecker;
    private final AtomicLong numVolumeChecks = new AtomicLong(0L);
    private final AtomicLong numAllVolumeChecks = new AtomicLong(0L);
    private final AtomicLong numAllVolumeSetsChecks = new AtomicLong(0L);
    private final AtomicLong numSkippedChecks = new AtomicLong(0L);
    private final long maxAllowedTimeForCheckMs;
    private final long minDiskCheckGapMs;
    private long lastAllVolumeSetsCheckComplete;
    private final Timer timer;
    private final ExecutorService checkVolumeResultHandlerExecutorService;
    private final DatanodeConfiguration dnConf;
    private final ScheduledExecutorService diskCheckerservice;
    private ScheduledFuture<?> periodicDiskChecker;
    private final List<VolumeSet> registeredVolumeSets;
    private final AtomicBoolean started;

    public StorageVolumeChecker(ConfigurationSource conf, Timer timer, String threadNamePrefix) {
        this.timer = timer;
        this.dnConf = (DatanodeConfiguration)((Object)conf.getObject(DatanodeConfiguration.class));
        this.maxAllowedTimeForCheckMs = this.dnConf.getDiskCheckTimeout().toMillis();
        this.minDiskCheckGapMs = this.dnConf.getDiskCheckMinGap().toMillis();
        this.lastAllVolumeSetsCheckComplete = timer.monotonicNow() - this.minDiskCheckGapMs;
        this.registeredVolumeSets = new ArrayList<VolumeSet>();
        this.delegateChecker = new ThrottledAsyncChecker<Boolean, VolumeCheckResult>(timer, this.minDiskCheckGapMs, this.maxAllowedTimeForCheckMs, Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat(threadNamePrefix + "DataNodeDiskCheckerThread-%d").setDaemon(true).build()));
        this.checkVolumeResultHandlerExecutorService = Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat(threadNamePrefix + "VolumeCheckResultHandlerThread-%d").setDaemon(true).build());
        ThreadFactory threadFactory = r -> {
            Thread t = new Thread(r, threadNamePrefix + "PeriodicHDDSVolumeChecker");
            t.setDaemon(true);
            return t;
        };
        this.diskCheckerservice = Executors.newSingleThreadScheduledExecutor(threadFactory);
        this.started = new AtomicBoolean(false);
    }

    public void start() {
        if (this.started.compareAndSet(false, true)) {
            long periodicDiskCheckIntervalMinutes = this.dnConf.getPeriodicDiskCheckIntervalMinutes();
            this.periodicDiskChecker = this.diskCheckerservice.scheduleWithFixedDelay(this::checkAllVolumeSets, periodicDiskCheckIntervalMinutes, periodicDiskCheckIntervalMinutes, TimeUnit.MINUTES);
        }
    }

    public synchronized void registerVolumeSet(VolumeSet volumeSet) {
        this.registeredVolumeSets.add(volumeSet);
    }

    public synchronized void checkAllVolumeSets() {
        long gap = this.timer.monotonicNow() - this.lastAllVolumeSetsCheckComplete;
        if (gap < this.minDiskCheckGapMs) {
            this.numSkippedChecks.incrementAndGet();
            if (LOG.isTraceEnabled()) {
                LOG.trace("Skipped checking all volumes, time since last check {} is less than the minimum gap between checks ({} ms).", (Object)gap, (Object)this.minDiskCheckGapMs);
            }
            return;
        }
        try {
            for (VolumeSet volSet : this.registeredVolumeSets) {
                volSet.checkAllVolumes(this);
            }
            this.lastAllVolumeSetsCheckComplete = this.timer.monotonicNow();
            this.numAllVolumeSetsChecks.incrementAndGet();
        }
        catch (IOException e) {
            LOG.warn("Exception while checking disks", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<? extends StorageVolume> checkAllVolumes(Collection<? extends StorageVolume> volumes) throws InterruptedException {
        ConcurrentHashMap.KeySetView healthyVolumes = ConcurrentHashMap.newKeySet();
        ConcurrentHashMap.KeySetView failedVolumes = ConcurrentHashMap.newKeySet();
        HashSet<StorageVolume> allVolumes = new HashSet<StorageVolume>();
        AtomicLong numVolumes = new AtomicLong(volumes.size());
        CountDownLatch latch = new CountDownLatch(1);
        for (StorageVolume storageVolume : volumes) {
            Optional<ListenableFuture<VolumeCheckResult>> olf = this.delegateChecker.schedule(storageVolume, null);
            if (olf.isPresent()) {
                LOG.info("Scheduled health check for volume {}", (Object)storageVolume);
                allVolumes.add(storageVolume);
                Futures.addCallback(olf.get(), (FutureCallback)new ResultHandler(storageVolume, healthyVolumes, failedVolumes, numVolumes, (ignored1, ignored2) -> latch.countDown()), (Executor)MoreExecutors.directExecutor());
                continue;
            }
            if (numVolumes.decrementAndGet() != 0L) continue;
            latch.countDown();
        }
        if (!latch.await(this.maxAllowedTimeForCheckMs, TimeUnit.MILLISECONDS)) {
            LOG.warn("checkAllVolumes timed out after {} ms", (Object)this.maxAllowedTimeForCheckMs);
        }
        this.numAllVolumeChecks.incrementAndGet();
        StorageVolumeChecker storageVolumeChecker = this;
        synchronized (storageVolumeChecker) {
            return new HashSet(Sets.difference(allVolumes, healthyVolumes));
        }
    }

    public boolean checkVolume(StorageVolume volume, Callback callback) {
        if (volume == null) {
            LOG.debug("Cannot schedule check on null volume");
            return false;
        }
        Optional<ListenableFuture<VolumeCheckResult>> olf = this.delegateChecker.schedule(volume, null);
        if (olf.isPresent()) {
            this.numVolumeChecks.incrementAndGet();
            Futures.addCallback(olf.get(), (FutureCallback)new ResultHandler(volume, ConcurrentHashMap.newKeySet(), ConcurrentHashMap.newKeySet(), new AtomicLong(1L), callback), (Executor)this.checkVolumeResultHandlerExecutorService);
            return true;
        }
        return false;
    }

    public void shutdownAndWait(int gracePeriod, TimeUnit timeUnit) {
        if (this.started.compareAndSet(true, false)) {
            this.periodicDiskChecker.cancel(true);
            this.diskCheckerservice.shutdownNow();
            this.checkVolumeResultHandlerExecutorService.shutdownNow();
            try {
                this.delegateChecker.shutdownAndWait(gracePeriod, timeUnit);
            }
            catch (InterruptedException e) {
                LOG.warn("{} interrupted during shutdown.", (Object)this.getClass().getSimpleName());
                Thread.currentThread().interrupt();
            }
        }
    }

    @VisibleForTesting
    void setDelegateChecker(AsyncChecker<Boolean, VolumeCheckResult> testDelegate) {
        this.delegateChecker = testDelegate;
    }

    public long getNumVolumeChecks() {
        return this.numVolumeChecks.get();
    }

    public long getNumAllVolumeChecks() {
        return this.numAllVolumeChecks.get();
    }

    public long getNumAllVolumeSetsChecks() {
        return this.numAllVolumeSetsChecks.get();
    }

    public long getNumSkippedChecks() {
        return this.numSkippedChecks.get();
    }

    private static class ResultHandler
    implements FutureCallback<VolumeCheckResult> {
        private final StorageVolume volume;
        private final Set<StorageVolume> failedVolumes;
        private final Set<StorageVolume> healthyVolumes;
        private final AtomicLong volumeCounter;
        private final @Nullable Callback callback;

        ResultHandler(StorageVolume volume, Set<StorageVolume> healthyVolumes, Set<StorageVolume> failedVolumes, AtomicLong volumeCounter, @Nullable Callback callback) {
            this.volume = volume;
            this.healthyVolumes = healthyVolumes;
            this.failedVolumes = failedVolumes;
            this.volumeCounter = volumeCounter;
            this.callback = callback;
        }

        public void onSuccess(@Nullable VolumeCheckResult result) {
            if (result == null) {
                LOG.error("Unexpected empty health check result for volume {}", (Object)this.volume);
                this.markHealthy();
            } else {
                switch (result) {
                    case HEALTHY: 
                    case DEGRADED: {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Volume {} is {}.", (Object)this.volume, (Object)result);
                        }
                        this.markHealthy();
                        break;
                    }
                    case FAILED: {
                        LOG.warn("Volume {} detected as being unhealthy", (Object)this.volume);
                        this.markFailed();
                        break;
                    }
                    default: {
                        LOG.error("Unexpected health check result {} for volume {}", (Object)result, (Object)this.volume);
                        this.markHealthy();
                    }
                }
            }
            this.cleanup();
        }

        public void onFailure(@Nonnull Throwable t) {
            Throwable exception = t instanceof ExecutionException ? t.getCause() : t;
            LOG.warn("Exception running disk checks against volume {}", (Object)this.volume, (Object)exception);
            if (!(t instanceof InterruptedException)) {
                this.markFailed();
                this.cleanup();
            }
        }

        private void markHealthy() {
            this.healthyVolumes.add(this.volume);
        }

        private void markFailed() {
            this.failedVolumes.add(this.volume);
        }

        private void cleanup() {
            this.invokeCallback();
        }

        private void invokeCallback() {
            try {
                long remaining = this.volumeCounter.decrementAndGet();
                if (this.callback != null && remaining == 0L) {
                    this.callback.call(this.healthyVolumes, this.failedVolumes);
                }
            }
            catch (Exception e) {
                LOG.warn("Unexpected exception", (Throwable)e);
            }
        }
    }

    public static interface Callback {
        public void call(Set<StorageVolume> var1, Set<StorageVolume> var2) throws IOException;
    }
}

