/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.replication;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import org.apache.bookkeeper.bookie.BookieCriticalThread;
import org.apache.bookkeeper.bookie.BookieImpl;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.common.component.ComponentStarter;
import org.apache.bookkeeper.common.component.LifecycleComponent;
import org.apache.bookkeeper.common.component.LifecycleComponentStack;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.MetadataClientDriver;
import org.apache.bookkeeper.replication.Auditor;
import org.apache.bookkeeper.replication.AuditorElector;
import org.apache.bookkeeper.replication.ReplicationException;
import org.apache.bookkeeper.replication.ReplicationWorker;
import org.apache.bookkeeper.server.conf.BookieConfiguration;
import org.apache.bookkeeper.server.http.BKHttpServiceProvider;
import org.apache.bookkeeper.server.service.AutoRecoveryService;
import org.apache.bookkeeper.server.service.HttpService;
import org.apache.bookkeeper.server.service.StatsProviderService;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AutoRecoveryMain {
    private static final Logger LOG = LoggerFactory.getLogger(AutoRecoveryMain.class);
    private final ServerConfiguration conf;
    final BookKeeper bkc;
    final AuditorElector auditorElector;
    final ReplicationWorker replicationWorker;
    final AutoRecoveryDeathWatcher deathWatcher;
    int exitCode;
    private volatile boolean shuttingDown = false;
    private volatile boolean running = false;
    private volatile Thread.UncaughtExceptionHandler uncaughtExceptionHandler = null;
    private static final Options opts = new Options();

    public AutoRecoveryMain(ServerConfiguration conf) throws IOException, InterruptedException, KeeperException, ReplicationException.UnavailableException, ReplicationException.CompatibilityException {
        this(conf, (StatsLogger)NullStatsLogger.INSTANCE);
    }

    public AutoRecoveryMain(ServerConfiguration conf, StatsLogger statsLogger) throws IOException, InterruptedException, KeeperException, ReplicationException.UnavailableException, ReplicationException.CompatibilityException {
        this.conf = conf;
        this.bkc = Auditor.createBookKeeperClient(conf, statsLogger.scope("bookkeeper_client"));
        MetadataClientDriver metadataClientDriver = this.bkc.getMetadataClientDriver();
        metadataClientDriver.setSessionStateListener(() -> {
            LOG.error("Client connection to the Metadata server has expired, so shutting down AutoRecoveryMain!");
            CompletableFuture.runAsync(() -> this.shutdown(3));
        });
        this.auditorElector = new AuditorElector(BookieImpl.getBookieId(conf).toString(), conf, this.bkc, statsLogger.scope("auditor"), false);
        this.replicationWorker = new ReplicationWorker(conf, this.bkc, false, statsLogger.scope("replication_worker"));
        this.deathWatcher = new AutoRecoveryDeathWatcher(this);
    }

    public void start() {
        this.auditorElector.start();
        this.replicationWorker.start();
        if (null != this.uncaughtExceptionHandler) {
            this.deathWatcher.setUncaughtExceptionHandler(this.uncaughtExceptionHandler);
        }
        this.deathWatcher.start();
        this.running = true;
    }

    public void join() throws InterruptedException {
        this.deathWatcher.join();
    }

    public void shutdown() {
        this.shutdown(0);
    }

    private void shutdown(int exitCode) {
        LOG.info("Shutting down auto recovery: {}", (Object)exitCode);
        if (this.shuttingDown) {
            return;
        }
        LOG.info("Shutting down AutoRecovery");
        this.shuttingDown = true;
        this.running = false;
        this.exitCode = exitCode;
        try {
            this.auditorElector.shutdown();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.warn("Interrupted shutting down auditor elector", (Throwable)e);
        }
        this.replicationWorker.shutdown();
        try {
            this.bkc.close();
        }
        catch (BKException e) {
            LOG.warn("Failed to close bookkeeper client for auto recovery", (Throwable)e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.warn("Interrupted closing bookkeeper client for auto recovery", (Throwable)e);
        }
    }

    private int getExitCode() {
        return this.exitCode;
    }

    public void setExceptionHandler(Thread.UncaughtExceptionHandler exceptionHandler) {
        this.uncaughtExceptionHandler = exceptionHandler;
    }

    @VisibleForTesting
    public Auditor getAuditor() {
        return this.auditorElector.getAuditor();
    }

    @VisibleForTesting
    public ReplicationWorker getReplicationWorker() {
        return this.replicationWorker;
    }

    public boolean isAutoRecoveryRunning() {
        return this.running;
    }

    private static void printUsage() {
        HelpFormatter hf = new HelpFormatter();
        hf.printHelp("AutoRecoveryMain [options]\n", opts);
    }

    private static void loadConfFile(ServerConfiguration conf, String confFile) throws IllegalArgumentException {
        try {
            conf.loadConf(new File(confFile).toURI().toURL());
        }
        catch (MalformedURLException e) {
            LOG.error("Could not open configuration file: " + confFile, (Throwable)e);
            throw new IllegalArgumentException();
        }
        catch (ConfigurationException e) {
            LOG.error("Malformed configuration file: " + confFile, (Throwable)e);
            throw new IllegalArgumentException();
        }
        LOG.info("Using configuration file " + confFile);
    }

    private static ServerConfiguration parseArgs(String[] args) throws IllegalArgumentException {
        try {
            BasicParser parser = new BasicParser();
            CommandLine cmdLine = parser.parse(opts, args);
            if (cmdLine.hasOption('h')) {
                throw new IllegalArgumentException();
            }
            ServerConfiguration conf = new ServerConfiguration();
            CharSequence[] leftArgs = cmdLine.getArgs();
            if (cmdLine.hasOption('c')) {
                if (null != leftArgs && leftArgs.length > 0) {
                    throw new IllegalArgumentException("unexpected arguments [" + String.join((CharSequence)" ", leftArgs) + "]");
                }
                String confFile = cmdLine.getOptionValue("c");
                AutoRecoveryMain.loadConfFile(conf, confFile);
            }
            if (null != leftArgs && leftArgs.length > 0) {
                throw new IllegalArgumentException("unexpected arguments [" + String.join((CharSequence)" ", leftArgs) + "]");
            }
            return conf;
        }
        catch (ParseException e) {
            throw new IllegalArgumentException(e);
        }
    }

    public static void main(String[] args) {
        int retCode = AutoRecoveryMain.doMain(args);
        Runtime.getRuntime().exit(retCode);
    }

    static int doMain(String[] args) {
        LifecycleComponentStack server;
        ServerConfiguration conf;
        try {
            conf = AutoRecoveryMain.parseArgs(args);
        }
        catch (IllegalArgumentException iae) {
            LOG.error("Error parsing command line arguments : ", (Throwable)iae);
            if (iae.getMessage() != null) {
                System.err.println(iae.getMessage());
            }
            AutoRecoveryMain.printUsage();
            return 1;
        }
        try {
            server = AutoRecoveryMain.buildAutoRecoveryServer(new BookieConfiguration(conf));
        }
        catch (Exception e) {
            LOG.error("Failed to build AutoRecovery Server", (Throwable)e);
            return 2;
        }
        try {
            ComponentStarter.startComponent((LifecycleComponent)server).get();
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            LOG.info("AutoRecovery server is interrupted. Exiting ...");
        }
        catch (ExecutionException ee) {
            LOG.error("Error in bookie shutdown", ee.getCause());
            return 2;
        }
        return 0;
    }

    public static LifecycleComponentStack buildAutoRecoveryServer(BookieConfiguration conf) throws Exception {
        LifecycleComponentStack.Builder serverBuilder = LifecycleComponentStack.newBuilder().withName("autorecovery-server");
        StatsProviderService statsProviderService = new StatsProviderService(conf);
        StatsLogger rootStatsLogger = statsProviderService.getStatsProvider().getStatsLogger("");
        serverBuilder.addComponent((LifecycleComponent)statsProviderService);
        LOG.info("Load lifecycle component : {}", (Object)StatsProviderService.class.getName());
        AutoRecoveryService autoRecoveryService = new AutoRecoveryService(conf, rootStatsLogger);
        serverBuilder.addComponent((LifecycleComponent)autoRecoveryService);
        LOG.info("Load lifecycle component : {}", (Object)AutoRecoveryService.class.getName());
        if (conf.getServerConf().isHttpServerEnabled()) {
            BKHttpServiceProvider provider = new BKHttpServiceProvider.Builder().setAutoRecovery(autoRecoveryService.getAutoRecoveryServer()).setServerConfiguration(conf.getServerConf()).setStatsProvider(statsProviderService.getStatsProvider()).build();
            HttpService httpService = new HttpService(provider, conf, rootStatsLogger);
            serverBuilder.addComponent((LifecycleComponent)httpService);
            LOG.info("Load lifecycle component : {}", (Object)HttpService.class.getName());
        }
        return serverBuilder.build();
    }

    static {
        opts.addOption("c", "conf", true, "Bookie server configuration");
        opts.addOption("h", "help", false, "Print help message");
    }

    private class AutoRecoveryDeathWatcher
    extends BookieCriticalThread {
        private int watchInterval;
        private AutoRecoveryMain autoRecoveryMain;

        public AutoRecoveryDeathWatcher(AutoRecoveryMain autoRecoveryMain2) {
            super("AutoRecoveryDeathWatcher-" + autoRecoveryMain2.conf.getBookiePort());
            this.autoRecoveryMain = autoRecoveryMain2;
            this.watchInterval = autoRecoveryMain2.conf.getDeathWatchInterval();
            this.setUncaughtExceptionHandler((thread, cause) -> {
                LOG.info("AutoRecoveryDeathWatcher exited loop due to uncaught exception from thread {}", (Object)thread.getName(), (Object)cause);
                AutoRecoveryMain.this.shutdown();
            });
        }

        public void run() {
            do {
                try {
                    Thread.sleep(this.watchInterval);
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                }
            } while (this.autoRecoveryMain.auditorElector.isRunning() && this.autoRecoveryMain.replicationWorker.isRunning());
            LOG.info("AutoRecoveryDeathWatcher noticed the AutoRecovery is not running any more,exiting the watch loop!");
            throw new RuntimeException("AutoRecovery is not running any more");
        }
    }
}

