/*
 * Decompiled with CFR 0.152.
 */
package org.apache.uniffle.coordinator;

import java.io.Closeable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.commons.lang3.Range;
import org.apache.commons.lang3.StringUtils;
import org.apache.uniffle.common.Application;
import org.apache.uniffle.common.RemoteStorageInfo;
import org.apache.uniffle.common.util.JavaUtils;
import org.apache.uniffle.common.util.ThreadUtils;
import org.apache.uniffle.coordinator.CoordinatorConf;
import org.apache.uniffle.coordinator.QuotaManager;
import org.apache.uniffle.coordinator.access.checker.AccessQuotaChecker;
import org.apache.uniffle.coordinator.metric.CoordinatorMetrics;
import org.apache.uniffle.coordinator.strategy.storage.AppBalanceSelectStorageStrategy;
import org.apache.uniffle.coordinator.strategy.storage.LowestIOSampleCostSelectStorageStrategy;
import org.apache.uniffle.coordinator.strategy.storage.RankValue;
import org.apache.uniffle.coordinator.strategy.storage.SelectStorageStrategy;
import org.apache.uniffle.coordinator.util.CoordinatorUtils;
import org.apache.uniffle.guava.annotations.VisibleForTesting;
import org.apache.uniffle.guava.collect.Lists;
import org.apache.uniffle.guava.collect.Maps;
import org.apache.uniffle.guava.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ApplicationManager
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(ApplicationManager.class);
    private static final List<String> REMOTE_PATH_SCHEMA = Arrays.asList("hdfs");
    private final long expired;
    private final StrategyName storageStrategy;
    private final SelectStorageStrategy selectStorageStrategy;
    private final Map<String, RemoteStorageInfo> appIdToRemoteStorageInfo;
    private final Map<String, RankValue> remoteStoragePathRankValue;
    private final Map<String, String> remoteStorageToHost = JavaUtils.newConcurrentMap();
    private final Map<String, RemoteStorageInfo> availableRemoteStorageInfo;
    private final ScheduledExecutorService detectStorageScheduler;
    private final ScheduledExecutorService checkAppScheduler;
    private Map<String, Map<String, Long>> currentUserAndApp = JavaUtils.newConcurrentMap();
    private Map<String, String> appIdToUser = JavaUtils.newConcurrentMap();
    private QuotaManager quotaManager;
    private boolean hasErrorInStatusCheck = false;

    public ApplicationManager(CoordinatorConf conf) {
        this.storageStrategy = (StrategyName)((Object)conf.get(CoordinatorConf.COORDINATOR_REMOTE_STORAGE_SELECT_STRATEGY));
        this.appIdToRemoteStorageInfo = JavaUtils.newConcurrentMap();
        this.remoteStoragePathRankValue = JavaUtils.newConcurrentMap();
        this.availableRemoteStorageInfo = JavaUtils.newConcurrentMap();
        if (StrategyName.IO_SAMPLE == this.storageStrategy) {
            this.selectStorageStrategy = new LowestIOSampleCostSelectStorageStrategy(this.remoteStoragePathRankValue, this.appIdToRemoteStorageInfo, this.availableRemoteStorageInfo, conf);
        } else if (StrategyName.APP_BALANCE == this.storageStrategy) {
            this.selectStorageStrategy = new AppBalanceSelectStorageStrategy(this.remoteStoragePathRankValue, this.appIdToRemoteStorageInfo, this.availableRemoteStorageInfo, conf);
        } else {
            throw new UnsupportedOperationException("Unsupported selected storage strategy.");
        }
        this.expired = conf.getLong(CoordinatorConf.COORDINATOR_APP_EXPIRED);
        String quotaCheckerClass = AccessQuotaChecker.class.getCanonicalName();
        for (String checker : (List)conf.get(CoordinatorConf.COORDINATOR_ACCESS_CHECKERS)) {
            if (!quotaCheckerClass.equals(checker.trim())) continue;
            this.quotaManager = new QuotaManager(conf);
            this.currentUserAndApp = this.quotaManager.getCurrentUserAndApp();
            this.appIdToUser = this.quotaManager.getAppIdToUser();
            break;
        }
        this.checkAppScheduler = ThreadUtils.getDaemonSingleThreadScheduledExecutor((String)"ApplicationManager");
        this.checkAppScheduler.scheduleAtFixedRate(this::statusCheck, this.expired / 2L, this.expired / 2L, TimeUnit.MILLISECONDS);
        this.detectStorageScheduler = ThreadUtils.getDaemonSingleThreadScheduledExecutor((String)"detectStoragesScheduler");
        this.detectStorageScheduler.scheduleAtFixedRate(this.selectStorageStrategy::detectStorage, 1000L, conf.getLong(CoordinatorConf.COORDINATOR_REMOTE_STORAGE_SCHEDULE_TIME), TimeUnit.MILLISECONDS);
    }

    public void registerApplicationInfo(String appId, String user) {
        Map appAndTime = this.currentUserAndApp.computeIfAbsent(user, x -> JavaUtils.newConcurrentMap());
        this.appIdToUser.put(appId, user);
        if (!appAndTime.containsKey(appId)) {
            CoordinatorMetrics.counterTotalAppNum.inc();
            LOG.info("New application is registered: {}", (Object)appId);
        }
        if (this.quotaManager != null) {
            this.quotaManager.registerApplicationInfo(appId, appAndTime);
        } else {
            appAndTime.put(appId, System.currentTimeMillis());
        }
    }

    public void refreshAppId(String appId) {
        String user = this.appIdToUser.get(appId);
        if (user == null) {
            this.registerApplicationInfo(appId, "");
        } else {
            Map<String, Long> appAndTime = this.currentUserAndApp.get(user);
            appAndTime.put(appId, System.currentTimeMillis());
        }
    }

    public void refreshRemoteStorage(String remoteStoragePath, String remoteStorageConf) {
        if (!StringUtils.isEmpty((CharSequence)remoteStoragePath)) {
            LOG.info("Refresh remote storage with {} {}", (Object)remoteStoragePath, (Object)remoteStorageConf);
            HashSet<String> paths = Sets.newHashSet(remoteStoragePath.split(","));
            Map<String, Map<String, String>> confKVs = CoordinatorUtils.extractRemoteStorageConf(remoteStorageConf);
            for (String path : paths) {
                if (!this.availableRemoteStorageInfo.containsKey(path)) {
                    this.remoteStoragePathRankValue.computeIfAbsent(path, key -> {
                        this.addRemoteStorageMetrics(path);
                        return new RankValue(0);
                    });
                }
                String storageHost = this.getStorageHost(path);
                RemoteStorageInfo rsInfo = new RemoteStorageInfo(path, (Map)confKVs.getOrDefault(storageHost, Maps.newHashMap()));
                this.availableRemoteStorageInfo.put(path, rsInfo);
            }
            ArrayList<String> unusedPath = Lists.newArrayList();
            for (String existPath : this.availableRemoteStorageInfo.keySet()) {
                if (paths.contains(existPath)) continue;
                unusedPath.add(existPath);
            }
            for (String path : unusedPath) {
                this.availableRemoteStorageInfo.remove(path);
                this.removePathFromCounter(path);
            }
        } else {
            LOG.info("Refresh remote storage with empty value {}", (Object)remoteStoragePath);
            for (String path : this.availableRemoteStorageInfo.keySet()) {
                this.removePathFromCounter(path);
            }
            this.availableRemoteStorageInfo.clear();
        }
    }

    public RemoteStorageInfo pickRemoteStorage(String appId) {
        if (this.appIdToRemoteStorageInfo.containsKey(appId)) {
            return this.appIdToRemoteStorageInfo.get(appId);
        }
        RemoteStorageInfo pickStorage = this.selectStorageStrategy.pickStorage(appId);
        this.incRemoteStorageCounter(pickStorage.getPath());
        return this.appIdToRemoteStorageInfo.get(appId);
    }

    @VisibleForTesting
    public synchronized void incRemoteStorageCounter(String remoteStoragePath) {
        RankValue counter = this.remoteStoragePathRankValue.get(remoteStoragePath);
        if (counter != null) {
            counter.getAppNum().incrementAndGet();
        } else {
            LOG.warn("Remote storage path lost during assignment: {} doesn't exist, reset it to 1", (Object)remoteStoragePath);
            this.remoteStoragePathRankValue.put(remoteStoragePath, new RankValue(1));
        }
    }

    @VisibleForTesting
    public synchronized void decRemoteStorageCounter(String storagePath) {
        if (!StringUtils.isEmpty((CharSequence)storagePath)) {
            RankValue atomic = this.remoteStoragePathRankValue.get(storagePath);
            if (atomic != null) {
                double count = atomic.getAppNum().decrementAndGet();
                if (count < 0.0) {
                    LOG.warn("Unexpected counter for remote storage: {}, which is {}, reset to 0", (Object)storagePath, (Object)count);
                    atomic.getAppNum().set(0);
                }
            } else {
                LOG.warn("Can't find counter for remote storage: {}", (Object)storagePath);
                this.remoteStoragePathRankValue.putIfAbsent(storagePath, new RankValue(0));
            }
            if (this.remoteStoragePathRankValue.get(storagePath).getAppNum().get() == 0 && !this.availableRemoteStorageInfo.containsKey(storagePath)) {
                this.remoteStoragePathRankValue.remove(storagePath);
            }
        }
    }

    public synchronized void removePathFromCounter(String storagePath) {
        RankValue atomic = this.remoteStoragePathRankValue.get(storagePath);
        if (atomic != null && atomic.getAppNum().get() == 0) {
            this.remoteStoragePathRankValue.remove(storagePath);
        }
    }

    public Set<String> getAppIds() {
        return this.appIdToUser.keySet();
    }

    @VisibleForTesting
    public Map<String, RankValue> getRemoteStoragePathRankValue() {
        return this.remoteStoragePathRankValue;
    }

    @VisibleForTesting
    public SelectStorageStrategy getSelectStorageStrategy() {
        return this.selectStorageStrategy;
    }

    @VisibleForTesting
    public Map<String, RemoteStorageInfo> getAvailableRemoteStorageInfo() {
        return this.availableRemoteStorageInfo;
    }

    @VisibleForTesting
    public Map<String, RemoteStorageInfo> getAppIdToRemoteStorageInfo() {
        return this.appIdToRemoteStorageInfo;
    }

    @VisibleForTesting
    public boolean hasErrorInStatusCheck() {
        return this.hasErrorInStatusCheck;
    }

    @VisibleForTesting
    public void closeDetectStorageScheduler() {
        this.detectStorageScheduler.shutdownNow();
    }

    protected void statusCheck() {
        ArrayList<Map<String, Long>> appAndNums = Lists.newArrayList(this.currentUserAndApp.values());
        HashMap<String, Long> appIds = Maps.newHashMap();
        HashSet<String> expiredAppIds = Sets.newHashSet();
        try {
            for (Map map : appAndNums) {
                for (Map.Entry appAndTime : map.entrySet()) {
                    String appId = (String)appAndTime.getKey();
                    long lastReport = (Long)appAndTime.getValue();
                    appIds.put(appId, lastReport);
                    if (System.currentTimeMillis() - lastReport <= this.expired) continue;
                    expiredAppIds.add(appId);
                    map.remove(appId);
                    this.appIdToUser.remove(appId);
                }
            }
            LOG.info("Start to check status for {} applications.", (Object)appIds.size());
            for (String string : expiredAppIds) {
                LOG.info("Remove expired application : {}.", (Object)string);
                appIds.remove(string);
                if (!this.appIdToRemoteStorageInfo.containsKey(string)) continue;
                this.decRemoteStorageCounter(this.appIdToRemoteStorageInfo.get(string).getPath());
                this.appIdToRemoteStorageInfo.remove(string);
            }
            CoordinatorMetrics.gaugeRunningAppNum.set((double)appIds.size());
            this.updateRemoteStorageMetrics();
            if (this.quotaManager != null) {
                this.quotaManager.updateQuotaMetrics();
            }
        }
        catch (Exception e) {
            this.hasErrorInStatusCheck = true;
            LOG.warn("Error happened in statusCheck", (Throwable)e);
        }
    }

    private void updateRemoteStorageMetrics() {
        for (String remoteStoragePath : this.availableRemoteStorageInfo.keySet()) {
            try {
                String storageHost = this.getStorageHost(remoteStoragePath);
                CoordinatorMetrics.updateDynamicGaugeForRemoteStorage(storageHost, this.remoteStoragePathRankValue.get(remoteStoragePath).getAppNum().get());
            }
            catch (Exception e) {
                LOG.warn("Update remote storage metrics for {} failed ", (Object)remoteStoragePath);
            }
        }
    }

    private void addRemoteStorageMetrics(String remoteStoragePath) {
        String storageHost = this.getStorageHost(remoteStoragePath);
        if (!StringUtils.isEmpty((CharSequence)storageHost)) {
            CoordinatorMetrics.addDynamicGaugeForRemoteStorage(this.getStorageHost(remoteStoragePath));
            LOG.info("Add remote storage metrics for {} successfully ", (Object)remoteStoragePath);
        }
    }

    private String getStorageHost(String remoteStoragePath) {
        if (this.remoteStorageToHost.containsKey(remoteStoragePath)) {
            return this.remoteStorageToHost.get(remoteStoragePath);
        }
        String storageHost = "";
        try {
            URI uri = new URI(remoteStoragePath);
            storageHost = uri.getHost();
            this.remoteStorageToHost.put(remoteStoragePath, storageHost);
        }
        catch (URISyntaxException e) {
            LOG.warn("Invalid format of remoteStoragePath to get host, {}", (Object)remoteStoragePath);
        }
        return storageHost;
    }

    public List<Application> getApplications(Set<String> appIds, int pageSize, int currentPage, String pHeartBeatStartTime, String pHeartBeatEndTime, String appIdRegex) {
        ArrayList applications = new ArrayList();
        for (Map.Entry<String, Map<String, Long>> entry : this.currentUserAndApp.entrySet()) {
            String user = entry.getKey();
            Map<String, Long> apps = entry.getValue();
            apps.forEach((appId, heartBeatTime) -> {
                boolean match;
                boolean bl = match = appIds.size() == 0 || appIds.contains(appId);
                if (StringUtils.isNotBlank((CharSequence)appIdRegex) && match) {
                    boolean bl2 = match = match && this.matchApplicationId((String)appId, appIdRegex);
                }
                if (StringUtils.isNotBlank((CharSequence)pHeartBeatStartTime) || StringUtils.isNotBlank((CharSequence)pHeartBeatEndTime)) {
                    match = this.matchHeartBeatStartTimeAndEndTime(pHeartBeatStartTime, pHeartBeatEndTime, (long)heartBeatTime);
                }
                if (match) {
                    RemoteStorageInfo remoteStorageInfo = this.appIdToRemoteStorageInfo.getOrDefault(appId, null);
                    Application application = new Application.Builder().applicationId(appId).user(user).lastHeartBeatTime(heartBeatTime.longValue()).remoteStoragePath(remoteStorageInfo).build();
                    applications.add(application);
                }
            });
        }
        Collections.sort(applications);
        int startIndex = (currentPage - 1) * pageSize;
        int endIndex = Math.min(startIndex + pageSize, applications.size());
        LOG.info("getApplications >> appIds = {}.", applications);
        return applications.subList(startIndex, endIndex);
    }

    private boolean matchApplicationId(String applicationId, String regex) {
        Pattern pattern = Pattern.compile(regex);
        return pattern.matcher(applicationId).matches();
    }

    private boolean matchHeartBeatStartTimeAndEndTime(String pStartTime, String pEndTime, long appHeartBeatTime) {
        long startTime = 0L;
        long endTime = Long.MAX_VALUE;
        if (StringUtils.isNotBlank((CharSequence)pStartTime)) {
            startTime = this.parseLongValue(pStartTime, "heartBeatStartTime");
        }
        if (StringUtils.isNotBlank((CharSequence)pEndTime)) {
            endTime = this.parseLongValue(pEndTime, "heartBeatEndTime");
        }
        Range heartBeatTime = Range.between((Comparable)Long.valueOf(startTime), (Comparable)Long.valueOf(endTime));
        return heartBeatTime.contains((Object)appHeartBeatTime);
    }

    private long parseLongValue(String strValue, String fieldName) {
        try {
            return Long.parseLong(strValue);
        }
        catch (NumberFormatException e) {
            throw new NumberFormatException(fieldName + " value must be a number!");
        }
    }

    public Map<String, Integer> getDefaultUserApps() {
        return this.quotaManager.getDefaultUserApps();
    }

    public QuotaManager getQuotaManager() {
        return this.quotaManager;
    }

    public static List<String> getPathSchema() {
        return REMOTE_PATH_SCHEMA;
    }

    @Override
    public void close() {
        if (this.detectStorageScheduler != null) {
            this.detectStorageScheduler.shutdownNow();
        }
        if (this.checkAppScheduler != null) {
            this.checkAppScheduler.shutdownNow();
        }
    }

    public static enum StrategyName {
        APP_BALANCE,
        IO_SAMPLE;

    }
}

