# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['msoffcrypto', 'msoffcrypto.format', 'msoffcrypto.method']

package_data = \
{'': ['*']}

install_requires = \
['cryptography>=2.3', 'olefile>=0.45']

entry_points = \
{'console_scripts': ['msoffcrypto-tool = msoffcrypto.__main__:main']}

setup_kwargs = {
    'name': 'msoffcrypto-tool',
    'version': '5.0.0',
    'description': 'Python tool and library for decrypting MS Office files with passwords or other keys',
    'long_description': '# msoffcrypto-tool\n\n[![PyPI](https://img.shields.io/pypi/v/msoffcrypto-tool.svg)](https://pypi.org/project/msoffcrypto-tool/)\n[![PyPI downloads](https://img.shields.io/pypi/dm/msoffcrypto-tool.svg)](https://pypistats.org/packages/msoffcrypto-tool)\n[![Build Status](https://travis-ci.com/nolze/msoffcrypto-tool.svg?branch=master)](https://travis-ci.com/nolze/msoffcrypto-tool)\n[![Coverage Status](https://codecov.io/gh/nolze/msoffcrypto-tool/branch/master/graph/badge.svg)](https://codecov.io/gh/nolze/msoffcrypto-tool)\n[![Documentation Status](https://readthedocs.org/projects/msoffcrypto-tool/badge/?version=latest)](http://msoffcrypto-tool.readthedocs.io/en/latest/?badge=latest)\n\nmsoffcrypto-tool (formerly ms-offcrypto-tool) is Python tool and library for decrypting encrypted MS Office files with password, intermediate key, or private key which generated its escrow key.\n\n## Contents\n\n* [Install](#install)\n* [Examples](#examples)\n* [Supported encryption methods](#supported-encryption-methods)\n* [Tests](#tests)\n* [Todo](#todo)\n* [Resources](#resources)\n* [Use cases and mentions](#use-cases-and-mentions)\n* [Contributors](#contributors)\n\n## Install\n\n```\npip install msoffcrypto-tool\n```\n\n## Examples\n\n### As CLI tool (with password)\n\n```\nmsoffcrypto-tool encrypted.docx decrypted.docx -p Passw0rd\n```\n\nPassword is prompted if you omit the password argument value:\n\n```bash\n$ msoffcrypto-tool encrypted.docx decrypted.docx -p\nPassword:\n```\n\nTest if the file is encrypted or not (exit code 0 or 1 is returned):\n\n```\nmsoffcrypto-tool document.doc --test -v\n```\n\n### As library\n\nPassword and more key types are supported with library functions.\n\nBasic usage:\n\n```python\nimport msoffcrypto\n\nencrypted = open("encrypted.docx", "rb")\nfile = msoffcrypto.OfficeFile(encrypted)\n\nfile.load_key(password="Passw0rd")  # Use password\n\nwith open("decrypted.docx", "wb") as f:\n    file.decrypt(f)\n\nencrypted.close()\n```\n\nBasic usage (in-memory):\n\n```python\nimport msoffcrypto\nimport io\nimport pandas as pd\n\ndecrypted = io.BytesIO()\n\nwith open("encrypted.xlsx", "rb") as f:\n    file = msoffcrypto.OfficeFile(f)\n    file.load_key(password="Passw0rd")  # Use password\n    file.decrypt(decrypted)\n\ndf = pd.read_excel(decrypted)\nprint(df)\n```\n\nAdvanced usage:\n\n```python\n# Verify password before decryption (default: False)\n# The ECMA-376 Agile/Standard crypto system allows one to know whether the supplied password is correct before actually decrypting the file\n# Currently, the verify_password option is only meaningful for ECMA-376 Agile/Standard Encryption\nfile.load_key(password="Passw0rd", verify_password=True)\n\n# Use private key\nfile.load_key(private_key=open("priv.pem", "rb"))\n\n# Use intermediate key (secretKey)\nfile.load_key(secret_key=binascii.unhexlify("AE8C36E68B4BB9EA46E5544A5FDB6693875B2FDE1507CBC65C8BCF99E25C2562"))\n\n# Check the HMAC of the data payload before decryption (default: False)\n# Currently, the verify_integrity option is only meaningful for ECMA-376 Agile Encryption\nfile.decrypt(open("decrypted.docx", "wb"), verify_integrity=True)\n```\n\n## Supported encryption methods\n\n### MS-OFFCRYPTO specs\n\n* [x] ECMA-376 (Agile Encryption/Standard Encryption)\n  * [x] MS-DOCX (OOXML) (Word 2007-2016)\n  * [x] MS-XLSX (OOXML) (Excel 2007-2016)\n  * [x] MS-PPTX (OOXML) (PowerPoint 2007-2016)\n* [x] Office Binary Document RC4 CryptoAPI\n  * [x] MS-DOC (Word 2002, 2003, 2004)\n  * [x] MS-XLS (Excel 2002, 2003, 2004) (experimental)\n  * [x] MS-PPT (PowerPoint 2002, 2003, 2004) (partial, experimental)\n* [x] Office Binary Document RC4\n  * [x] MS-DOC (Word 97, 98, 2000)\n  * [x] MS-XLS (Excel 97, 98, 2000) (experimental)\n* [ ] ECMA-376 (Extensible Encryption)\n* [ ] XOR Obfuscation\n\n### Other\n\n* [ ] Word 95 Encryption (Word 95 and prior)\n* [ ] Excel 95 Encryption (Excel 95 and prior)\n* [ ] PowerPoint 95 Encryption (PowerPoint 95 and prior)\n\nPRs are welcome!\n\n## Tests\n\nWith [coverage](https://github.com/nedbat/coveragepy) and [pytest](https://pytest.org/):\n\n```\npoetry install\npoetry run coverage run -m pytest -v\n```\n\n## Todo\n\n* [x] Add tests\n* [x] Support decryption with passwords\n* [x] Support older encryption schemes\n* [x] Add function-level tests\n* [x] Add API documents\n* [x] Publish to PyPI\n* [x] Add decryption tests for various file formats\n* [x] Integrate with more comprehensive projects handling MS Office files (such as [oletools](https://github.com/decalage2/oletools/)?) if possible\n* [x] Add the password prompt mode for CLI\n* [x] Improve error types (v4.12.0)\n* [ ] Redesign APIs (v5.0.0)\n* [ ] Introduce something like `ctypes.Structure`\n* [ ] Support encryption\n* [ ] Isolate parser\n\n## Resources\n\n* "Backdooring MS Office documents with secret master keys" <http://secuinside.com/archive/2015/2015-1-9.pdf>\n* Technical Documents <https://msdn.microsoft.com/en-us/library/cc313105.aspx>\n  * [MS-OFFCRYPTO] Agile Encryption <https://msdn.microsoft.com/en-us/library/dd949735(v=office.12).aspx>\n* LibreOffice/core <https://github.com/LibreOffice/core>\n* LibreOffice/mso-dumper <https://github.com/LibreOffice/mso-dumper>\n* wvDecrypt <http://www.skynet.ie/~caolan/Packages/wvDecrypt.html>\n* Microsoft Office password protection - Wikipedia <https://en.wikipedia.org/wiki/Microsoft_Office_password_protection#History_of_Microsoft_Encryption_password>\n* office2john.py <https://github.com/magnumripper/JohnTheRipper/blob/bleeding-jumbo/run/office2john.py>\n\n## Alternatives\n\n* herumi/msoffice <https://github.com/herumi/msoffice>\n* DocRecrypt <https://blogs.technet.microsoft.com/office_resource_kit/2013/01/23/now-you-can-reset-or-remove-a-password-from-a-word-excel-or-powerpoint-filewith-office-2013/>\n* Apache POI - the Java API for Microsoft Documents <https://poi.apache.org/>\n\n## Use cases and mentions\n\n### General\n\n* <https://repology.org/project/python:msoffcrypto-tool/versions> (kudos to maintainers!)\n* <https://checkroth.com/unlocking-password-protected-files.html>\n\n### Malware/maldoc analysis\n\n* <https://github.com/jbremer/sflock/commit/3f6a96abe1dbb4405e4fb7fd0d16863f634b09fb>\n* <https://isc.sans.edu/forums/diary/Video+Analyzing+Encrypted+Malicious+Office+Documents/24572/>\n\n### CTF\n\n* <https://github.com/shombo/cyberstakes-writeps-2018/tree/master/word_up>\n* <https://github.com/willi123yao/Cyberthon2020_Writeups/blob/master/csit/Lost_Magic>\n\n### In other languages\n\n* <https://github.com/dtjohnson/xlsx-populate>\n* <https://github.com/opendocument-app/OpenDocument.core/blob/233663b039/src/internal/ooxml/ooxml_crypto.h>\n\n## Contributors\n\n* <https://github.com/nolze/msoffcrypto-tool/graphs/contributors>\n',
    'author': 'nolze',
    'author_email': 'nolze@archlinux.us',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/nolze/msoffcrypto-tool',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
