/**
 * Copyright (c) 2011 Zoltan Ujhelyi, Mark Czotter, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *    Zoltan Ujhelyi, Mark Czotter - initial API and implementation
 *    Andras Okros - minor changes
 */
package org.eclipse.incquery.patternlanguage.emf.jvmmodel;

import com.google.inject.Inject;
import java.util.Arrays;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.incquery.patternlanguage.emf.eMFPatternLanguage.PatternModel;
import org.eclipse.incquery.patternlanguage.emf.jvmmodel.PatternGroupClassInferrer;
import org.eclipse.incquery.patternlanguage.emf.jvmmodel.PatternMatchClassInferrer;
import org.eclipse.incquery.patternlanguage.emf.jvmmodel.PatternMatchProcessorClassInferrer;
import org.eclipse.incquery.patternlanguage.emf.jvmmodel.PatternMatcherClassInferrer;
import org.eclipse.incquery.patternlanguage.emf.jvmmodel.PatternQuerySpecificationClassInferrer;
import org.eclipse.incquery.patternlanguage.emf.specification.SpecificationBuilder;
import org.eclipse.incquery.patternlanguage.emf.util.EMFJvmTypesBuilder;
import org.eclipse.incquery.patternlanguage.emf.util.EMFPatternLanguageJvmModelInferrerUtil;
import org.eclipse.incquery.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.incquery.patternlanguage.helper.CorePatternLanguageHelper;
import org.eclipse.incquery.patternlanguage.patternLanguage.Pattern;
import org.eclipse.incquery.runtime.api.IPatternMatch;
import org.eclipse.incquery.runtime.api.IncQueryMatcher;
import org.eclipse.incquery.runtime.api.impl.BaseMatcher;
import org.eclipse.incquery.runtime.api.impl.BasePatternMatch;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.xbase.jvmmodel.AbstractModelInferrer;
import org.eclipse.xtext.xbase.jvmmodel.IJvmDeclaredTypeAcceptor;
import org.eclipse.xtext.xbase.jvmmodel.IJvmModelAssociator;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * <p>Infers a JVM model from the source model.</p>
 * 
 * <p>The JVM model should contain all elements that would appear in the Java code
 * which is generated from the source model. Other models link against the JVM model rather than the source model.</p>
 */
@SuppressWarnings("all")
public class EMFPatternLanguageJvmModelInferrer extends AbstractModelInferrer {
  private final static String JVM_MODEL_INFERRER_PREFIX = "org.eclipse.incquery.patternlanguage.emf.inferrer";
  
  public final static String INVALID_PATTERN_MODEL_CODE = (EMFPatternLanguageJvmModelInferrer.JVM_MODEL_INFERRER_PREFIX + ".invalid.patternmodel");
  
  public final static String INVALID_TYPEREF_CODE = (EMFPatternLanguageJvmModelInferrer.JVM_MODEL_INFERRER_PREFIX + ".invalid.typeref");
  
  public final static String SPECIFICATION_BUILDER_CODE = (EMFPatternLanguageJvmModelInferrer.JVM_MODEL_INFERRER_PREFIX + ".specification.builder");
  
  @Inject
  private Logger logger;
  
  @Inject
  private IErrorFeedback errorFeedback;
  
  /**
   * convenience API to build and initialize JvmTypes and their members.
   */
  @Inject
  @Extension
  private EMFJvmTypesBuilder _eMFJvmTypesBuilder;
  
  @Inject
  @Extension
  private EMFPatternLanguageJvmModelInferrerUtil _eMFPatternLanguageJvmModelInferrerUtil;
  
  @Inject
  @Extension
  private PatternMatchClassInferrer _patternMatchClassInferrer;
  
  @Inject
  @Extension
  private PatternMatcherClassInferrer _patternMatcherClassInferrer;
  
  @Inject
  @Extension
  private PatternQuerySpecificationClassInferrer _patternQuerySpecificationClassInferrer;
  
  @Inject
  @Extension
  private PatternMatchProcessorClassInferrer _patternMatchProcessorClassInferrer;
  
  @Inject
  @Extension
  private PatternGroupClassInferrer _patternGroupClassInferrer;
  
  @Inject
  @Extension
  private IJvmModelAssociator associator;
  
  /**
   * Is called for each Pattern instance in a resource.
   * 
   * @param pattern - the model to create one or more JvmDeclaredTypes from.
   * @param acceptor - each created JvmDeclaredType without a container should be passed to the acceptor in order get attached to the
   *                   current resource.
   * @param isPreLinkingPhase - whether the method is called in a pre linking phase, i.e. when the global index isn't fully updated. You
   *        must not rely on linking using the index if this is <code>true</code>
   */
  public void infer(final Pattern pattern, final IJvmDeclaredTypeAcceptor acceptor, final SpecificationBuilder builder, final boolean isPrelinkingPhase) {
    String _name = pattern.getName();
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(_name);
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      boolean _isPrivate = CorePatternLanguageHelper.isPrivate(pattern);
      final boolean isPublic = (!_isPrivate);
      try {
        if (isPublic) {
          this.inferPublic(pattern, acceptor, builder, isPrelinkingPhase);
        } else {
          this.inferPrivate(pattern, acceptor, builder, isPrelinkingPhase);
        }
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          this.logger.error(("Exception during Jvm Model Infer for: " + pattern), e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
  }
  
  private void inferPublic(final Pattern pattern, final IJvmDeclaredTypeAcceptor acceptor, final SpecificationBuilder builder, final boolean isPrelinkingPhase) {
    String _name = pattern.getName();
    String _plus = ("Inferring Jvm Model for pattern" + _name);
    this.logger.debug(_plus);
    final String packageName = this._eMFPatternLanguageJvmModelInferrerUtil.getPackageName(pattern);
    final String utilPackageName = this._eMFPatternLanguageJvmModelInferrerUtil.getUtilPackageName(pattern);
    String _matchClassName = this._eMFPatternLanguageJvmModelInferrerUtil.matchClassName(pattern);
    final Procedure1<JvmGenericType> _function = new Procedure1<JvmGenericType>() {
      @Override
      public void apply(final JvmGenericType it) {
        it.setPackageName(packageName);
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _typeRef = EMFPatternLanguageJvmModelInferrer.this._typeReferenceBuilder.typeRef(BasePatternMatch.class);
        EMFPatternLanguageJvmModelInferrer.this._eMFJvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
      }
    };
    final JvmGenericType matchClass = this._eMFJvmTypesBuilder.toClass(pattern, _matchClassName, _function);
    String _matcherClassName = this._eMFPatternLanguageJvmModelInferrerUtil.matcherClassName(pattern);
    final Procedure1<JvmGenericType> _function_1 = new Procedure1<JvmGenericType>() {
      @Override
      public void apply(final JvmGenericType it) {
        it.setPackageName(packageName);
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _typeRef = EMFPatternLanguageJvmModelInferrer.this._typeReferenceBuilder.typeRef(matchClass);
        JvmTypeReference _typeRef_1 = EMFPatternLanguageJvmModelInferrer.this._typeReferenceBuilder.typeRef(BaseMatcher.class, _typeRef);
        EMFPatternLanguageJvmModelInferrer.this._eMFJvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef_1);
      }
    };
    final JvmGenericType matcherClass = this._eMFJvmTypesBuilder.toClass(pattern, _matcherClassName, _function_1);
    final JvmDeclaredType querySpecificationClass = this._patternQuerySpecificationClassInferrer.inferQuerySpecificationClass(pattern, isPrelinkingPhase, utilPackageName, matcherClass, this._typeReferenceBuilder, this._annotationTypesBuilder);
    final JvmDeclaredType processorClass = this._patternMatchProcessorClassInferrer.inferProcessorClass(pattern, isPrelinkingPhase, utilPackageName, matchClass, this._typeReferenceBuilder, this._annotationTypesBuilder);
    final Procedure1<JvmDeclaredType> _function_2 = new Procedure1<JvmDeclaredType>() {
      @Override
      public void apply(final JvmDeclaredType it) {
        EMFPatternLanguageJvmModelInferrer.this._patternQuerySpecificationClassInferrer.initializePublicSpecification(it, pattern, matcherClass, matchClass, builder);
      }
    };
    acceptor.<JvmDeclaredType>accept(querySpecificationClass, _function_2);
    final Procedure1<JvmDeclaredType> _function_3 = new Procedure1<JvmDeclaredType>() {
      @Override
      public void apply(final JvmDeclaredType it) {
        EMFPatternLanguageJvmModelInferrer.this._patternMatchProcessorClassInferrer.inferProcessorClassMethods(processorClass, pattern, matchClass);
      }
    };
    acceptor.<JvmDeclaredType>accept(processorClass, _function_3);
    final Procedure1<JvmGenericType> _function_4 = new Procedure1<JvmGenericType>() {
      @Override
      public void apply(final JvmGenericType it) {
        EMFPatternLanguageJvmModelInferrer.this._patternMatchClassInferrer.inferMatchClassElements(it, pattern, querySpecificationClass, EMFPatternLanguageJvmModelInferrer.this._typeReferenceBuilder, EMFPatternLanguageJvmModelInferrer.this._annotationTypesBuilder);
      }
    };
    acceptor.<JvmGenericType>accept(matchClass, _function_4);
    final Procedure1<JvmGenericType> _function_5 = new Procedure1<JvmGenericType>() {
      @Override
      public void apply(final JvmGenericType it) {
        EMFPatternLanguageJvmModelInferrer.this._patternMatcherClassInferrer.inferMatcherClassElements(it, pattern, querySpecificationClass, matchClass, EMFPatternLanguageJvmModelInferrer.this._typeReferenceBuilder, EMFPatternLanguageJvmModelInferrer.this._annotationTypesBuilder);
      }
    };
    acceptor.<JvmGenericType>accept(matcherClass, _function_5);
    this.associator.associatePrimary(pattern, matcherClass);
    this.associator.associate(pattern, matcherClass);
    this.associator.associate(pattern, querySpecificationClass);
    this.associator.associate(pattern, processorClass);
  }
  
  private void inferPrivate(final Pattern pattern, final IJvmDeclaredTypeAcceptor acceptor, final SpecificationBuilder builder, final boolean isPrelinkingPhase) {
    String _name = pattern.getName();
    String _plus = ("Inferring Jvm Model for private pattern " + _name);
    this.logger.debug(_plus);
    final String utilPackageName = this._eMFPatternLanguageJvmModelInferrerUtil.getUtilPackageName(pattern);
    JvmTypeReference _typeRef = this._typeReferenceBuilder.typeRef(IPatternMatch.class);
    JvmTypeReference _typeRef_1 = this._typeReferenceBuilder.typeRef(IncQueryMatcher.class, _typeRef);
    final JvmType matcherClass = _typeRef_1.getType();
    final JvmDeclaredType querySpecificationClass = this._patternQuerySpecificationClassInferrer.inferQuerySpecificationClass(pattern, isPrelinkingPhase, utilPackageName, matcherClass, this._typeReferenceBuilder, this._annotationTypesBuilder);
    this.associator.associatePrimary(pattern, querySpecificationClass);
    final Procedure1<JvmDeclaredType> _function = new Procedure1<JvmDeclaredType>() {
      @Override
      public void apply(final JvmDeclaredType it) {
        EMFPatternLanguageJvmModelInferrer.this._patternQuerySpecificationClassInferrer.initializePrivateSpecification(querySpecificationClass, pattern, matcherClass, null, builder);
      }
    };
    acceptor.<JvmDeclaredType>accept(querySpecificationClass, _function);
  }
  
  /**
   * Is called for each PatternModel instance in a resource.
   * 
   * @param model - the model to create one or more JvmDeclaredTypes from.
   * @param acceptor - each created JvmDeclaredType without a container should be passed to the acceptor in order get attached to the
   *                   current resource.
   * @param isPreLinkingPhase - whether the method is called in a pre linking phase, i.e. when the global index isn't fully updated. You
   *        must not rely on linking using the index if this is <code>true</code>
   */
  protected void _infer(final PatternModel model, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPrelinkingPhase) {
    try {
      final SpecificationBuilder builder = new SpecificationBuilder();
      EList<Pattern> _patterns = model.getPatterns();
      for (final Pattern pattern : _patterns) {
        this.infer(pattern, acceptor, builder, isPrelinkingPhase);
      }
      String _modelFileName = this._eMFPatternLanguageJvmModelInferrerUtil.modelFileName(model);
      String _plus = ("Inferring Jvm Model for Pattern model " + _modelFileName);
      this.logger.debug(_plus);
      EList<Pattern> _patterns_1 = model.getPatterns();
      boolean _isEmpty = _patterns_1.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final JvmGenericType groupClass = this._patternGroupClassInferrer.inferPatternGroupClass(model, this._typeReferenceBuilder);
        final Procedure1<JvmGenericType> _function = new Procedure1<JvmGenericType>() {
          @Override
          public void apply(final JvmGenericType it) {
            EMFPatternLanguageJvmModelInferrer.this._patternGroupClassInferrer.initializePatternGroup(it, model, EMFPatternLanguageJvmModelInferrer.this._typeReferenceBuilder);
          }
        };
        acceptor.<JvmGenericType>accept(groupClass, _function);
        this.associator.associatePrimary(model, groupClass);
      }
    } catch (final Throwable _t) {
      if (_t instanceof IllegalArgumentException) {
        final IllegalArgumentException e = (IllegalArgumentException)_t;
        String _message = e.getMessage();
        this.errorFeedback.reportErrorNoLocation(model, _message, EMFPatternLanguageJvmModelInferrer.INVALID_PATTERN_MODEL_CODE, Severity.ERROR, IErrorFeedback.JVMINFERENCE_ERROR_TYPE);
      } else if (_t instanceof Exception) {
        final Exception e_1 = (Exception)_t;
        this.logger.error(("Exception during Jvm Model Infer for pattern model: " + model), e_1);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  public void infer(final EObject model, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPrelinkingPhase) {
    if (model instanceof PatternModel) {
      _infer((PatternModel)model, acceptor, isPrelinkingPhase);
      return;
    } else if (model != null) {
      _infer(model, acceptor, isPrelinkingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(model, acceptor, isPrelinkingPhase).toString());
    }
  }
}
