/*******************************************************************************
 * Copyright (c) 2003, 2013 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.debug.internal.ui.sourcelookup;

import java.util.Iterator;

import org.eclipse.debug.core.sourcelookup.ISourceContainer;
import org.eclipse.debug.core.sourcelookup.ISourceContainerType;
import org.eclipse.debug.core.sourcelookup.ISourceLookupDirector;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.sourcelookup.ISourceContainerBrowser;
import org.eclipse.jface.viewers.IStructuredSelection;

/**
 * Action used to edit source containers on a source lookup path
 */
public class EditContainerAction extends SourceContainerAction {

	private ISourceLookupDirector fDirector;
	private ISourceContainer[] fContainers;
	private ISourceContainerBrowser fBrowser;

	public EditContainerAction() {
		super(SourceLookupUIMessages.EditContainerAction_0);
	}

	/**
	 * Prompts for a project to add.
	 *
	 * @see org.eclipse.jface.action.IAction#run()
	 */
	@Override
	public void run() {
		ISourceContainer[] replacements = fBrowser.editSourceContainers(getShell(), fDirector, fContainers);
		int j = 0;
		ISourceContainer[] existing = getViewer().getEntries();
		for (int i = 0; i < existing.length && j < replacements.length; i++) {
			ISourceContainer toBeReplaced = fContainers[j];
			ISourceContainer container = existing[i];
			if (container.equals(toBeReplaced)) {
				existing[i] = replacements[j];
				j++;
			}
		}
		getViewer().setEntries(existing);
	}

	public void setSourceLookupDirector(ISourceLookupDirector director) {
		fDirector = director;
	}

	@Override
	protected boolean updateSelection(IStructuredSelection selection) {
		if(selection == null || selection.isEmpty()) {
			return false;
		}
		if (getViewer().getTree().getSelection()[0].getParentItem()==null) {
			// can only edit top level items of same type
			fContainers = new ISourceContainer[selection.size()];
			Iterator<ISourceContainer> iterator = selection.iterator();
			ISourceContainer container = iterator.next();
			ISourceContainerType type = container.getType();
			fContainers[0] = container;
			int i = 1;
			while (iterator.hasNext()) {
				container = iterator.next();
				fContainers[i] = container;
				i++;
				if (!container.getType().equals(type)) {
					return false;
				}
			}
			// all the same type, see if editing is supported
			fBrowser = DebugUITools.getSourceContainerBrowser(type.getId());
			if (fBrowser != null) {
				return fBrowser.canEditSourceContainers(fDirector, fContainers);
			}
		}
		return false;
	}
}
