/********************************************************************************
 * Copyright (c) 2007 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the terms
 * of the Eclipse Public License v1.0 which accompanies this distribution, and is 
 * available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Initial Contributors:
 * The following IBM employees contributed to the Remote System Explorer
 * component that contains this file: David McKnight.
 * 
 * Contributors:
 * Martin Oberhuber (Wind River) - [168975] Move RSE Events API to Core
 * Martin Oberhuber (Wind River) - [177523] Unify singleton getter methods
 * Martin Oberhuber (Wind River) - [186773] split ISystemRegistryUI from ISystemRegistry
 ********************************************************************************/

package org.eclipsecon.tmtutorial.jarsigning;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.rse.core.RSECorePlugin;
import org.eclipse.rse.core.events.ISystemResourceChangeEvents;
import org.eclipse.rse.core.events.SystemResourceChangeEvent;
import org.eclipse.rse.core.model.IHost;
import org.eclipse.rse.core.model.ISystemProfile;
import org.eclipse.rse.core.model.ISystemRegistry;
import org.eclipse.rse.core.model.SystemRemoteResourceSet;
import org.eclipse.rse.core.model.SystemWorkspaceResourceSet;
import org.eclipse.rse.core.subsystems.ISubSystem;
import org.eclipse.rse.core.subsystems.SubSystem;
import org.eclipse.rse.files.ui.dialogs.SystemRemoteFolderDialog;
import org.eclipse.rse.files.ui.resources.UniversalFileTransferUtility;
import org.eclipse.rse.services.clientserver.messages.SystemMessageException;
import org.eclipse.rse.services.files.IFileService;
import org.eclipse.rse.shells.ui.RemoteCommandHelpers;
import org.eclipse.rse.subsystems.files.core.model.RemoteFileUtility;
import org.eclipse.rse.subsystems.files.core.subsystems.IRemoteFile;
import org.eclipse.rse.subsystems.files.core.subsystems.IRemoteFileSubSystem;
import org.eclipse.rse.subsystems.shells.core.model.SimpleCommandOperation;
import org.eclipse.rse.subsystems.shells.core.subsystems.IRemoteCmdSubSystem;
import org.eclipse.rse.subsystems.shells.core.subsystems.IRemoteCommandShell;
import org.eclipse.rse.ui.SystemBasePlugin;
import org.eclipse.rse.ui.messages.SystemMessageDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipsecon.tmtutorial.host.CreateEclipseHostActionDelegate;


public class JarSigningActionDelegate implements IObjectActionDelegate 
{
    private List _selectedFiles = new ArrayList();
    private static String defaultDirPath = "/home/data/httpd/download-staging.priv"; //$NON-NLS-1$
    
    private class JarSigningJob extends Job
    {

		public JarSigningJob(String name) {
			super(name);
		}

		public IStatus run(final IProgressMonitor monitor) 
		{
			monitor.beginTask("Signing Jars", 20 + 80*_selectedFiles.size());
			
	    	// upload the jar(s)
			IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 10);
	    	SystemRemoteResourceSet set = exportJars(_selectedFiles, subMonitor);
	    	if (set==null) return Status.CANCEL_STATUS;
	    	
	    	//End of Example 4: Remove return stmt co continue with Example 5
	    	//if (set!=null) return Status.OK_STATUS;
	    	
	    	// sign the jar(s)
	    	List unsignedSet = new ArrayList();
	    	SystemRemoteResourceSet signedSet = new SystemRemoteResourceSet(set.getSubSystem());
	    	for (int i = 0; i < set.size(); i++) {
	    		if (monitor.isCanceled()) return Status.CANCEL_STATUS;
				subMonitor = new SubProgressMonitor(monitor, 80);
	    		IRemoteFile obj = (IRemoteFile)set.get(i);
	    		IRemoteFile signed = signJar(obj, subMonitor);
	    		if (signed!=null) {
	    			unsignedSet.add(_selectedFiles.get(i));
	    			signedSet.addResource(signed);
	    			//mark stale to ensure it will be downloaded completely
	    			signed.markStale(true, true);
	    		}
	    	}
	    	
	    	// download the jar(s)
			subMonitor = new SubProgressMonitor(monitor, 10);
	    	if (signedSet.size()>0) {
		    	List resultJars = downloadFiles(signedSet, unsignedSet, subMonitor);
		    	System.out.println("signing done: "+resultJars);
	    	}
	    	
	    	monitor.done();
	    	return Status.OK_STATUS;
		}
    	
		/**
	     * Upload the selected jar files to a user-specified host where the signing
	     * will take place (by default build.eclipse.org)
	     * 
	     * @param selectedFiles the set of Eclipse IFiles to sign
	     * @param monitor a progress monitor for uploading
	     * @return the set of uploaded IRemoteFiles on the specified host, 
	     * 		or <code>null</code> in case of an error or canceled operation
	     */
	    private SystemRemoteResourceSet exportJars(List selectedFiles, final IProgressMonitor monitor)
	    {
	    	SystemRemoteResourceSet results = null;
	    	monitor.beginTask("Exporting Jars", 5*selectedFiles.size() + 16);
	    	
	    	// get the host where the jar signer resides
	    	monitor.subTask("Getting host for signing");
	    	IHost theHost = getEclipseBuildHost();
	    	if (theHost == null) {
	    		// if this host object doesn't yet exist... create it
	    		CreateEclipseHostActionDelegate createHostAction = new CreateEclipseHostActionDelegate();
	    		createHostAction.run(null);
	    		theHost = getEclipseBuildHost();
	    	}
	    	monitor.worked(1);

	    	// get the IRemoteFileSubSystem
    		final ISystemRegistry registry = RSECorePlugin.getTheSystemRegistry();
    		final IRemoteFileSubSystem fss = RemoteFileUtility.getFileSubSystem(theHost);
    		
    		// make sure we are connected
    		if (!fss.isConnected()) {
    	    	monitor.subTask("Connecting");
    			connect(fss, new SubProgressMonitor(monitor, 5));
    		} else {
    			monitor.worked(5);
    		}
    		
    		// get the signing location
    		monitor.subTask("Getting signing location");
    		// Read IRemoteFile into cache while still in background job
    		// This is in order to speed up display of the remote folder dialog
    		try {
        		IRemoteFile defaultDir = fss.getRemoteFileObject(defaultDirPath, null);
        		monitor.worked(1);
        		fss.list(defaultDir.getParentRemoteFile(), IFileService.FILE_TYPE_FILES, monitor);
        		monitor.worked(1);
    		} catch(Exception e) {
    			e.printStackTrace();
    		}
    		
    		// Select remote folder
    		final IRemoteFile[] rc = new IRemoteFile[1];
    		rc[0] = null;
    		Display dis = Display.getCurrent();
    		if (dis==null) dis = Display.getDefault();
    		if (dis!=null) {
    			dis.syncExec(new Runnable() {
    				public void run() {
    					String title="Select Remote Jar Signing Location";
    		    		IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 10);
    		    		rc[0] = promptForFolder(fss, defaultDirPath, title, subMonitor);
    				}
    			});
    		}

	    	if (rc[0]!=null) {
	    		IRemoteFile targetFolder = rc[0];
	    		defaultDirPath = targetFolder.getAbsolutePath();
	    		
	    		// create a workspace set to contain each of the IFiles to upload
	    		SystemWorkspaceResourceSet workspaceSet = new SystemWorkspaceResourceSet();
	    		for (int i = 0; i < _selectedFiles.size(); i++) {
	    			workspaceSet.addResource(_selectedFiles.get(i));
	    		}
	    		
	    		// use the UniversalFileTransferUtility to upload the jars
	    		monitor.subTask("Uploading");
	    		IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 5*_selectedFiles.size());
	    		results = UniversalFileTransferUtility.copyWorkspaceResourcesToRemote(workspaceSet, targetFolder, subMonitor, false);
	    		
	    		// refresh parent (if applicable in ui)
	    		targetFolder.markStale(true);

	    		// fire a refresh event on the system registry
	    		final IRemoteFile finFolder = targetFolder;
	    		Display.getDefault().asyncExec(new Runnable() {
	    			public void run() {
	    	    		registry.fireEvent(new SystemResourceChangeEvent(
	    	    				finFolder, ISystemResourceChangeEvents.EVENT_REFRESH, finFolder));
	    			}
	    		});
	    	}
	    	monitor.done();
	    	return results;
	    }

	    /**
	     * Connect the given subsystem from a background job.
	     * @param subsys Subsytem to connect
	     * @param monitor Progress monitor
	     */
		private void connect(final ISubSystem subsys, final IProgressMonitor monitor)
		{
			monitor.beginTask("Connecting", 10);
			if (!subsys.isConnected()) {
				//TODO API is missing fss.connect(IProgressMonitor) -> do it manually
    			//fss.connect();
				final Exception[] exception=new Exception[1];
				exception[0]=null;
				Display.getDefault().syncExec(new Runnable() {
					public void run() {
						try {
							((SubSystem)subsys).promptForPassword();
						} catch(Exception e) {
							exception[0]=e;
						}
					}
				});
				try {
					subsys.getConnectorService().connect(monitor);
				} catch(Exception e) {
					e.printStackTrace();
				}
				if (subsys.isConnected()) {
					//Notify connect status change
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							RSECorePlugin.getTheSystemRegistry().connectedStatusChange(subsys, true, false);
						}
					});
				}
			}
			monitor.done();
		}
		
	    /**
	     * Open a dialog to prompt for a folder on the remote file subsystem.
	     * Prerequisite: file subsystem must already be connected.
	     * This must be called on the display thread since it opens a dialog.
	     * 
	     * @param fss Remote File Subsystem on which to prompt
	     * @param defaultPath absolute path with default folder to prompt for
	     * @param title Title for prompting dialog
	     * @param monitor Progress Monitor for long-running operations like connect
	     * @return IRemoteFile of selected folder, or <code>null</code> in case of
	     *     an error or canceled operation
	     */
	    private IRemoteFile promptForFolder(final IRemoteFileSubSystem fss,
	    		final String defaultPath, final String title, final IProgressMonitor monitor)
	    {
	    	IRemoteFile result = null;
	    	monitor.beginTask("Prompt for Folder", 11);
	    	Shell shell = SystemBasePlugin.getActiveWorkbenchShell();
    		try {
    	    	// reusable RSE dialog for browsing folders of remote systems
    			if (fss.isConnected()) {
        	    	SystemRemoteFolderDialog dlg = new SystemRemoteFolderDialog(
        	    		shell, title, fss.getHost());
        	    	//TODO API currently missing a progress monitor here!
            		IRemoteFile defaultDir = fss.getRemoteFileObject(defaultPath, null);
            		monitor.worked(1);
            		dlg.setPreSelection(defaultDir);
            		monitor.worked(10);
        	    	int dlgResult = dlg.open();
        	    	
        	    	// get selected file
        	    	if (dlgResult==Window.OK) {
        		    	Object output = dlg.getOutputObject();
        		    	if (output instanceof IRemoteFile) {
        		    		result = (IRemoteFile)output;
        		    	}
        	    	}
    			}
    		} catch(SystemMessageException e) {
    			SystemMessageDialog msgdlg = new SystemMessageDialog(shell, e.getSystemMessage());
    			msgdlg.open();
    		} catch(Exception e) {
    			e.printStackTrace();
    		}
    		monitor.done();
		    return result;
	    }
	    
	    /**
	     * Sign a remote file with the script on build.eclipse.org.
	     * The file must already be in a proper location for signing.
	     * @param jarToSign
	     * @param monitor
	     * @return IRemoteFile the signed file, or <code>null</code> in case of an error.
	     */
	    private IRemoteFile signJar(final IRemoteFile jarToSign, final IProgressMonitor monitor)
	    {
	    	IRemoteFile result = null;
	    	IRemoteCommandShell cmdShell = null;
	    	monitor.beginTask("Signing", 130);
	    	IHost host = jarToSign.getSystemConnection();
	    	IRemoteFileSubSystem fileSS = RemoteFileUtility.getFileSubSystem(host);
	    	IRemoteCmdSubSystem cmdSS = RemoteCommandHelpers.getCmdSubSystem(host);

			try {
				//check if already signed: see also nm example
		    	monitor.subTask("Checking signed status for "+jarToSign.getName());
		    	SimpleCommandOperation op = new SimpleCommandOperation(cmdSS, jarToSign.getParentRemoteFile(), true);
		    	String cmd="jarsigner -verify " + jarToSign.getAbsolutePath(); //$NON-NLS-1$
		    	op.runCommand(cmd, true);
		    	cmdShell = op.getCommandShell();
				String line = op.readLine(true);
				while (line != null && !monitor.isCanceled()) {
					monitor.worked(1);
					//System.out.println(line);
					line = op.readLine(true);
					if ("jar verified.".equals(line)) {
						//jar is already verified -> nothing to do
						System.out.println("Already signed: "+jarToSign.getName());
						op.removeShell();
						monitor.done();
						return null;
					}
				}
				op.removeShell();
				cmdShell=null;
				
				//get or create the output dir
		    	monitor.subTask("Checking output space");
		    	IRemoteFile parent = jarToSign.getParentRemoteFile();
		    	IRemoteFile outdir = fileSS.getRemoteFileObject(parent, "rseout", null);
		    	if (!outdir.exists()) fileSS.createFolder(outdir, null);
		    	monitor.worked(1);

		    	//ensure the target does not exist yet
				IRemoteFile outputFile = fileSS.getRemoteFileObject(outdir, jarToSign.getName(), null);
				if (outputFile.exists()) {
					fileSS.delete(outputFile, monitor);
				}
		    	monitor.worked(1);

				//send the command
		    	monitor.subTask("Signing "+jarToSign.getName());
		    	op = new SimpleCommandOperation(cmdSS, jarToSign.getParentRemoteFile(), true);
		    	cmd="sign " + jarToSign.getAbsolutePath() + " nomail " + outdir.getAbsolutePath(); //$NON-NLS-1$ //$NON-NLS-2$
		    	op.runCommand(cmd, true);
		    	cmdShell = op.getCommandShell();
				
				//wait for completion locally
				long maxWait = System.currentTimeMillis() + 120000; //max 2 minutes
				while(System.currentTimeMillis() < maxWait && !monitor.isCanceled()) {
					outputFile.markStale(true, true);
					outputFile = fileSS.getRemoteFileObject(outputFile.getAbsolutePath(), null);
					if (outputFile.exists()) {
						//Also check that signed file is longer than original one
						//To ensure we do not try downloading while the file is not complete yet
						if (outputFile.getLength() > jarToSign.getLength()) {
							result = outputFile;
							break;
						} else {
							outputFile.markStale(true, true);
						}
					}
					Thread.sleep(1000);
					monitor.worked(1);
				}
			}
			catch (SystemMessageException msg) {
		    	Shell shell = SystemBasePlugin.getActiveWorkbenchShell();
    			final SystemMessageDialog msgdlg = new SystemMessageDialog(shell, msg.getSystemMessage());
    			Display.getDefault().syncExec(new Runnable() {
    				public void run() { msgdlg.open(); }
    			});
			}
			catch (Exception e) {
				e.printStackTrace();
			}
			finally {
				if (cmdShell!=null) try { 
					cmdSS.removeShell(cmdShell);
				} catch(Exception e) { 
					e.printStackTrace();
				}
			}
			monitor.done();
			return result;
	    }
	    
	    /**
	     * Download a list of remote files into a list of local workspace files
	     * @param remoteFiles list of remote files to download
	     * @param targetFiles list of {@link IFile} target files in workspace
	     * @param monitor Monitor for progress reporting and cancellation
	     * @return List of {@link IFile} of successfully downloaded files.
	     */
	    private List downloadFiles(final SystemRemoteResourceSet remoteFiles, 
	    		final List targetFiles, final IProgressMonitor monitor)
	    {
	    	List results = new ArrayList();
	    	monitor.beginTask("Downloading", 20+remoteFiles.size()*2);
	    	
	    	// download the set of remote files to the RSE temp files cache
	    	monitor.subTask("Downloading files");
	    	final SystemWorkspaceResourceSet[] tempFiles = new SystemWorkspaceResourceSet[1];
	    	final IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 20);
	    	////TODO We need to run this in the display thread for now, because
	    	////in FileServiceSubSystem:454, a dialog is not pushed to the display thread
	    	////This is a known bug
			//tempFiles[0] = UniversalFileTransferUtility.copyRemoteResourcesToWorkspace(remoteFiles, subMonitor);	
	    	Display.getDefault().syncExec(new Runnable() {
	    		public void run() {
	    			//Run in display thread because error messages may be shown
	    			tempFiles[0] = UniversalFileTransferUtility.copyRemoteResourcesToWorkspace(remoteFiles, subMonitor);	
	    		}
	    	});
	    	
	    	// copy the temp files back to the initial project
	    	IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
	    	for (int i = 0; i < tempFiles[0].size() && !monitor.isCanceled(); i++)	{
	    		IFile file = (IFile)tempFiles[0].get(i);	    		
	    		IFile originalFile = (IFile)targetFiles.get(i);
	    			    		
	    		IPath path = originalFile.getFullPath();
	    		IPath backup = path.addFileExtension("orig"); //$NON-NLS-1$
	    		try {
	    			if (!workspaceRoot.getFile(backup).exists()) {
	    				originalFile.move(backup, false, monitor);
	    			} else {
	    				originalFile.delete(true, true, monitor);
	    			}
	    			monitor.worked(1);
	    			file.copy(path, true, monitor);
	    			monitor.worked(1);
	    			results.add(path);
	    		}
	    		catch (Exception e) {
	    			e.printStackTrace();
	    			break;
	    		}	    		
	    	}
	    	monitor.done();
	    	return results;
	    }
	    
	    /**
	     * Returns the host object for the jar signing server
	     * @return the host where to sign jars
	     */
	    private IHost getEclipseBuildHost()
	    {
	    	ISystemRegistry registry = RSECorePlugin.getTheSystemRegistry();
	    	
	    	ISystemProfile profile = registry.getSystemProfileManager().getDefaultPrivateSystemProfile();
	    	return registry.getHost(profile, "build.eclipse.org"); //$NON-NLS-1$
	    }
    }

	 public void run(IAction action) 
	 {
		 JarSigningJob job = new JarSigningJob("Sign Jars");
		 job.schedule();
	 }

    

    /** (non-Javadoc)
     * Method declared on IActionDelegate
     */
    public void selectionChanged(IAction action, ISelection selection) {
    	_selectedFiles.clear();
    	// store the selected jars to be used when running
    	Iterator theSet = ((IStructuredSelection)selection).iterator();
    	while (theSet.hasNext())
    	 {
    		 Object obj = theSet.next();
    		 if (obj instanceof IFile)
    		 {
    			 _selectedFiles.add(obj);
    		 }
    	 }
    }

    /** (non-Javadoc)
     * Method declared on IObjectActionDelegate
     */
    public void setActivePart(IAction action, IWorkbenchPart targetPart) {
    }
}
