/* mem.c - Memory routines.
   Copyright (C) 2001 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

/* Written by Marc Tardif <intmktg@cam.org>.  */

/* The memory allocation algorithm in this program can be summarised
   with the following illustration:

    contents      next                                           lim
       v            v                                             v
      +-----------------------------------------------------------+
      *
   pointer

   The pointer variable contains the address of the previously allo-
   cated object.  The first pointer is NULL to identify the object
   as the last element in the list.  This creates a singly linked
   list which can be traversed backwards.

   The memory block starts immediately after the pointer variable at
   the address of contents.  For the current active block of memory
   (the last allocated), two other pointers are maintained:
   - next, the address of the next available character.
   - lim, the end of the current memory block.

   Both the pointer and memory block are allocated in a single call
   to malloc.  This is accomplished by requesting memory past the
   contents variable and manipulating this area with the pointers
   mentioned above.
*/

#include <stdlib.h>

#include "mem.h"

#define BLOCK_SIZE 4096

/* Block of memory distributed in segments.                              */
struct block
{
  struct block *prev;         /* Address of prior block or NULL.         */
  char contents[4];           /* Objects begin here.                     */
};

/* Control current object in current block.                              */
struct mem
{
  void *block;                /* Address of current struct block.        */
  char *lim;                  /* Address of char after current block.    */
  char *next;                 /* Next char to current block.             */
};

static struct mem *mem;

void *
mem_init (int extra)
{
  register struct block *block;

  mem = malloc (ALIGN (sizeof (struct mem)) + BLOCK_SIZE + extra);
  if (!mem)
    return 0;

  block = mem->block = (struct block *)
                         ((char *)mem + ALIGN (sizeof (struct mem)));
  mem->next = block->contents + extra;
  mem->lim = (char *) block + BLOCK_SIZE + extra;
  block->prev = NULL;
  return mem->next - extra;
}

void *
mem_alloc (int length)
{
  register void *ptr;

  if (mem->next + length > mem->lim)
    {
      register struct block *block;
      register long new_size;

      new_size = length + sizeof (struct block *);
      new_size += BLOCK_SIZE - (new_size % BLOCK_SIZE);

      block = malloc (new_size);
      if (!block)
        return NULL;

      block->prev = mem->block;
      mem->block = block;
      mem->next = block->contents;
      mem->lim = (char *) block + new_size;
    }
  ptr = mem->next;
  mem->next = (char *)ALIGN (mem->next + length);
  return ptr;
}

void
mem_free (void)
{
  register struct block *lp, *plp;

  lp = mem->block;
  while (lp->prev != 0)
    {
      plp = lp->prev;
      free(lp);
      lp = plp;
    }
  free(mem);
}

