/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is MacroTracker Bug Tracking System Source Code
 *
 * The Initial Developer of the Original Code is
 * R.J. Keller.
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */
package org.mozdev.MacroTracker.classes;

import org.mozdev.MacroTracker.classes.User;
import org.mozdev.MacroTracker.classes.Product;
import org.mozdev.MacroTracker.classes.bug.*;
import java.util.*;
import java.io.Serializable;

import com.trfenv.rlk.Debug;

/**
 *Class Bug represents a bug held in the MacroTracker bug tracking system.
 *
 *The Bug holds any resolution, status, assignee, reporter, and other
 *information that describes who should fix this bug, general info about
 *the bug, and information that can help the bug to become the FIXED
 *status.
 *@author R.J. Keller <rlk@trfenv.com>
 *@version 1.0
 */
public class Bug implements Serializable
{
	private int         mBugNumber;
	private Status      mStatus;
	private Status      mResolution;
	private User        mAssignedTo;
	private double      mVersion;
	private byte        mPriority;
	private String      mSeverity;
	private User        mQAContact;
	private Product     mProduct;
	private String      mComponent;
	private String      mDescription;
	private String      mURL;
	private String      mSummary;
	private String      mStatusWhiteboard;
	private String      mKeywords;
	private String      mHardware;
	private String      mOS;
	private double      mTargetMilestone;
	private User        mReporter;

	private final List comments = new LinkedList();
	private final List cc = new LinkedList();
	private final List blocks = new LinkedList();
	private final List dependsOn = new LinkedList();
	
	//---- USER COMMENTS ----//
	public void addUserComment(UserComment aUserComment) { comments.add(aUserComment); }
	public Iterator commentIterator() { return comments.iterator(); }

	public void addCC(User aUserComment) { cc.add(aUserComment); }
	public User getCC(int aIndex) { return (User)cc.get(aIndex); }
	public int ccSize() { return cc.size(); }
	public Iterator ccIterator() { return cc.iterator(); }

	public void addBlocker(Bug aBug) { blocks.add(aBug); }
	public Bug getBlocker(int aIndex) { return (Bug)blocks.get(aIndex); }
	public int blockerSize() { return comments.size(); }
	public Iterator blockerIterator() { return blocks.iterator(); }

	public void addDependency(Bug aBug) { dependsOn.add(aBug); }
	public Bug getDependency(int aIndex) { return (Bug)dependsOn.get(aIndex); }
	public int dependencySize() { return dependsOn.size(); }
	public Iterator dependencyIterator() { return dependsOn.iterator(); }

	/** Creates a Bug object with default settings (everything 0). */
	public Bug()
	{
		this(0, "");
	}

	/** Creates a Bug object with the bug number set. */
	public Bug(int bugNumber)
	{
		this(bugNumber, "");
	}

	/** Creates a Bug object with the Bug number and the summary set
	 *in the constructor.
	 *@param bugNumber The number of the bug.
	 * (Precondition: bugNumber > 0)
	 *@param summary The summary of the bug.
	 */
	public Bug(int bugNumber, String summary)
	{
		mBugNumber = bugNumber;
		mVersion = 0.0d;
		mPriority = 0;
		mAssignedTo = null;
		mQAContact = null;
		mProduct = null;
		mComponent = "";
		mDescription = "";
		mURL = "";
		mSummary = summary;
		mStatusWhiteboard = "";
		mKeywords = "";
		mHardware = "";
		mOS = "";
		mTargetMilestone = 0.0d;
		mReporter = null;

		Debug.createObject(this);
	}

	/** Returns the bug number of the bug. */
	public int getBugNumber()		{	return mBugNumber; }
	/** Returns the status of this bug according to the status constants. */
	public Status getStatus()		{	return mStatus; }
	/** Returns the resolution (if any) of this bug according to the status constants.*/
	public Status getResolution()	{	return mResolution; }
	/** Returns the name of the person responsible for fixing the bug. */
	public User getAssignedTo()	{	return mAssignedTo; }
	/** Returns the product version this bug was found in. */
	public double getVersion()		{	return mVersion; }
	/** Returns the priority of this bug to get fixed. */
	public int getPriority()		{	return mPriority; }
	/** Returns the severity of the bug.*/
	public String getSeverity()		{	return mSeverity; }
	/** Returns the QA Contact of this bug. */
	public User getQAContact()		{	return mQAContact; }
	/** Returns the Product this bug was found in. */
	public Product getProduct()		{	return mProduct; }
	/** Returns the component of the Product the bug was found in. */
	public String getComponent()	{	return mComponent; }
	/** Returns a long description of the bug. */
	public String getDescription()	{	return mDescription; }
	/** Returns the URL that may contain a testcase or other information
	 *related to the bug.
	 */
	public String getURL()			{	return mURL; }
	/** Returns a short summary of the bug.*/
	public String getSummary()		{	return mSummary; }
	/** Returns the Status Whiteboard of the bug. */
	public String getStatusWB()		{	return mStatusWhiteboard; }
	/** Returns any keywords that this bug describes. */
	public String getKeywords()		{	return mKeywords; }
	/** Returns the Platform the bug was found in.*/
	public String getHardware()		{	return mHardware; }
	/** Returns the Operating System that the bug in the Product was found
	 *in.
	 */
	public String getOS()			{	return mOS; }
	/** Returns the version of the product the bug is planned on being fixed
	 *in.
	 */
	public double getTM()			{	return mTargetMilestone; }
	/** Returns the User who reported the bug.*/
	public User getReporter()		{	return mReporter; }

	/** Changes the bug number of the current bug.
	 *@param bugNumber The number the bug should be.<br>(Precondition: bugNumber > 0)
	 *@throws IllegalArgumentException
	 */
	public void setBugNumber(int bugNumber) throws IllegalArgumentException
	{
		if (bugNumber > 0)
			mBugNumber = bugNumber;
		else
			Debug.printException(new IllegalArgumentException("Cannot set the bug number to this number."));
	}

	/**
	 *The status field indicates the general health of a bug.
	 *@param status The status object to make this bug's status equal.
	 */
	public void setStatus(Status status)
	{
		mStatus = status;
	}
	/**
	 *A <strong>resolution</strong> field indicates what happened to this bug. See the
	 *resolution constants for more information on specific resolution.
	 *@param resolution The resolution object to set this bug's resolution to.
	 */
	public void setResolution(Status resolution)
	{
		mResolution = resolution;
	}
	/**
	 *This is the person in charge of resolving the bug. Every time this field
	 *changes, the status changes to NEW to make it easy to see which new bugs
	 *have appeared on a person's list. The default status for queries is set
	 *to NEW, ASSIGNED and REOPENED. When searching for bugs that have been
	 *resolved or verified, remember to set the status field appropriately.
	 */
	public void setAssignedTo(User name)
	{
		mAssignedTo = name;
	}

	/**
	 *Sets the version of the Product that this bug was found in.
	 *@param version The version of the product that this bug was found in.<br>(precondition: version > 0)
	 */
	public void setVersion(double version)
	{
		if (version > 0)
			mVersion = version;
		else
			Debug.printException(new IllegalArgumentException("Cannot set the version to this number."));
	}

	/**
	 *Sets how high of a priority this bug is to be fixed (on a scale of 1
	 *to 5, 1 being the highest).
	 *@param priority The priority of this bug.
	 */
	public void setPriority(byte priority)
	{
		if (priority > 0 && priority <= 5)
			mPriority = priority;
		else
			Debug.printException(new IllegalArgumentException("Invalid Priority given (should this even happen?)"));
	}

	/**
	 *Sets the severity of the bug according to the severity constants.
	 *@param severity The severity of the bug.<br>(precondition: severity == BLOCKER, CRITICAL, MAJOR, MINOR, TRIVIAL, ENHANCEMENT, or NORMAL)
	 */
	public void setSeverity(String severity)
	{
		mSeverity = severity;
	}

	/** Sets the person who is responsible for QA and making sure the resolution
	 *was the approripate one.
	 *@param qaContact The person who is responsible for QA work on the bug.
	 */
	public void setQAContact(User qaContact)
	{
		mQAContact = qaContact;
	}

	public void setProduct(Product product)
	{
		this.mProduct = product;
	}

	public void setComponent(String component)
	{
		mComponent = component;
	}

	/**
	 *Sets the description of the bug.
	 *@param description An expanded version of the summary.
	 */
	public void setDescription(String description)
	{
		this.mDescription = description;
	}

	/**
	 *URL that demonostrates the problem you are seeing.
	 */
	public void setURL(String url)
	{
		this.mURL = url;
	}


	/** Sets the summary, which should be a sentence that summarizes the problem.
	 *@param summary The summary of the bug.
	 */
	public void setSummary(String summary)
	{
		mSummary = summary;
	}

	/** Sets the status whiteboard of the bug.
	 *@param statusWhtbd The text to set in the status whiteboard field.
	 */
	public void setStatusWB(String statusWhtbd)
	{
		mStatusWhiteboard = statusWhtbd;
	}

	/** Sets the hardware that the bug found in the Product was found on.
	 *@param hardware The hardware the bug was found on.
	 */
	public void setHardware(String hardware)
	{
		this.mHardware = hardware;
	}

	/** The operating system that the bug in the Product was found on.
	 *@param os The OS this bug was found on.
	 */
	public void setOS(String os)
	{
		this.mOS = os;
	}

	/** Sets the estimated date that this bug should be fixed by.
	 *@param tm The date this bug should be fixed by.
	 */
	public void setTargetMilestone(double tm)
	{
		this.mTargetMilestone = tm;
	}

	/** Sets the person who reported this bug.
	 *@param reporter The reporter of this bug.
	 */
	public void setReporter(User reporter)
	{
		this.mReporter = reporter;
	}
}
