using System;
using System.Collections.Generic;
using System.Text;
using CoreSync;
using System.Windows.Forms;

namespace SmartSync
{
    #region Known actions that can be applied over a Contact
    class Actions
    {
        public const String NOTHING = "0";
        public const String ADD = "1";
        public const String UPDATE = "2";
        public const String UPDATEXML = "4";
        public const String DELETE = "8";

        public const String ADDPPC = "64";
        public const String UPDATEPPC = "128";
        public const String DELETEPPC = "256";

        public const String QUESTION = "2048";
    }
    #endregion

    class Synchronizer
    {
        private void setProgress(progress pg, int iStatus)
        {
            if (pg != null)
                pg.setStep(iStatus);
        }
        public void Synchronize()
        {
            Synchronize(null);
        }
        public void Synchronize(progress pg)
        {
            #region 1. Loading data
            Utils IO = new Utils();
            Comparer comparer = new Comparer();
            setProgress(pg, 1);
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Loading local contacts from XML file.");
            Contacts local = IO.LoadLocalContacts();
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Done.");
            setProgress(pg, 2);
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Loading remote (PocketPC) contacts from device.");
            Contacts remote = IO.LoadRemoteContacts();
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Done.");
            #endregion
            #region 2. Initializing sync engine
            if (local == null)
                local = new Contacts();
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Initializing synchronization engine.");
            comparer.setLocalList(local);
            comparer.setRemoteList(remote);
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Done.");
            //IO.SaveLocalContacts(remote);
            #endregion
            #region 3. Synchronizing
            setProgress(pg, 3);
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Performing synchronization.");
            comparer.MergeLists();
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Done.");
            #endregion
            #region 4. Saving contact lists
            setProgress(pg, 4);
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Saving local contact list.");
            IO.SaveLocalContacts(comparer.Merged);
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Done.");
            setProgress(pg, 5);
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Saving device contact list.");
            IO.SaveRemoteContacts(comparer.Merged);
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Done.");
            #endregion
            #region 5. Saving comparison tables
            setProgress(pg, 6);
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Saving comparison tables.");
            ComparisonHelper helper = new ComparisonHelper();
            helper.populate(comparer.Merged);
            helper.SaveComparisonTable();
            Logger.AddEntry(Logger.INFO, "Synchronizer", "Done.");
            #endregion
            setProgress(pg, 7);
        }
    }

    class Utils
    {
        #region Private attributes
        private String pvt_sFileName = "";
        #endregion

        #region Class constructors
        public Utils()
        {
            /*
             * Default constructor with no PocketPC identification
             */
            pvt_sFileName = AppSettings.getAppDir() + "\\PCContacts.xml";
        }

        public Utils(String PPCName)
        {
            /*
             * Constructor with PocketPC identification
             */
            pvt_sFileName = AppSettings.getAppDir() + "\\" + PPCName + "_Contacts.xml";
        }
        #endregion

        #region Contact loading methods: LoadLocalContacts(), LoadRemoteContacts()
        public Contacts LoadLocalContacts()
        {
            try
            {
                System.IO.StreamReader reader = new System.IO.StreamReader(pvt_sFileName);
                String xmlData = reader.ReadToEnd();
                Contacts loaded = XMLDataExchange.ContactsFromXML(xmlData);
                reader.Close();
                Contacts retval = new Contacts();
                /*
                 * Since the list loaded is not used to command
                 *  actions over the PocketPC but to command
                 *  actions over the Thunderbird extension, the
                 *  contacts which action is DELETEPPC must not
                 *  be loaded.
                 */
                if (loaded != null)
                {
                    for (int i = 0; i < loaded.getCount(); i++)
                    {
                        Contact oCt = loaded.get(i);
                        if (oCt.action != Actions.DELETEPPC)
                            retval.add(oCt);
                    }
                }

                return retval;
            }
            catch (Exception ex)
            {
                Logger.AddEntry(Logger.EXCEPTION, "Utils.LoadLocalContacts", "An exception has been raised. Exception description: " + ex.Message);
                return null;
            }
        }

        public Contacts LoadRemoteContacts()
        {
            try
            {
                Contacts retval;
                retval = ObjectStore.getContacts();
                EnsureUIDPresence(ref retval);
                return retval;
            }
            catch (Exception ex)
            {
                Logger.AddEntry(Logger.EXCEPTION, "Utils.LoadRemoteContacts", "An exception has been raised. Exception description: " + ex.Message);
                return null;
            }
        }
        #endregion

        #region Locally useful functions
        private void EnsureUIDPresence(ref Contacts oList)
        {
            /*
             * Since Unique identifiers are the core of this
             *  piece of software, we ensure that EVERY contact has
             *  its own Unique ID.
             */
            for (int i = 0; i < oList.getCount(); i++)
            {
                Contact oCt = oList.get(i);
                if (oCt.customID == "" || oCt.customID=="0")
                {
                    oCt.customID = Guid.NewGuid().ToString();
                    ObjectStore.putContact(oCt);
                }
            }
        }
        private Contacts copyList(Contacts source)
        {
            Contacts list = new Contacts();
            for (int i = 0; i < source.getCount(); i++)
            {
                list.add(source.get(i));
            }
            return list;
        }
        #endregion

        #region Contact saving methods: SaveLocalContacts(oList), SaveRemoteContacts(oList)
        public bool SaveLocalContacts(Contacts oList)
        {
            try
            {
                Contacts toSave = oList;
                /*
                 * Since this list is produced to command actions
                 *  of the Thunderbird sync extension, contacts
                 *  deleted from PocketPC should not be added to
                 *  this list. It's useless and may lead to sync
                 *  problems.
                 * Still, it can be useful to restore contacts that
                 *  have been deleted by error.
                 */

                /*
                toSave = new Contacts();
                bool noAdd=false;
                for (int i = 0; i < oList.getCount(); i++)
                {
                    Contact oCt = oList.get(i);
                    noAdd = (oCt.action == Actions.DELETEPPC);
                    if (oCt.action == Actions.ADDPPC)
                        oCt.action = Actions.NOTHING;
                    if (oCt.action == Actions.UPDATEPPC)
                        oCt.action = Actions.NOTHING;
                    if (noAdd == false)
                        toSave.add(oCt);
                }
                // */

                System.IO.StreamWriter writer = new System.IO.StreamWriter(pvt_sFileName, false, Encoding.UTF8);
                String xmlData = XMLDataExchange.ContactsToXML(toSave);
                writer.Write(xmlData);
                writer.Flush();
                writer.Close();
                return true;
            }
            catch (Exception ex)
            {
                Logger.AddEntry(Logger.EXCEPTION, "Utils.SaveLocalContacts", "An exception has been raised. Exception description: " + ex.Message);
                return false;
            }
        }

        public bool SaveRemoteContacts(Contacts oList)
        {
            try
            {
                for (int i = 0; i < oList.getCount(); i++)
                {
                    Contact ct = oList.get(i);
                    switch (ct.action)
                    {
                        case Actions.ADDPPC:
                            Logger.AddEntry(Logger.INFO, "Utils.SaveRemoteContacts", "Adding a contact on the PocketPC.");
                            ct.OID = 0;
                            ObjectStore.putContact(ct);
                            break;
                        case Actions.DELETEPPC:
                            Logger.AddEntry(Logger.INFO, "Utils.SaveRemoteContacts", "Removing a contact from the PocketPC.");
                            ObjectStore.removeContact(ct.OID);
                            break;
                        case Actions.UPDATEPPC:
                            Logger.AddEntry(Logger.INFO, "Utils.SaveRemoteContacts", "Updating a contact on the PocketPC.");
                            ObjectStore.putContact(ct);
                            break;
                    }
                }
                return true;
            }
            catch (Exception ex)
            {
                Logger.AddEntry(Logger.EXCEPTION, "Utils.SaveRemoteContacts", "An exception has been raised. Exception description: " + ex.Message);
                return false;
            }
        }
        #endregion
    }

    class Comparer
    {
        #region Private attributes
        private Contacts pvt_oLocalList=null;
        private Contacts pvt_oRemoteList=null;
        private Contacts pvt_oMerged=null;
        #endregion

        #region Class properties: Merged
        public Contacts Merged
        {
            get
            {
                if (pvt_oMerged == null)
                    MergeLists();
                return pvt_oMerged;
            }
        }
        #endregion

        #region 1. Initialization methods: setLocalList(oList), setRemoteList(oList)
        public void setLocalList(Contacts oList)
        {
            pvt_oLocalList = copyList(oList);
        }

        public void setRemoteList(Contacts oList)
        {
            pvt_oRemoteList = copyList(oList);
        }
        #endregion

        #region 2. Synchronization method: public void MergeLists()
        public void MergeLists()
        {
            #region first sync
            if (AppSettings.firstSync == true)
            {
                //if this is the first sync, the user must be prompted for action.
                // he has the following choices:
                // 1. copy PocketPC contacts on the local DB and vice-versa
                // 2. delete PocketPC contacts and replace them with local DB ones
                // 3. delete local DB contacts and replace them with PocketPC ones
                // 4. do not sync and leave the flag untouched
                Logger.AddEntry(Logger.NOTIFY, "Comparer", "System detected that this is the first sync. frmFirstSync is going to be displayed.");
                frmFirstSync fs = new frmFirstSync();
                int iAction = -1;
                DialogResult result=fs.ShowDialog();
                Logger.AddEntry(Logger.NOTIFY, "Comparer", "frmFirstSync has been closed.");
                iAction = fs.ChosenAction;
                if (result == DialogResult.Cancel || iAction==4)
                    iAction = -1;
                #region PocketPC<-->LocalDB
                if (iAction==1)      // PocketPC<-->LocalDB
                {
                    pvt_oMerged = copyList(pvt_oLocalList);
                    for (int i = 0; i < pvt_oMerged.getCount(); i++)
                    {
                        Contact ct = pvt_oMerged.get(i);
                        ct.action = Actions.ADDPPC;
                    }
                    for (int i = 0; i < pvt_oRemoteList.getCount(); i++)
                    {
                        Contact ct = pvt_oRemoteList.get(i);
                        pvt_oMerged.add(ct);
                        ct.action = Actions.ADD;
                    }
                }
                #endregion
                #region LocalDB-->PocketPC
                if (iAction == 2)   //LocalDB-->PocketPC
                {
                    pvt_oMerged = copyList(pvt_oLocalList);
                    for (int i = 0; i < pvt_oMerged.getCount(); i++)
                    {
                        Contact ct = pvt_oMerged.get(i);
                        ct.action = Actions.ADDPPC;
                    }
                    for (int i = 0; i < pvt_oRemoteList.getCount(); i++)
                    {
                        Contact ct = pvt_oRemoteList.get(i);
                        pvt_oMerged.add(ct);
                        ct.action = Actions.DELETEPPC;
                    }
                }
                #endregion
                #region PocketPC-->LocalDB
                if (iAction == 3)   //PocketPC-->LocalDB
                {
                    pvt_oMerged = copyList(pvt_oLocalList);
                    for (int i = 0; i < pvt_oMerged.getCount(); i++)
                    {
                        Contact ct = pvt_oMerged.get(i);
                        ct.action = Actions.DELETE;
                    }
                    for (int i = 0; i < pvt_oRemoteList.getCount(); i++)
                    {
                        Contact ct = pvt_oRemoteList.get(i);
                        pvt_oMerged.add(ct);
                        ct.action = Actions.ADD;
                    }
                }
                #endregion
                #region No Sync
                if (iAction != -1)  // No Sync
                {
                    for (int i = 0; i < pvt_oMerged.getCount(); i++)
                    {
                        Contact ct = pvt_oMerged.get(i);
                        if (ct.customID == "")
                            ct.customID = Guid.NewGuid().ToString();
                        ComparisonHelper helper = new ComparisonHelper();
                        helper.Add(new ComparisonItem(ct.customID, CoreSync.Utils.GetMD5(ct)));
                        helper.SaveComparisonTable();
                    }
                }
                #endregion
                AppSettings.firstSync = false;
            }
            #endregion
            #region subsequent sync
            else //firstSync==false
            {
                ComparisonHelper helper = new ComparisonHelper();
                helper.LoadComparisonTable();
                // 1. create a list of added contacts (customID not in Comparison Table and not in one address book)
                // 2. create a list of deleted contacts (customID present in Comparison Table and in one address book)
                // 3. determine which contacts has been updated

                // 1
                #region contact tables declaration
                Contacts newOnPPC = new Contacts();
                Contacts newOnPC = new Contacts();
                Contacts deletedOnPPC = new Contacts();
                Contacts deletedOnPC = new Contacts();
                Contacts updatedOnPC = new Contacts();
                Contacts updatedOnPPC = new Contacts();
                Contacts updatedOnBoth = new Contacts();
                #endregion
                #region building contact tables
                for (int i = 0; i<pvt_oLocalList.getCount(); i++)
                {
                    Contact ct = pvt_oLocalList.get(i);
                    #region customID is unknown
                    if (helper.isCustomIDKnown(ct.customID) == false)
                    {
                        #region the contact is not inside pvt_oRemoteList: new contact detected
                        if (isInsideList(ct, pvt_oRemoteList) == false)
                        {
                            // the contact is a new one
                            newOnPC.add(ct);

                        }
                        #endregion
                        #region the contact is inside pvt_oRemoteList: integrity problem detected
                        else
                        {
                            // the contact is present on both lists but the customID is not present in hash list. Add it to hash table.
                            helper.Add(new ComparisonItem(ct.customID, CoreSync.Utils.GetMD5(ct)));
                        }
                        #endregion
                    }
                    #endregion
                    #region customID is known
                    else
                    {
                        #region the contact is not inside pvt_oRemoteList: deleted contact detected
                        if (isInsideList(ct, pvt_oRemoteList) == false)
                        {
                            // the contact must be deleted
                            deletedOnPPC.add(ct);
                        }
                        #endregion
                        #region the contact is inside pvt_oRemoteList: updated contact detected
                        else
                        {
                            // the contact is present on both lists and the customID is present in hash list.
                            // the contact must be checked against update
                            Contact lct = getFromList(ct.customID, pvt_oRemoteList);
                            if (lct != null)
                            {
                                ComparisonResult cresult = helper.CompareContacts(lct, ct);
                                switch (cresult)
                                {
                                    case ComparisonResult.PC_CHANGED:
                                        updatedOnPC.add(ct);
                                        break;
                                    case ComparisonResult.PPC_CHANGED:
                                        updatedOnPPC.add(lct);
                                        break;
                                    case ComparisonResult.BOTH_CHANGED:
                                        updatedOnBoth.add(lct);
                                        break;
                                    case ComparisonResult.NO_CHANGE:
                                        break;
                                    case ComparisonResult.UNKNOWN:
                                        break;
                                }
                            }
                        }
                        #endregion
                    }
                    #endregion
                }
                for (int i = 0; i<pvt_oRemoteList.getCount(); i++)
                {
                    Contact ct = pvt_oRemoteList.get(i);
                    #region customID is unknown
                    if (helper.isCustomIDKnown(ct.customID) == false)
                    {
                        #region the contact is not inside pvt_oLocalList: new contact detected
                        if (isInsideList(ct, pvt_oLocalList) == false)
                        {
                            // the contact is a new one
                            newOnPPC.add(ct);
                        }
                        #endregion
                        #region the contact is inside pvt_oLocalList: integrity problem detected
                        else
                        {
                            // the contact is present on both contact lists but its customID is not present in hashes list.
                            // add the contact to the hash list
                            helper.Add(new ComparisonItem(ct.customID, CoreSync.Utils.GetMD5(ct)));
                        }
                        #endregion
                    }
                    #endregion
                    #region customID is known
                    else
                    {
                        #region the contact is not inside pvt_oLocalList: deleted contact detected
                        if (isInsideList(ct, pvt_oLocalList) == false)
                        {
                            // the contact must be deleted
                            deletedOnPC.add(ct);
                        }
                        #endregion
                        #region the contact is inside pvt_oLocalList: updated contact detected
                        else
                        {
                            // the contact is present on both lists and the customID is present in hash list.
                            // the contact must be checked against update
                            Contact lct = getFromList(ct.customID, pvt_oLocalList);
                            if (lct != null)
                            {
                                ComparisonResult cresult = helper.CompareContacts(ct, lct);
                                switch (cresult)
                                {
                                    case ComparisonResult.PC_CHANGED:
                                        updatedOnPC.add(lct);
                                        break;
                                    case ComparisonResult.PPC_CHANGED:
                                        updatedOnPPC.add(ct);
                                        break;
                                    case ComparisonResult.BOTH_CHANGED:
                                        updatedOnBoth.add(ct);
                                        break;
                                    case ComparisonResult.NO_CHANGE:
                                        break;
                                    case ComparisonResult.UNKNOWN:
                                        break;
                                }
                            }
                        }
                        #endregion
                    }
                    #endregion
                }
                #endregion
                #region building final contact table and storing data
                ////////////////////////////////////////////////////
                /*
                 * now that we have a list of added, removed and
                 *  updated contacts, all we have to do is to save
                 *  this data.
                 * How to perform this?
                 * Build a new Contact list and put, in order:
                 * 1. Contacts to add to PocketPC
                 * 2. Contacts to add to LocalDB
                 * 3. Contacts to remove from PocketPC
                 * 4. Contacts to remove from LocalDB
                 * 5. Contacts to update on PocketPC
                 * 6. Contacts to update on LocalDB
                 * 7. Contacts to update following user directions
                 * 8. All contacts that did not change (all
                 *    contacts excluded the already included ones)
                 * 
                 * The list so generated is the final contact list,
                 *  that must be stored on both the PocketPC and
                 *  the LocalDB. Storing data, however, is not
                 *  the purpose of this method, so the only action
                 *  performed is setting the pvt_oMerged instance
                 *  variable.
                 */
                Contacts result = new Contacts();
                Contacts curList;
                String curAction = "";
                #region 1. Contacts to add to PocketPC
                curList = removeDuplicates(newOnPC);
                curAction = Actions.ADDPPC;
                for (int i = 0; i < curList.getCount(); i++)
                {
                    Contact ct = curList.get(i);
                    ct.action = curAction;
                    result.add(ct);
                }
                #endregion
                #region 2. Contacts to add to LocalDB
                curList = removeDuplicates(newOnPPC);
                curAction = Actions.ADD;
                for (int i = 0; i < curList.getCount(); i++)
                {
                    Contact ct = curList.get(i);
                    ct.action = curAction;
                    result.add(ct);
                }
                #endregion
                #region 3. Contacts to remove from PocketPC
                curList = removeDuplicates(deletedOnPC);
                curAction = Actions.DELETEPPC;
                for (int i = 0; i < curList.getCount(); i++)
                {
                    Contact ct = curList.get(i);
                    ct.action = curAction;
                    result.add(ct);
                }
                #endregion
                #region 4. Contacts to remove from LocalDB
                curList = removeDuplicates(deletedOnPPC);
                curAction = Actions.DELETE;
                for (int i = 0; i < curList.getCount(); i++)
                {
                    Contact ct = curList.get(i);
                    ct.action = curAction;
                    result.add(ct);
                }
                #endregion
                #region 5. Contacts to update following user directions
                curList = removeDuplicates(updatedOnBoth);
                curAction = Actions.NOTHING;
                for (int i = 0; i < curList.getCount(); i++)
                {
                    Contact ct = curList.get(i);
                    Contact ppc = getFromList(ct.customID, pvt_oRemoteList);
                    Contact pc = getFromList(ct.customID, pvt_oLocalList);
                    ct.action = frmConflictSolver.showMe(pc, ppc);
                    result.add(ct);
                }
                #endregion
                #region 6. Contacts to update on PocketPC
                curList = removeDuplicates(updatedOnPC);
                curAction = Actions.UPDATEPPC;
                for (int i = 0; i < curList.getCount(); i++)
                {
                    Contact ct = curList.get(i);
                    ct.action = curAction;
                    if(isInsideList(ct, result)==false)
                        result.add(ct);
                }
                #endregion
                #region 7. Contacts to update on LocalDB
                curList = removeDuplicates(updatedOnPPC);
                curAction = Actions.UPDATE;
                for (int i = 0; i < curList.getCount(); i++)
                {
                    Contact ct = curList.get(i);
                    ct.action = curAction;
                    if (isInsideList(ct, result) == false)
                        result.add(ct);
                }
                #endregion
                #region 8. All contacts that did not change
                for (int i = 0; i < pvt_oLocalList.getCount(); i++)
                {
                    Contact ct = pvt_oLocalList.get(i);
                    if (isInsideList(ct, result) == false)
                        result.add(ct);
                }
                for (int i = 0; i < pvt_oRemoteList.getCount(); i++)
                {
                    Contact ct = pvt_oRemoteList.get(i);
                    if (isInsideList(ct, result) == false)
                        result.add(ct);
                }
                #endregion

                pvt_oMerged = result;
                #endregion
            }
            #endregion
        }
        #endregion

        #region locally useful functions

        private Contacts copyList(Contacts source)
        {
            Contacts list = new Contacts();
            for (int i = 0; i < source.getCount(); i++)
            {
                list.add(source.get(i));
            }
            return list;
        }

        private bool isInsideList(Contact oCt, Contacts oList)
        {
            for (int i = 0; i < oList.getCount(); i++)
            {
                if (oList.get(i).customID == oCt.customID)
                    return true;
            }
            return false;
        }

        private Contact getFromList(String customID, Contacts oList)
        {
            for (int i = 0; i < oList.getCount(); i++)
            {
                if (oList.get(i).customID == customID)
                    return oList.get(i);
            }
            return null;
        }

        private void appendToList(ref Contacts oDestination, Contacts oSource)
        {
            for (int i = 0; i < oSource.getCount(); i++)
                oDestination.add(oSource.get(i));
        }

        private Contacts removeDuplicates(Contacts oList)
        {
            Contacts retval = new Contacts();
            for (int i = 0; i < oList.getCount(); i++)
            {
                Contact oCt=oList.get(i);
                if (isInsideList(oCt, retval) == false)
                    retval.add(oCt);
            }
            return retval;
        }

        #endregion
    }
}
