# SPDX-License-Identifier: MIT
# Copyright (c) 2022, Mathias Laurin

from __future__ import annotations

import enum
import os
import sys
from collections import abc
from typing import Optional, Sequence, Tuple, Union, overload

from mbedtls._tlsi import (
    Certificate,
    DTLSConfiguration,
    DTLSVersion,
    TLSConfiguration,
    TLSVersion,
)

if sys.version_info < (3, 8):
    from typing_extensions import Literal
else:
    from typing import Literal

_VERSION = Tuple[int, int]

def ciphers_available() -> Sequence[str]: ...
def _tls_to_version(maj_min_version: _VERSION) -> TLSVersion: ...
def _dtls_to_version(maj_min_version: _VERSION) -> DTLSVersion: ...
def _tls_from_version(version: TLSVersion) -> Tuple[int, int]: ...
def _dtls_from_version(version: DTLSVersion) -> Tuple[int, int]: ...

_SUPPORTED_TLS_VERSION: Sequence[TLSVersion]
_SUPPORTED_DTLS_VERSION: Sequence[DTLSVersion]

class HandshakeStep(enum.Enum):
    HELLO_REQUEST: int
    CLIENT_HELLO: int
    SERVER_HELLO: int
    SERVER_CERTIFICATE: int
    SERVER_KEY_EXCHANGE: int
    CERTIFICATE_REQUEST: int
    SERVER_HELLO_DONE: int
    CLIENT_CERTIFICATE: int
    CLIENT_KEY_EXCHANGE: int
    CERTIFICATE_VERIFY: int
    CLIENT_CHANGE_CIPHER_SPEC: int
    CLIENT_FINISHED: int
    SERVER_CHANGE_CIPHER_SPEC: int
    SERVER_FINISHED: int
    FLUSH_BUFFERS: int
    HANDSHAKE_WRAPUP: int
    HANDSHAKE_OVER: int
    SERVER_NEW_SESSION_TICKET: int
    SERVER_HELLO_VERIFY_REQUEST_SENT: int


class TLSError(Exception): ...
class WantWriteError(TLSError): ...
class WantReadError(TLSError): ...
class RaggedEOF(TLSError): ...
class HelloVerifyRequest(TLSError): ...

class TrustStore(abc.Sequence[Certificate]):
    def __init__(
        self, db: Optional[Union[Sequence[Certificate], TrustStore]] = ...
    ) -> None: ...
    @classmethod
    def system(cls) -> TrustStore: ...
    @classmethod
    def from_pem_file(
        cls, path: Union[str, os.PathLike[str]]
    ) -> TrustStore: ...
    def __eq__(self, other: object) -> bool: ...
    def __bool__(self) -> bool: ...
    def __len__(self) -> int: ...
    @overload
    def __getitem__(self, index: int) -> Certificate: ...
    @overload
    def __getitem__(self, s: slice) -> TrustStore: ...
    def add(self, crt: Certificate) -> None: ...

class Purpose(enum.IntEnum):
    SERVER_AUTH: int
    CLIENT_AUTH: int

class _BaseContext:
    def __init__(
        self, configuration: Union[TLSConfiguration, DTLSConfiguration]
    ) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    @property
    def configuration(self) -> Union[TLSConfiguration, DTLSConfiguration]: ...
    # Don't use `_conf`, for testing purpose only.
    @property
    def _conf(self) -> Union[TLSConfiguration, DTLSConfiguration]: ...
    @property
    def _purpose(self) -> Purpose: ...

class MbedTLSBuffer:
    def __init__(
        self, context: _BaseContext, server_hostname: Optional[str] = None
    ) -> None: ...
    @property
    def _input_buffer(self) -> bytes: ...
    @property
    def _output_buffer(self) -> bytes: ...
    @property
    def context(self) -> _BaseContext: ...
    @property
    def _server_hostname(self) -> str: ...
    def shutdown(self) -> None: ...
    def setcookieparam(self, info: bytes) -> None: ...
    def setmtu(self, mtu: int) -> None: ...
    def read(self, amt: int) -> bytes: ...
    def readinto(self, buffer: bytes, amt: int) -> int: ...
    def write(self, buffer: bytes) -> int: ...
    def receive_from_network(self, data: bytes) -> None: ...
    def peek_outgoing(self, amt: int) -> bytes: ...
    def consume_outgoing(self, amt: int) -> None: ...
    @overload
    def getpeercert(self, binary_form: Literal[False]) -> str: ...
    @overload
    def getpeercert(self, binary_form: Literal[True]) -> bytes: ...
    def selected_npn_protocol(self) -> None: ...
    def negotiated_protocol(self) -> str: ...
    def cipher(self) -> bytes: ...
    @property
    def _handshake_state(self) -> HandshakeStep: ...
    def do_handshake(self) -> None: ...
    def negotiated_tls_version(self) -> Union[TLSVersion, DTLSVersion]: ...

class TLSSession: ...
