/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <setjmp.h>
#include <math.h>

#include "cmp.h"
#include "kalc.h"
#include "realfunctions.h"


void f_sinh(void)
{
  /*
   * This function calls the _f_sinh function through the wrapper.
   */

  run1_1_Function(_f_sinh, "sinh");
}


Object _f_sinh(Object n, int *err)
{
  /*
   * This function returns the hyperbolic sine of its argument.
   *
   * Definition:
   *    sinh(x) = (exp(x) - exp(-x)) / 2 
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_sinh(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_sinh(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_cosh(void)
{
  /*
   * This function calls the _f_cosh function through the wrapper.
   */

  run1_1_Function(_f_cosh, "cosh");
}


Object _f_cosh(Object n, int *err)
{
  /*
   * This function returns the hyperbolic co-sine of its argument.
   *
   * Definition:
   *    cosh(x) = (exp(x) + exp(-x)) / 2
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_cosh(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_cosh(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_tanh(void)
{
  /*
   * This function calls the _f_tanh function through the wrapper.
   */

  run1_1_Function(_f_tanh, "tanh");
}


Object _f_tanh(Object n, int *err)
{
  /*
   * This function returns the hyperbolic tangent of its argument.
   *
   * Definition:
   *    tanh(x) = sinh(x) / cosh(x)
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_tanh(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_tanh(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_sech(void)
{
  /*
   * This function calls the _f_sech function through the wrapper.
   */

  run1_1_Function(_f_sech, "sech");
}


Object _f_sech(Object n, int *err)
{
  /*
   * This function returns the hyperbolic secant of its argument.
   *
   * Definition:
   *    sech(x) = 1/cosh(x) 
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_sech(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_sech(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_csch(void)
{
  /*
   * This function calls the _f_csch function through the wrapper.
   */

  run1_1_Function(_f_csch, "csch");
}


Object _f_csch(Object n, int *err)
{
  /*
   * This function returns the hyperbolic co-secant of its argument.
   *
   * Definition:
   *    csch(x) = 1/sinh(x) 
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_csch(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_csch(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_coth(void)
{
  /*
   * This function calls the _f_coth function through the wrapper.
   */

  run1_1_Function(_f_coth, "coth");
}


Object _f_coth(Object n, int *err)
{
  /*
   * This function returns the hyperbolic co-tangent of its argument.
   *
   * Definition:
   *    coth(x) = cosh(x)/sinh(x) 
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_coth(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_coth(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_asinh(void)
{
  /*
   * This function calls the _f_asinh function through the wrapper.
   */

  run1_1_Function(_f_asinh, "asinh");
}


Object _f_asinh(Object n, int *err)
{
  /*
   * This function returns the hyperbolic arc sine of its argument.
   *
   * Definition:
   *    asinh(x) = ln(z + sqrt(z^2 + 1))
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_asinh(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_asinh(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_acosh(void)
{
  /*
   * This function calls the _f_acosh function through the wrapper.
   */

  run1_1_Function(_f_acosh, "acosh");
}


Object _f_acosh(Object n, int *err)
{
  /*
   * This function returns the hyperbolic arc co-sine of its argument.
   *
   * Definition:
   *    acosh(x) = ln(z + sqrt(z^2 - 1))
   */

  switch (type(n)) {
  case TYPE_REAL:
    /* If the number is less than 1, the result is a complex number. */
    if (n.value.real >= 1) {
      *err = ERR_NOERR;
      n.value.real = re_cosh(n.value.real);
      break;
    }
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_acosh(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_atanh(void)
{
  /*
   * This function calls the _f_atanh function through the wrapper.
   */
  
  run1_1_Function(_f_atanh, "atanh");
}


Object _f_atanh(Object n, int *err)
{
  /*
   * This function returns the hyperbolic arc tangent of its argument.
   *
   * Definition:
   *               1    1 + z
   *    atanh(z) = - ln -----
   *               2    1 - z
   */
   

  switch (type(n)) {
  case TYPE_REAL:
  /* If the number is outside the range -1 <= n <= 1 (|n| <= 1), the
    result is a complex number. */
    if (fabs(n.value.real) <= 1) {
      *err = ERR_NOERR;
      n.value.real = re_atanh(n.value.real);
      break;
    }
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_atanh(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_asech(void)
{
  /*
   * This function calls the _f_asech function through the wrapper.
   */

  run1_1_Function(_f_asech, "asech");
}


Object _f_asech(Object n, int *err)
{
  /*
   * This function returns the hyperbolic arc sine of its argument.
   *
   * Definition:
   *    asech(x) = acosh(1/x)
   */

  switch (type(n)) {
  case TYPE_REAL:
    /* If the result is outside the range 0 <= x <= 1, the result is a
       complex number. */
    if (0 <= n.value.real && n.value.real <= 1) {
      *err = ERR_NOERR;
      n.value.real = re_asech(n.value.real);
      break;
    }
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_asech(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_acsch(void)
{
  /*
   * This function calls the _f_acsch function through the wrapper.
   */

  run1_1_Function(_f_acsch, "acsch");
}


Object _f_acsch(Object n, int *err)
{
  /*
   * This function returns the hyperbolic arc co-sine of its argument.
   *
   * Definition:
   *    acsch(x) = asinh(1/x)
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_acsch(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_acsch(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_acoth(void)
{
  /*
   * This function calls the _f_acoth function through the wrapper.
   */
  
  run1_1_Function(_f_acoth, "acoth");
}


Object _f_acoth(Object n, int *err)
{
  /*
   * This function returns the hyperbolic arc tangent of its argument.
   *
   * Definition:
   *    acoth(x) = atanh(1/x)
   */
   

  switch (type(n)) {
  case TYPE_REAL:
  /* If the number is in the range -1 < n < 1 (|n| <= 1), the
    result is a complex number. */
    if (fabs(n.value.real) >= 1) {
      *err = ERR_NOERR;
      n.value.real = re_acoth(n.value.real);
      break;
    }
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_acoth(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_gd(void) 
{
  /*
   * This function calls the _f_gd function through the wrapper.
   */

  run1_1_Function(_f_gd, "gd");
}


Object _f_gd(Object n, int *err)
{
  /*
   * This function returns the Gudermannian function of its argument.
   *
   * Definition:
   *    gd(x) = 2atan(e^x) - pi/2
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_gd(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_gd(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_invgd(void) 
{
  /*
   * This function calls the _f_invgd function through the wrapper.
   */

  run1_1_Function(_f_invgd, "invgd");
}


Object _f_invgd(Object n, int *err)
{
  /*
   * This function returns the inverse Gudermannian function of its argument.
   *
   * Definition:
   *    invgd(x) = ln(sec(x) + tan(x))
   */

  if (type(n) == TYPE_REAL || type(n) == TYPE_CMP) {
    /* Since it is difficult to tell whether the result of this function
       is a real number at for a value, the calculations are always done
       with complex numbers, and at the end if the imaginary part is zero
       the result is converted back into a real number. */
    __f_reTOc(&n);
    n.value.cmp = cmp_invgd(n.value.cmp);
    if (n.value.cmp.im == 0) /* { */
      n.type = TYPE_REAL;
      /* n.value.real = n.value.cmp.re; */
    /* } */
    *err = ERR_NOERR;
  } else
    *err = ERR_BADARGUMENTTYPE;
  
  return n;
}
