/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>
#ifdef HAVE_LIBREADLINE
#  include <readline/tilde.h>
#endif

#include "cmp.h"
#include "kalc.h"
#include "mem.h"


static hashBucket **memoryStack[MEMLEVELS];
int currMemLevel = -1;


void doUnquoted(char *id)
{
  /*
   * This function is the action run when a name with no delimiters is
   * to be evaluated. First it is checked against function names. If it
   * matches, the function is run. If not, then the object it refers is
   * fetched and evaluated, if any. If it represents no object, then
   * it is put as an identifier in the stack.
   */

  void (*f)(void);
  
  f = getFuncAddress(id);
  if (f)
    (*f)();
  else {
    Object *objp = recallObject(id);

    if (!objp || (nArrows != -1 && lastArrowLevel[nArrows] == progNesting))
      insertId(id);
    else
      evaluateObj(*objp, 1);
  }
}


void doSto(void)
{
  /*
   * This is the function called by the user command sto.
   */

  if (enoughArgs(2)) {
    if (type(**tos) == TYPE_ID) {
      Object *oldContents;
      /* The identifier will be dropped, so we need to duplicate the name. */
      char *name = strdup((**tos).value.str);
      if (!name)
	doError("Fatal", ERR_NOTENOUGHMEMORY);

      oldContents = storeObject(name, *(tos - 1));

      if (oldContents) {
	freeLastArgs();
	lastArgN = 2;
	lastArgs[0] = *tos;
	lastArgs[1] = oldContents;
	tos--;
	tos--;
      } else {
	saveLastArgs(1);
	/* Drop the identifier */
	_f_drop();
	/* And silently remove the object from the stack */
	tos--;
      }
    } else
      doError("sto", ERR_BADARGUMENTTYPE);
  } else
    doError("sto", ERR_TOOFEWARGUMENTS);
}


void diskSto(void)
{
  /*
   * This is the function called by the user command disksto.
   */
  
  if (enoughArgs(2)) {
    if (type(**tos) == TYPE_ID) {
#ifdef HAVE_LIBREADLINE
      char *fileName = tilde_expand((**tos).value.str);
#else
      char *fileName = (**tos).value.str;
#endif
      diskStoObject(fileName, *(tos - 1));

      /* Drop the identifier and the object */
      _f_drop();
      _f_drop();
#ifdef HAVE_LIBREADLINE
      free(fileName);
#endif
    } else
      doError("disksto", ERR_BADARGUMENTTYPE);
  } else
    doError("disksto", ERR_TOOFEWARGUMENTS);
}


void doRcl(void)
{
  /*
   * This is the function called by the user command rcl.
   */

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_ID) {
      Object *objp = recallObject((**tos).value.str);
      if (objp) {
	_f_drop();
	insertObject(*objp);
      } else
	doError("rcl", ERR_UNDEFINEDNAME);
    } else
      doError("rcl", ERR_BADARGUMENTTYPE);
  } else
    doError("rcl", ERR_TOOFEWARGUMENTS);
}


void diskRcl(void)
{
  /*
   * This is the function called by the user command diskrcl.
   */

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_ID) {
#ifdef HAVE_LIBREADLINE
      char *fileName = tilde_expand((**tos).value.str);
#else
      char *fileName = (**tos).value.str;
#endif
      Object *objp = diskRclObject(fileName);
#ifdef HAVE_LIBREADLINE
      free(fileName);
#endif
      if (objp) {
	_f_drop();
	insertObject(*objp);
      } else
	doError("diskrcl", ERR_COULDNOTOPENFILE);
    } else
      doError("diskrcl", ERR_BADARGUMENTTYPE);
  } else
    doError("diskrcl", ERR_TOOFEWARGUMENTS);
}


void doPurge(void)
{
  /*
   * This is the function called by the user command purge.
   */

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_ID) {
      Object *oldContents = purgeObject((**tos).value.str);
      if (oldContents) {
	freeLastArgs();
	lastArgN = 2;
	lastArgs[0] = *tos;
	lastArgs[1] = oldContents;
	tos--;
      } else {
	saveLastArgs(1);
	_f_drop();
      }
    } else
      doError("purge", ERR_BADARGUMENTTYPE);
  } else
    doError("purge", ERR_TOOFEWARGUMENTS);
}


void doVars(void)
{
  /*
   * This is the function run by the user command vars. It displays the
   * names of all variables stored in memory.
   *
   * The HP48 command returns a list with the variables. But since there is
   * not yet support for list, the names are printed, and the stack is
   * not shown, so that the results of the command stand out.
   *
   * If there are no variables, the message "Memory empty" is displayed
   * instead.
   */

  int printed = 0, currCol = 0;
  hashBucket *h;
  register int i;

  linesToDisplay = 0;

  for (i = 0; i < HASHBUCKETS; i++) {
    h = memoryStack[0][i]->next;
    while (h != NULL) {
      printed = 1;
      printf("'%s' ", h->name);
      currCol += strlen(h->name) + 3;
      if (currCol >= userOptions.width - 7) {
	putchar('\n');
	currCol = 0;
      }
      h = h->next;
    }
  }

  if (!printed)
    printf("Memory empty");

  putchar('\n');
}


void doClvar(void)
{
  /*
   * This is the function run by the user function clvar.
   * It erases all memory. Since this is an irreversible operation, the
   * user is asked confirmation before doing this.
   */
  
  char answer[10];
  
  printf("Warning: This will erase ALL your memory. Are you sure (y/n)? ");
  fgets(answer, 9, stdin);

  if (toupper(*answer) == 'Y')
    _doClvar();
}


void _doClvar(void)
{
  /*
   * This function erases all global memory. It does not ask
   * confirmation before.
   */

  clearMemoryLevel(0);
}


void clearMemoryLevel(int level) 
{
  /*
   * This function clears the specified level of the memory stack.
   */

  register int i;
  hashBucket *h, *toPurge;

  for (i = 0; i < HASHBUCKETS; i++) {
    h = memoryStack[level][i]->next;
    while (h != NULL) {
      toPurge = h;
      h = h->next;

      free(toPurge->name);
      freeObjectSpecials(*toPurge->obj);
      free(toPurge->obj);
      free(toPurge);
    }
    memoryStack[level][i]->next = NULL;
  }
}


void doPwd(void)
{
  /*
   * This is the function run by the user command pwd.
   * It displays the current working directory. The stack is not shown,
   * so that the information stands out.
   */

  char cwd[PATH_MAX];

  getcwd(cwd, PATH_MAX);
  printf("%s\n", cwd);
  linesToDisplay = 0;
}


void doCd(void)
{
  run1_0_Function(_f_cd, "cd");
}


int _f_cd(Object n)
{
  if (type(n) == TYPE_ID)
    return (chdir(n.value.str) ? ERR_COULDNOTCHDIR : ERR_NOERR);

  return ERR_BADARGUMENTTYPE;
}


void saveMemory(FILE *fp)
{
  /*
   * This file saves the memory.
   *
   * Each object in memory has the following format:
   * - The name is stored via the saveString() function.
   * - The actual object is stored via saveObject().
   *
   * The above items repeat for each object. The end of the objects is
   * marked by a length field whose value is -1.
   */

  hashBucket *h;
  register int i;
  int minusOne = -1;

  for (i = 0; i < HASHBUCKETS; i++) {
    h = memoryStack[0][i]->next;
    while (h != NULL) {
      saveString(h->name, fp);
      saveObject(h->obj, fp);
      h = h->next;
    }
  }

  /* Add end of memory marker */
  fwrite(&minusOne, sizeof(minusOne), 1, fp);
}


void loadMemory(FILE *fp)
{
  /*
   * This functions load the memory saved in a file, stored in the format
   * specified in the saveMemory function.
   */

  int length;
  char *name;
  Object *obj;

  while (1) {
    fread(&length, sizeof(length), 1, fp);
    if (length == -1)
      break;

    name = (char *) calloc(length + 1, 1);
    if (!name)
      doError("Fatal", ERR_NOTENOUGHMEMORY);
    
    fread(name, 1, length, fp);

    obj = (Object *) malloc(sizeof(Object));
    if (!obj)
      doError("Fatal", ERR_NOTENOUGHMEMORY);
    
    *obj = loadObject(fp);

    storeObject(name, obj);
  }
}


void initNextMemoryLevel(void)
{
  /*
   * This function initializes a new level of the memory stack.
   */

  register int i;

  if (++currMemLevel == MEMLEVELS)
    doError("Fatal", ERR_MAXLOCALENVS);

  if (!memoryStack[currMemLevel]) {
    memoryStack[currMemLevel] = (hashBucket **) calloc(HASHBUCKETS,
						       sizeof(hashBucket *));
    if (!memoryStack[currMemLevel])
      doError("Fatal", ERR_NOTENOUGHMEMORY);
  }
  
  for (i = 0; i < HASHBUCKETS; i++)
    memoryStack[currMemLevel][i] = (hashBucket *) calloc(1,
							 sizeof(hashBucket));
}


Object *storeObject(char *name, Object *objp)
{
  /*
   * This function stores the object pointed by objp in the memory,
   * with the specified name. If there was already something stored
   * with that name, that is returned.
   */

  unsigned long hashCode = getHashCode(name);
  hashBucket *newBucket;
  Object *toReturn = NULL;

  /* First, see if there is already an object stored with that name */
  newBucket = getMemoryContents(name, hashCode);
  if (newBucket) {
    /* freeObjectSpecials(*newBucket->obj); */
    /* free(newBucket->obj); */
    toReturn = newBucket->obj;
    newBucket->obj = objp;
  } else {
    hashBucket *last;
    newBucket = (hashBucket *) malloc(sizeof(hashBucket));
    last = getLastPointer(hashCode, 0);

    newBucket->name = name;
    newBucket->obj = objp;
    newBucket->next = NULL;

    last->next = newBucket;
  }

  return toReturn;
}


void bindObject(char *name, Object *objp)
{
  /*
   * Binds the specified object in the topmost temporary environment.
   */

  unsigned long hashCode = getHashCode(name);
  hashBucket *new, *last;

  new = (hashBucket *) malloc(sizeof(hashBucket));
  last = getLastPointer(hashCode, currMemLevel);

  new->name = name;
  new->obj = objp;
  new->next = NULL;

  last->next = new;
}


void diskStoObject(char *name, Object *objp)
{
  /*
   * This function stores an object into a disk file, in the current
   * directory, under the given name.
   *
   * The files are saved in the following format:
   * - An 8-character header MAGICOBJVER (macro defined elsewhere)
   * - The actual object, stored in whichever way is convenient (see
   *   the function saveObject.
   */

  FILE *fp;

  if ((fp = fopen(name, "wb")) == NULL)
    doError("disksto", ERR_COULDNOTOPENFILE);

  fwrite(MAGICOBJVER, 1, 8, fp);
  saveObject(objp, fp);

  fclose(fp);
}


Object *recallObject(char *name)
{
  /*
   * This function tries to recall the object with the given name.
   * If found, returns a pointer to the object, otherwise returns NULL.
   */

  unsigned long hashCode = getHashCode(name);
  hashBucket *h = getMemoryContents(name, hashCode);

  return (h ? h->obj : NULL);
}


Object *diskRclObject(char *name)
{
  /*
   * This function tries to recall from the file whose name is given an
   * object. If this was possible, returns a pointer to that object,
   * otherwise returns NULL.
   */

  FILE *fp;
  char version[8];
  Object *objp = NULL;

  if ((fp = fopen(name, "rb")) == NULL)
    return NULL;

  fread(&version, 1, 8, fp);
  if (strcmp(version, MAGICOBJVER) == 0
      || strcmp(version, MAGICOBJVER210) == 0) {
    objp = (Object *) malloc(sizeof(Object));
    *objp = loadObject(fp);
  } else if (strcmp(version, MAGICOBJVER200) == 0) {
    objp = (Object *) malloc(sizeof(Object));
    *objp = loadObject200(fp);
  } else {
    fclose(fp);
    doError("diskrcl", ERR_INVALIDVERSION);
  }

  fclose(fp);

  return objp;
} 


Object *purgeObject(char *name)
{
  /*
   * This function tries to purge an object with the given name. If
   * there is no object with such name, this function does nothing and
   * does not complain.
   *
   * If there was an object with that name, its previous contents are
   * returned.
   */

  unsigned long hashCode = getHashCode(name);
  hashBucket *h = getPreviousPointer(name, hashCode);
  Object *old;

  if (h) {
    hashBucket *toPurge = h->next;
    free(toPurge->name);
    /* freeObjectSpecials(*toPurge->obj); */
    /* free(toPurge->obj); */
    h->next = toPurge->next;
    old = toPurge->obj;
    free(toPurge);
    return old;
  }

  return NULL;
}


hashBucket *getLastPointer(unsigned long hashCode, int level)
{
  /*
   * This function searches the memory contents for the last object
   * of the linked list in the bucket hashCode, because the new objects
   * are always added to the end of the linked list. A pointer to that
   * object is returned.
   *
   * Only works in specified level. If binding, should work only in
   * the topmost level. If sto()ring, then should only store new objects
   * in global memory (level 0), it is not allowed to add new LAMs to an
   * existing environment.
   */

  hashBucket *p = memoryStack[level][hashCode];

  while (p->next != NULL)
    p = p->next;

  return p;
}


hashBucket *getMemoryContents(char *name, unsigned long hashCode)
{
  /*
   * This function tries to find an object stored in the memory with the
   * given name, at the bucket hashCode. If it is found, returns a pointer
   * to that memory slot (a hashBucket structure), otherwise returns NULL.
   */

  register int i;
  
  for (i = currMemLevel; i >= 0; --i) {
    hashBucket *h = memoryStack[i][hashCode]->next;

    while (1) {
      if (h == NULL)
	break;
      if (strcmp(h->name, name) == 0)
	return h;
      h = h->next;
    }
  }

  return NULL;
}


hashBucket *getPreviousPointer(char *name, unsigned long hashCode)
{
  /*
   * This function returns a pointer to the linked list node which is
   * immediately before the one with the given name. If there is no object
   * with the given name, returns NULL.
   *
   * Only works for the global level because it is only called by the
   * 'purge' function, which does not work with LAMs.
   */

  hashBucket *h = memoryStack[0][hashCode];

  while(1) {
    if (h->next == NULL)
      return NULL;
    if (strcmp(h->next->name, name) == 0)
      return h;
    h = h->next;
  }
}


unsigned long getHashCode(char *name)
{
  /*
   * This function returns an integer associated with the string name.
   *
   * The function was taken from the book "Practical Algorithms in C++",
   * by Bryan Flamig.
   */

  register unsigned long h = 0;
  register unsigned long g;

  while (*name) {
    h = (h << 4) + *name++;
    g = h & 0xF0000000L;
    if (g)
      h ^= g >> 24;
    h &= ~g;
  }
  
  return h % HASHBUCKETS;
}
