
/* Copyright (c) Miguel Angel Sepulveda, 1998. */

/* This program is freely distributable without licensing fees 
   and is provided without guarantee or warrantee expressed or 
   implied. This program is -not- in the public domain. */

#include "Gd_definitions.h" 
#include "Gd_noise.h" 
#include "Gd_function.h" 
#include "Gd_monomer.h" 
#include "Gd_polymer.h" 
#include "Gd_polymer_driver.h" 
#include "Gd_properties.h" 
 
unsigned long GdPolymerDriver::seed = 97296; 
 
 
GdPolymerDriver::GdPolymerDriver(GdPolymer &c){ 
  // Store reference to polymer 
  polymer = &c; 
  polymerSize = c.getSize(); 
   
   
  // Integration time step 
  timeStep = 1.0; 
   
  // Potential cutoff distance 
  rCutoff = 1.0; 
  rLimit = 1.0; 
   
  // Physical Properties of the Solution 
  temperature = 1.0; 
  friction = 1.0; 
   
  // Callback functions 
  is_force = false; 
  is_force_a = false; 
  is_force_r = false; 
  repulsion = 0; 
  attraction = 0; 
   
  // Update driver parameters 
  updateParameters(); 
}; 
 
 
GdPolymerDriver::~GdPolymerDriver(){ 
  delete repulsion; 
  delete attraction; 
}; 
 
 
 
void GdPolymerDriver::initRepelFunc(GDfunc1D f, int resolution, double rmin, double rmax){ 
  // Create table 
  is_force_r = true; 
  is_force = is_force_a & is_force_r; 
  repulsion = new GdTableFunc1D(f, resolution, rmin, rmax); 
}; 
 
void GdPolymerDriver::initAttrcFunc(GDfunc1D f, int resolution, double rmin, double rmax){ 
  // Create table 
  is_force_a = true; 
  is_force = is_force_a && is_force_r; 
  attraction = new GdTableFunc1D(f, resolution, rmin, rmax); 
}; 
 
void GdPolymerDriver::setCutoff(double rc, double rl){ 
  rCutoff = rc * rc; 
  rLimit = rl * rl; 
}; 
 
void GdPolymerDriver::setTemperature(double T){ 
  temperature = T; 
  updateParameters(); 
}; 
 
void GdPolymerDriver::setFriction(double G){ 
  friction = G; 
  updateParameters(); 
}; 
 
void GdPolymerDriver::setTimeStep(double dt){ 
  timeStep = dt; 
  updateParameters(); 
}; 
 
void GdPolymerDriver::updateParameters(){ 
  static const double root3 = sqrt(3.0);  
  static const double c46 = 4.0/6.0; 
   
  // Noise distribution parameters  
  double ft = friction * timeStep;  
  double e1 = exp(-ft);  
  sigma_v = 2.0 * temperature * ft;  
  sigma_r = temperature * c46 * pow(ft, 5.0);  
  sigma_v = sqrt(sigma_v);  
  sigma_r = sqrt(sigma_r);  
   
  // Integrator parameters  
  c0 = e1;  
  c1 = (1.0 - c0)/ft;  
  c2 = (1.0 - c1)/ft;  
  r1 = c1 * timeStep;  
  r2 = c2 * timeStep * timeStep;  
  v1 = c0;  
  v2 = (c1 -c2)*timeStep;  
  v3 = c2 * timeStep;  
   
  // Uniform distribution parameters  
  n1Half = root3 * sigma_r;  
  n2Half = root3 * sigma_v;  
  n1 = 2.0 * n1Half;  
  n2 = 2.0 * n2Half;  
}; 
 
 
void GdPolymerDriver::propagate(int nr_steps) { 
   
  // Loop over number of integration steps 
  //     (Modified Verlet Algorithm) 
   
  for(int count = 0; count < nr_steps; count++){ 
    int i; 
     
    // 1.   Find new positions 
    for (i = 0; i < polymerSize; i++){ 
      // Reference to monomer 
      GdMonomer &m = (*polymer)[i]; 
       
      // Retrieve positions, accelerations & velocities 
      double ax, ay, az; 
      double vx, vy, vz; 
      double x, y, z; 
      m.getPosition(x, y, z); 
      m.getVelocity(vx, vy, vz); 
      m.getAcceleration(ax, ay, az); 
       
      // Find new position 
      x += r1 * vx + r2 * ax + n1 * ran1(&seed) -n1Half; 
      y += r1 * vy + r2 * ay + n1 * ran1(&seed) -n1Half; 
      z += r1 * vz + r2 * az + n1 * ran1(&seed) -n1Half; 
       
      // Compute interm velocity 
      vx = v1 * vx + v2 * ax + n2 * ran1(&seed) -n2Half; 
      vy = v1 * vy + v2 * ay + n2 * ran1(&seed) -n2Half; 
      vz = v1 * vz + v2 * az + n2 * ran1(&seed) -n2Half; 
       
      // Store new position and interm. velocity 
      m.setPosition(x, y, z); 
      m.setVelocity(vx, vy, vz); 
    }; 
     
     
    // 2.   Compute Forces 
    if(is_force == true){	 
      polymer->resetAcceleration(); 
      updateForces(); 
    }; 
     
    // 3.   Find new velocities 
    for (i = 0; i < polymerSize; i++){ 
      // Reference to monomer 
      GdMonomer &m = (*polymer)[i]; 
       
      // Retrieve accelerations & velocities 
      double ax, ay, az; 
      double vx, vy, vz; 
      m.getVelocity(vx, vy, vz); 
      m.getAcceleration(ax, ay, az); 
       
      // Update velocities 
      vx += v3 * ax; 
      vy += v3 * ay; 
      vz += v3 * az; 
       
       
      // Store velocities 
      m.setVelocity(vx, vy, vz); 
    }; 
  }; 
  // End loop over number of time steps 
}; 
 
 
 
 
 
 
// This is a very simple-minded force update function 
void GdPolymerDriver::updateForces(){ 
  int left; 
   
  //  UPDATE REPULSIVE FORCE 
  // Loop over all pairs of distinct monomers 
  for(left = 0; left < polymerSize-1; left++) { 
     
    // Get a reference to the left monomer 
    GdMonomer &lmon = (*polymer)[left]; 
     
    // Get position for left monomer 
    double lx, ly, lz; 
    lmon.getPosition(lx, ly, lz); 
     
    for(int right = left+1; right < polymerSize; right++){ 
      // Get a reference to the right monomer 
      GdMonomer &rmon = (*polymer)[right]; 
       
      // Get position for right monomer 
      double rx, ry, rz; 
      rmon.getPosition(rx, ry, rz); 
       
      // Distance vector 
      double dx, dy, dz; 
      dx = lx - rx; 
      dy = ly - ry; 
      dz = lz - rz; 
       
      double distance2 = dx * dx + dy * dy + dz * dz;  
       
      // Are monomers within cutoff distance? 
      if (distance2 < rCutoff) {  
         
        // Yes they are, so we have to compute repulsion 
        // force and update the accelerations 
         
        // Compute repulsion force 
        double f= repulsion->evaluate(distance2); 
        double fx = -dx * f; 
        double fy = -dy * f; 
        double fz = -dz * f; 
         
        // Add force to monomers 
        lmon.addAcceleration(fx, fy, fz); 
        rmon.addAcceleration(-fx, -fy, -fz); 
      }; 
    }; 
  }; 
   
  //  UPDATE ATTRACTIVE FORCE 
  // Loop over all bonds on monomers 
  for(left = 0; left < polymerSize; left++) { 
    // References to left  monomer 
    GdMonomer &lmon = (*polymer)[left]; 
     
    // Get positions 
    double lx, ly, lz; 
    lmon.getPosition(lx, ly, lz); 
     
    // Loop over currently bonded monomers 
    for(unsigned short right = 0; right < lmon.getNrBonds(); right++){ 
       
      // Get positions 
      GdMonomer &rmon = lmon.getBond(right); 
      double rx, ry, rz; 
      rmon.getPosition(rx, ry, rz); 
       
      // Distance vector 
      double dx, dy, dz; 
      dx = lx - rx; 
      dy = ly - ry; 
      dz = lz - rz; 
      double distance2 = dx * dx + dy * dy + dz * dz;  
       
      // Compute attraction force 
      double f = attraction->evaluate(distance2); 
      double fx = -dx * f; 
      double fy = -dy * f; 
      double fz = -dz * f; 
       
      // Add force to monomers 
      lmon.addAcceleration(fx, fy, fz); 
      rmon.addAcceleration(-fx, -fy, -fz); 
    }; 
  }; 
}; 
 
 
 
 
