/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

#include "cmp.h"
#include "kalc.h"


void insertObject(Object obj)
{
  /*
   * This function insert a copy of the Object obj in the stack.
   */

  insertNoDup(*objdup(&obj));
}


void insertNoDup(Object obj) 
{
  /*
   * This function inserts the object in the stack, without duplicating
   * it first.
   */

  if (type(obj) == TYPE_NOTANOBJECT)
    return;
  
  if (!freeSpace()) {
    /* Grow stack */
    Object **newStack;
    int currDepth = _f_depth();

    currStackCap += STACKINCREMENT;
    newStack = (Object **) realloc(s, (currStackCap + 1)*sizeof(Object *));
    if (newStack == NULL)
      doError("", ERR_STACKOVERFLOW);

    s = newStack;
    tos = s + currDepth;
  }
  
  tos++;
  if ((*tos = (Object *) malloc(sizeof(Object))) == NULL) {
    tos--;
    doError("Fatal", ERR_NOTENOUGHMEMORY);
  }
  **tos = obj;
  dirty = 1;
}


Object *objdup(Object *obj) 
{
  /*
   * This function makes a copy of the object. The copied things are the
   * value for identifiers and strings, and for tagged objects the tag and
   * the object, which is duplicated by a recursive call to this function.
   * For composite objects, each object is duplicated with a recursive call.
   */

  Object *returnObj;

  if (type(*obj) == TYPE_NOTANOBJECT)
    return obj;
  
  returnObj = (Object *) malloc(sizeof(Object));

  if (!returnObj)
    doError("Fatal", ERR_NOTENOUGHMEMORY);

  returnObj->type = obj->type;
  
  if (type(*obj) == TYPE_STR || type(*obj) == TYPE_ID
      || type(*obj) == TYPE_UNQUOTEDID) {
    char *str = strdup(obj->value.str);
    
    if (!str)
      doError("Fatal", ERR_NOTENOUGHMEMORY);

    returnObj->value.str = str;
  } else if (type(*obj) == TYPE_TAGGED) {
    char *newtag = strdup(obj->value.tagged.tag);
    Object *newobj = objdup(obj->value.tagged.obj);

    if (!newtag)
      doError("Fatal", ERR_NOTENOUGHMEMORY);
    
    returnObj->value.tagged.tag = newtag;
    returnObj->value.tagged.obj = newobj;
  } else if (type(*obj) == TYPE_PROG) {
    Composite *comp;
    Composite *oldComp, *lastComp = NULL;

    if (obj->value.comp == NULL) {
      returnObj->value.comp = NULL;
      return returnObj;
    }

    comp = (Composite *) malloc(sizeof(Composite));
    if (!comp)
      doError("Fatal", ERR_NOTENOUGHMEMORY);

    returnObj->value.comp = comp;

    oldComp = obj->value.comp;

    while (oldComp) {
      comp->prev = lastComp;
      comp->obj = objdup(oldComp->obj);
      if (lastComp)
	lastComp->next = comp;
      oldComp = oldComp->next;
      lastComp = comp;
      comp = (Composite *) malloc(sizeof(Composite));
    }
    free(comp);
    lastComp->next = NULL;
  } else
    returnObj->value = obj->value;
  
  return returnObj;
} 


void insertReal(double n)
{
  /*
   * This function inserts a real number in the stack.
   */

  Object obj;
  obj.type = TYPE_REAL;
  obj.value.real = n;
  insertObject(obj);
}


#if 0
void insertComplex(Complex cmp)
{
  /*
   * This function inserts a complex number in the stack.
   */

  Object obj;
  obj.type = TYPE_CMP;
  obj.value.cmp = cmp;
  insertObject(obj);
}


void insertString(char *str)
{
  /*
   * This function inserts a string in the stack.
   */

  Object obj;
  obj.type = TYPE_STR;
  obj.value.str = str;
  insertObject(obj);
}
#endif


void insertId(char *str)
{
  /*
   * This function inserts an identifier in the stack.
   */

  Object obj;
  obj.type = TYPE_ID;
  obj.value.str = str;
  insertObject(obj);
}


void f_dup(void)
{
  /*
   * This function makes a copy of the object in level one.
   */

  if (enoughArgs(1))
    insertObject(**tos);
  else
    doError("dup", ERR_TOOFEWARGUMENTS);
}


void f_dupdup(void)
{
  /*
   * This function makes two copy of the object in level one.
   */

  if (enoughArgs(1)) {
    insertObject(**tos);
    insertObject(**tos);
  } else
    doError("dupdup", ERR_TOOFEWARGUMENTS);
}


void f_ndupn(void)
{
  /*
   * This function makes n (in level one) copies of the object in level
   * two. If n is zero, drops the object in level two.
   */

  if (enoughArgs(2)) {
    if (type(**tos) == TYPE_REAL) {
      int n = (int) (**tos).value.real;

      _f_drop();
      if (n == 0)
	_f_drop();
      else
	while (--n)
	  insertObject(**tos);
    } else
      doError("ndupn", ERR_BADARGUMENTTYPE);
  } else
    doError("ndupn", ERR_TOOFEWARGUMENTS);
}     


void f_dup2(void)
{
  /*
   * This function makes copies of the objects in level one and two.
   */

  if (enoughArgs(2)) {
    insertObject(**(tos - 1));
    insertObject(**(tos - 1));
  }
}


void f_dupn(void)
{
  /*
   * This function makes copies of n (level one) objects, from levels
   * n+1 to 2.
   */

  int n;
  register int i;

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_REAL) {
      n = (int) (**tos).value.real;
      _f_drop();
      if (enoughArgs(n))
	for (i = 1; i <= n; i++)
	  insertObject(**(tos - n + 1));
    } else
      doError("dupn", ERR_BADARGUMENTTYPE);
  }
}

    
void f_drop(void)
{
  /*
   * This function drops an object, checking if there is something to drop
   * before.
   */

  if (enoughArgs(1)) {
    saveLastArgs(1);
    _f_drop();
  } else
    doError("drop", ERR_TOOFEWARGUMENTS);
}


void f_nip(void) 
{
  /*
   * This function drops the object in level two.
   */

  if (enoughArgs(2)) {
    _f_swap();
    _f_drop();
  } else
    doError("nip", ERR_TOOFEWARGUMENTS);
}


void f_drop2(void)
{
  /*
   * This function drops two objects, checking if there are enough to
   * drop before.
   */

  if (enoughArgs(2)) {
    saveLastArgs(2);
    _f_drop();
    _f_drop();
  } else
    doError("drop2", ERR_TOOFEWARGUMENTS);
}


void f_dropn(void)
{
  /*
   * This function checks if there is a real number in the stack, and then
   * tries to drop that many objects.
   */

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_REAL) {
      saveLastArgs(1);
      _f_dropn((int) (**tos).value.real + 1, "dropn");
    } else
      doError("dropn", ERR_BADARGUMENTTYPE);
  } else
    doError("dropn", ERR_TOOFEWARGUMENTS);
}


void _f_dropn(register int n, char *name)
{
  /*
   * This function drops n objects from the stack, checking if there are
   * enough objects before.
   */

  if (enoughArgs(n))
    for (; n > 0; n--)
      _f_drop();
  else
    doError(name, ERR_TOOFEWARGUMENTS);
}


void _f_drop(void)
{
  /*
   * This is a low-level function that drops the stack, without checking if
   * there is something to drop.
   */

  freeObjectSpecials(**tos);
  free(*tos);
  tos--;
  dirty = 1;
}


void f_clear(void)
{
  /*
   * This function drops the whole stack.
   */

  _f_dropn(_f_depth(), "clear");
}


void f_over(void)
{
  /*
   * This function tries to make a copy of the object in level two, if
   * there is such an object.
   */

  if (enoughArgs(2))
    insertObject(**(tos - 1));
  else
    doError("over", ERR_TOOFEWARGUMENTS);
}


void f_pick3(void)
{
  /*
   * This function does a 3 PICK, ie, it inserts a copy of the object in
   * level three.
   */

  if (enoughArgs(3))
    insertObject(**(tos - 2));
  else
    doError("pick3", ERR_TOOFEWARGUMENTS);
}


void f_pick(void)
{
  /*
   * This function tries to make a copy of the object in level n+1,
   * doing all checks before.
   */

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_REAL) {
      int item = (int) (**tos).value.real;
    
      if (enoughArgs(item + 1)) {
	saveLastArgs(1);
	
	_f_drop();
	if (item > 0)
	  insertObject(**(tos - item + 1));
      } else
	doError("pick", ERR_TOOFEWARGUMENTS);
    } else
      doError("pick", ERR_BADARGUMENTTYPE);
  } else 
    doError("pick", ERR_TOOFEWARGUMENTS);
}


void f_unpick(void) 
{
  /*
   * This function replaces the object in level n+2 (n is in level 1)
   * with the object in level one.
   */

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_REAL) {
      int n = (int) (**tos).value.real;

      if (n > 0) {
	if (enoughArgs(n+2)) {
	  Object *replacement = *(tos - 1);

	  _f_drop();
	  tos--;

	  freeLastArgs();
	  lastArgN = 1;
	  lastArgs[0] = *(tos - --n);
	  *(tos - n) = replacement;
	} else
	  doError("unpick", ERR_TOOFEWARGUMENTS);
      } else if (n == 0) {
        /* For n==0, behave as drop */
        if (enoughArgs(2)) {
          _f_drop();
          saveLastArgs(1);
          _f_drop();
        } else {
          doError("unpick", ERR_TOOFEWARGUMENTS);
        }
      } else {
	_f_drop();
	doError("unpick", ERR_BADARGUMENTVALUE);
      }
    } else
      doError("unpick", ERR_BADARGUMENTTYPE);
  } else
    doError("unpick", ERR_TOOFEWARGUMENTS);
}      


void f_swap(void)
{
  /*
   * This function checks if there are two objects in the stack and
   * swaps them if there are.
   */

  if (enoughArgs(2))
    _f_swap();
  else
    doError("swap", ERR_TOOFEWARGUMENTS);
}


void _f_swap(void)
{
  /*
   * This function swaps two objects, without cheking if there are two
   * objects.
   */

  Object *d;

  d = *tos;
  *tos = *(tos - 1);
  *(tos - 1) = d;
}  


void f_rot(void)
{
  /*
   * This function calls the roll function with an argument of three.
   */

  _f_roll(3, "rot");
}


void f_roll(void)
{
  /*
   *This function checks for argument presence and type and calls the
   * _f_roll function if everything is correct.
   */

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_REAL) {
      int n = (int) (**tos).value.real;

      saveLastArgs(1);

      _f_drop();
      if (n > 0)
	_f_roll(n, "roll");
    } else
      doError("roll", ERR_BADARGUMENTTYPE);
  } else
    doError("roll", ERR_TOOFEWARGUMENTS);
}


void _f_roll(int n, char *name)
{
  /*
   * This function rolls (upward) n stack elements. It makes no check to
   * see if there are n elements.
   */

  register int i;
  Object *tmp;

  if (enoughArgs(n)) {
    tmp = *(tos - n + 1);
    for (i = n - 1; i >= 1; i--)
      *(tos - i) = *(tos - i + 1);
    *tos = tmp;
  } else
    doError(name, ERR_TOOFEWARGUMENTS);
}


void f_unrot(void)
{
  /*
   * This function calls the rolld function with an argument of three.
   */

  _f_rolld(3, "unrot");
}


void f_rolld(void)
{
  /*
   * This function checks for argument presence and type and calls the
   * _f_rolld function if everything is correct.
   */

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_REAL) {
      int n = (int) (**tos).value.real;

      saveLastArgs(1);

      _f_drop();
      _f_rolld(n, "rolld");
    } else
      doError("rolld", ERR_TOOFEWARGUMENTS);
  } else
    doError("rolld", ERR_TOOFEWARGUMENTS);
}


void _f_rolld(int n, char *name)
{
  /*
   * This function rolls (downwards) n objects from the stack. It does not
   * check if there are n objects.
   */

  register int i;
  Object *tmp;

  if (enoughArgs(n)) {
    tmp = *tos;
    for (i = 0; i < n - 1; i++)
      *(tos - i) = *(tos - i - 1);
    *(tos - n + 1) = tmp;
  } else
    doError(name, ERR_TOOFEWARGUMENTS);
}


void f_keep(void) 
{
  /*
   * This function drops all elements except the bottom n (in level one)
   * objects.
   */
  int n;
  register int dep, i;

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_REAL) {      
      i = n = (**tos).value.real;

      saveLastArgs(1);

      _f_drop();

      dep = _f_depth();
      
      if (i < 0)
	doError("keep", ERR_BADARGUMENTVALUE);
      else if (i < dep) {
	while (i--)
	  _f_rolld(dep, "keep");

        _f_dropn(dep - n, "keep");
      }
    } else
      doError("keep", ERR_BADARGUMENTTYPE);
  } else
    doError("keep", ERR_TOOFEWARGUMENTS);
}    


void f_depth(void)
{
  /*
   * This function inserts the number of objects in the stack.
   */

  insertReal((double) _f_depth());
}
