/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.GlusterBrickBuilder;
import org.ovirt.engine.sdk4.types.GlusterBrick;

public interface GlusterBricksService extends Service {
    public interface ActivateRequest extends Request<ActivateRequest, ActivateResponse> {
        /**
         * Indicates if the activation should be performed asynchronously.
         */
        ActivateRequest async(Boolean async);
        ActivateRequest bricks(List<GlusterBrick> bricks);
        ActivateRequest bricks(GlusterBrick... bricks);
        ActivateRequest bricks(GlusterBrickBuilder... bricks);
    }
    
    public interface ActivateResponse extends Response {
    }
    
    ActivateRequest activate();
    
    /**
     * Adds given list of bricks to the volume, and updates the database accordingly. The properties `serverId` and
     * `brickDir`are required.
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest bricks(List<GlusterBrick> bricks);
        /**
         * The list of bricks to be added to the volume
         */
        AddRequest bricks(GlusterBrick... bricks);
        /**
         * The list of bricks to be added to the volume
         */
        AddRequest bricks(GlusterBrickBuilder... bricks);
    }
    
    /**
     * Adds given list of bricks to the volume, and updates the database accordingly. The properties `serverId` and
     * `brickDir`are required.
     */
    public interface AddResponse extends Response {
        /**
         * The list of bricks to be added to the volume
         */
        List<GlusterBrick> bricks();
    }
    
    /**
     * Adds given list of bricks to the volume, and updates the database accordingly. The properties `serverId` and
     * `brickDir`are required.
     */
    AddRequest add();
    
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Sets the maximum number of bricks to return. If not specified all the bricks are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of bricks to return. If not specified all the bricks are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of bricks to return. If not specified all the bricks are returned.
         */
        ListRequest max(BigInteger max);
    }
    
    public interface ListResponse extends Response {
        List<GlusterBrick> bricks();
    }
    
    ListRequest list();
    
    public interface MigrateRequest extends Request<MigrateRequest, MigrateResponse> {
        /**
         * Indicates if the migration should be performed asynchronously.
         */
        MigrateRequest async(Boolean async);
        MigrateRequest bricks(List<GlusterBrick> bricks);
        MigrateRequest bricks(GlusterBrick... bricks);
        MigrateRequest bricks(GlusterBrickBuilder... bricks);
    }
    
    public interface MigrateResponse extends Response {
    }
    
    MigrateRequest migrate();
    
    /**
     * Removes the given list of bricks brick from the volume and deletes them from the database.
     */
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
        RemoveRequest bricks(List<GlusterBrick> bricks);
        /**
         * The list of bricks to be removed
         */
        RemoveRequest bricks(GlusterBrick... bricks);
        /**
         * The list of bricks to be removed
         */
        RemoveRequest bricks(GlusterBrickBuilder... bricks);
    }
    
    /**
     * Removes the given list of bricks brick from the volume and deletes them from the database.
     */
    public interface RemoveResponse extends Response {
    }
    
    /**
     * Removes the given list of bricks brick from the volume and deletes them from the database.
     */
    RemoveRequest remove();
    
    public interface StopMigrateRequest extends Request<StopMigrateRequest, StopMigrateResponse> {
        /**
         * Indicates if the action should be performed asynchronously.
         */
        StopMigrateRequest async(Boolean async);
        StopMigrateRequest bricks(List<GlusterBrick> bricks);
        StopMigrateRequest bricks(GlusterBrick... bricks);
        StopMigrateRequest bricks(GlusterBrickBuilder... bricks);
    }
    
    public interface StopMigrateResponse extends Response {
    }
    
    StopMigrateRequest stopMigrate();
    
    GlusterBrickService brickService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

