/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.xml;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import org.ovirt.api.metamodel.runtime.util.ArrayListWithHref;
import org.ovirt.api.metamodel.runtime.util.ListWithHref;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.engine.sdk4.internal.containers.ExternalHostProviderContainer;
import org.ovirt.engine.sdk4.types.ExternalHostProvider;

public class XmlExternalHostProviderReader {
    
    public static ExternalHostProvider readOne(XmlReader reader) {
        // Do nothing if there aren't more tags:
        if (!reader.forward()) {
            return null;
        }
        
        // Create the object:
        ExternalHostProviderContainer object = new ExternalHostProviderContainer();
        
        // Process the attributes:
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String name = reader.getAttributeLocalName(i);
            String image = reader.getAttributeValue(i);
            switch (name) {
                case "href":
                object.href(image);
                break;
                case "id":
                object.id(image);
                break;
                default:
                break;
            }
        }
        
        // Process the inner elements:
        reader.next();
        while (reader.forward()) {
            String name = reader.getLocalName();
            switch (name) {
                case "authentication_url":
                object.authenticationUrl(reader.readString());
                break;
                case "certificates":
                object.certificates(XmlCertificateReader.readMany(reader));
                break;
                case "comment":
                object.comment(reader.readString());
                break;
                case "compute_resources":
                object.computeResources(XmlExternalComputeResourceReader.readMany(reader));
                break;
                case "description":
                object.description(reader.readString());
                break;
                case "discovered_hosts":
                object.discoveredHosts(XmlExternalDiscoveredHostReader.readMany(reader));
                break;
                case "host_groups":
                object.hostGroups(XmlExternalHostGroupReader.readMany(reader));
                break;
                case "hosts":
                object.hosts(XmlHostReader.readMany(reader));
                break;
                case "name":
                object.name(reader.readString());
                break;
                case "password":
                object.password(reader.readString());
                break;
                case "properties":
                object.properties(XmlPropertyReader.readMany(reader));
                break;
                case "requires_authentication":
                object.requiresAuthentication(reader.readBoolean());
                break;
                case "url":
                object.url(reader.readString());
                break;
                case "username":
                object.username(reader.readString());
                break;
                case "link":
                readLink(reader, object);
                break;
                default:
                reader.skip();
                break;
            }
        }
        
        // Discard the end tag:
        reader.next();
        
        return object;
    }
    
    public static Iterator<ExternalHostProvider> iterateMany(final XmlReader reader) {
        return new Iterator<ExternalHostProvider>() {
            private boolean first = true;
            
            @Override
            public void remove() {
                // Empty on purpose
            }
            
            @Override
            public boolean hasNext() {
                if (first) {
                    if (!reader.forward()) {
                        return false;
                    }
                    reader.next();
                    first = false;
                }
                if (!reader.forward()) {
                    reader.next();
                    return false;
                }
                return true;
            }
            
            @Override
            public ExternalHostProvider next() {
                ExternalHostProvider next = readOne(reader);
                if (next == null) {
                    throw new NoSuchElementException();
                }
                return next;
            }
        };
    }
    
    public static List<ExternalHostProvider> readMany(XmlReader reader) {
        List<ExternalHostProvider> list = new ArrayList<>();
        Iterator<ExternalHostProvider> iterator = iterateMany(reader);
        while (iterator.hasNext()) {
            list.add(iterator.next());
        }
        return list;
    }
    
    private static void readLink(XmlReader reader, ExternalHostProviderContainer object) {
        // Process the attributes:
        String rel = null;
        String href = null;
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String attrName = reader.getAttributeLocalName(i);
            String attrVal = reader.getAttributeValue(i);
            switch (attrName) {
                case "href":
                href = attrVal;
                break;
                case "rel":
                rel = attrVal;
                break;
                default:
                reader.skip();
                break;
            }
        }
        if (href != null) {
            ListWithHref list = new ArrayListWithHref();
            list.href(href);
            switch (rel) {
                case "certificates":
                object.certificates(list);
                break;
                case "computeresources":
                object.computeResources(list);
                break;
                case "discoveredhosts":
                object.discoveredHosts(list);
                break;
                case "hostgroups":
                object.hostGroups(list);
                break;
                case "hosts":
                object.hosts(list);
                break;
                default:
                break;
            }
        }
        reader.skip();
    }
}

