/****************************************************************
Copyright (C) Lucent Technologies 1997
All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the name Lucent Technologies or any of
its entities not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.

LUCENT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
IN NO EVENT SHALL LUCENT OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
THIS SOFTWARE.
****************************************************************/

#define	DEBUG
#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include "awklib.h"
#include "awkgram.h"
#include "proto.h"

#define	FULLTAB	2	/* rehash when table gets this x full */
#define	GROWTAB 4	/* grow table by this factor */

static void	rehash(awkarray_t *);

static awkcell_t	*nullloc;	/* a guaranteed empty cell */

/* initialize symbol table with builtin vars */
void
awklib_syminit(awk_t *awkp)
{
	awkp->literal0 = setawkvar(awkp, "0", "0", 0.0, NUM|STR|CON|DONTFREE, awkp->symtab);
	/* this is used for if(x)... tests: */
	nullloc = setawkvar(awkp, "$zero&null", "", 0.0, NUM|STR|CON|DONTFREE, awkp->symtab);
	awkp->nullnode = awklib_celltonode(awkp, nullloc, CCON);

	awkp->fsloc = setawkvar(awkp, "FS", " ", 0.0, STR|DONTFREE, awkp->symtab);
	awkp->FS = &awkp->fsloc->sval;
	awkp->RS = &setawkvar(awkp, "RS", "\n", 0.0, STR|DONTFREE, awkp->symtab)->sval;
	awkp->OFS = &setawkvar(awkp, "OFS", " ", 0.0, STR|DONTFREE, awkp->symtab)->sval;
	awkp->ORS = &setawkvar(awkp, "ORS", "\n", 0.0, STR|DONTFREE, awkp->symtab)->sval;
	awkp->OFMT = &setawkvar(awkp, "OFMT", "%.6g", 0.0, STR|DONTFREE, awkp->symtab)->sval;
	awkp->CONVFMT = &setawkvar(awkp, "CONVFMT", "%.6g", 0.0, STR|DONTFREE, awkp->symtab)->sval;
	awkp->FILENAME = &setawkvar(awkp, "FILENAME", "", 0.0, STR|DONTFREE, awkp->symtab)->sval;
	awkp->nfloc = setawkvar(awkp, "NF", "", 0.0, NUM, awkp->symtab);
	awkp->NF = &awkp->nfloc->fval;
	awkp->nrloc = setawkvar(awkp, "NR", "", 0.0, NUM, awkp->symtab);
	awkp->NR = &awkp->nrloc->fval;
	awkp->fnrloc = setawkvar(awkp, "FNR", "", 0.0, NUM, awkp->symtab);
	awkp->FNR = &awkp->fnrloc->fval;
	awkp->SUBSEP = &setawkvar(awkp, "SUBSEP", "\034", 0.0, STR|DONTFREE, awkp->symtab)->sval;
	awkp->rstartloc = setawkvar(awkp, "RSTART", "", 0.0, NUM, awkp->symtab);
	awkp->RSTART = &awkp->rstartloc->fval;
	awkp->rlengthloc = setawkvar(awkp, "RLENGTH", "", 0.0, NUM, awkp->symtab);
	awkp->RLENGTH = &awkp->rlengthloc->fval;
	awkp->symtabloc = setawkvar(awkp, "SYMTAB", "", 0.0, ARR, awkp->symtab);
	awkp->symtabloc->sval = (char *) awkp->symtab;
}

/* set up ARGV and ARGC */
void
awklib_arginit(awk_t *awkp, int ac, char **av)
{
	awkcell_t *cp;
	int i;
	char temp[50];

	awkp->ARGC = &setawkvar(awkp, "ARGC", "", (awkfloat_t) ac, NUM, awkp->symtab)->fval;
	cp = setawkvar(awkp, "ARGV", "", 0.0, ARR, awkp->symtab);
	awkp->ARGVtab = awklib_makesymtab(awkp, NSYMTAB);	/* could be (int) ARGC as well */
	cp->sval = (char *) awkp->ARGVtab;
	for (i = 0; i < ac; i++) {
		(void) snprintf(temp, sizeof(temp), "%d", i);
		if (awklib_is_number(*av)) {
			setawkvar(awkp, temp, *av, atof(*av), STR|NUM, awkp->ARGVtab);
		} else {
			setawkvar(awkp, temp, *av, 0.0, STR, awkp->ARGVtab);
		}
		av++;
	}
}

extern	char	**environ;

/* set up ENVIRON variable */
void
awklib_envinit(awk_t *awkp, char **envp)
{
	awkcell_t 	 *cp;
	char 		 *p;

	if (!awkp->safe) {
		cp = setawkvar(awkp, "ENVIRON", "", 0.0, ARR, awkp->symtab);
		awkp->ENVtab = awklib_makesymtab(awkp, NSYMTAB);
		cp->sval = (char *) awkp->ENVtab;
		if (envp == NULL) {
			envp = environ;
		}
		for ( ; *envp; envp++) {
			if ((p = strchr(*envp, '=')) == NULL)
				continue;
			if (p == *envp) /* no left hand side name in env string */
				continue;
			*p++ = 0;	/* split into two strings at = */
			if (awklib_is_number(p))
				setawkvar(awkp, *envp, p, atof(p), STR|NUM, awkp->ENVtab);
			else
				setawkvar(awkp, *envp, p, 0.0, STR, awkp->ENVtab);
			p[-1] = '=';	/* restore in case env is passed down to a shell */
		}
	}
}

/* set a variable in the awk struct */
int
awklib_setvar(awk_t *awkp, const char *varname, int ivalue, void *ptr)
{
	if (strcmp(varname, "FS") == 0) {
		if (*awkp->FS) {
			(void) free(*awkp->FS);
		}
		*awkp->FS = strdup(ptr);
	} else if (strcmp(varname, "safe") == 0) {
		awkp->stage = ivalue;
	} else if (strcmp(varname, "stage") == 0) {
		awkp->stage = ivalue;
	} else if (strcmp(varname, "cmdname") == 0) {
		if (awkp->cmdname) {
			(void) free(awkp->cmdname);
		}
		awkp->cmdname = strdup(ptr);
	} else if (strcmp(varname, "lexprog") == 0) {
		if (awkp->lexprog) {
			(void) free(awkp->lexprog);
		}
		awkp->lexprog = strdup(ptr);
	} else if (strcmp(varname, "script") == 0) {
		if (awkp->scriptsize == 0) {
			awkp->scripts = calloc(20, sizeof(char *));
			if (awkp->scripts == NULL) {
				(void) fprintf(stderr, "can't calloc space for scripts array\n");
				return 0;
			}
		} else if (awkp->scriptc == awkp->scriptsize) {
			char	**newarray;
			int	  newsize = awkp->scriptsize * 2;

			newarray = realloc(awkp->scripts, newsize * sizeof(char *));
			if (newarray == NULL) {
				(void) fprintf(stderr, "can't realloc space for scripts array\n");
				return 0;
			}
			awkp->scripts = newarray;
			awkp->scriptsize = newsize;
		}
		awkp->scripts[awkp->scriptc++] = strdup(ptr);
	} else if (strcmp(varname, "stage") == 0) {
		awkp->stage = ivalue;
	} else if (strcmp(varname, "debug") == 0) {
		awkp->dbg = ivalue;
	} else if (strcmp(varname, "symtab") == 0) {
		awkp->symtab = ptr;
	}
	return 1;
}

/* get the value as an integer */
int
awklib_getvarint(awk_t *awkp, const char *varname)
{
	if (strcmp(varname, "script count") == 0) {
		return awkp->scriptc;
	}
	if (strcmp(varname, "debug") == 0) {
		return awkp->dbg;
	}
	if (strcmp(varname, "current record size") == 0) {
		return awkp->recsize;
	}
	if (strcmp(varname, "error flag") == 0) {
		return awkp->errorflag;
	}
	return 0;
}

/* get the value as a pointer */
void *
awklib_getvarptr(awk_t *awkp, const char *varname)
{
	if (strcmp(varname, "cmdname") == 0) {
		return awkp->cmdname;
	}
	if (strcmp(varname, "winner") == 0) {
		return awkp->winner;
	}
	return NULL;
}

/* make a new symbol table */
awkarray_t *
awklib_makesymtab(awk_t *awkp, int n)
{
	awkarray_t *ap;
	awkcell_t **arr;

	ap = malloc(sizeof(awkarray_t));
	arr = calloc(n, sizeof(awkcell_t *));
	if (ap == NULL || arr == NULL)
		FATAL(awkp, "out of space in awklib_makesymtab");
	ap->nelem = 0;
	ap->size = n;
	ap->tab = arr;
	return(ap);
}

/* free a symbol table */
void
awklib_freesymtab(awk_t *awkp, awkcell_t *ap)
{
	awkcell_t *cp, *temp;
	awkarray_t *arr;
	int i;

	if (!ISARR(ap))
		return;
	arr = (awkarray_t *) ap->sval;
	if (arr == NULL)
		return;
	for (i = 0; i < arr->size; i++) {
		for (cp = arr->tab[i]; cp != NULL; cp = temp) {
			XFREE(cp->nval);
			if (FREEABLE(cp))
				XFREE(cp->sval);
			temp = cp->cnext;	/* avoids freeing then using */
			free(cp); 
			arr->nelem--;
		}
		arr->tab[i] = 0;
	}
	if (arr->nelem != 0)
		WARNING(awkp, "can't happen: inconsistent element count freeing %s", ap->nval);
	free(arr->tab);
	free(arr);
}

/* free elem s from ap (i.e., ap["s"] */
void
awklib_freeelem(awkcell_t *ap, const char *s)
{
	awkarray_t *arr;
	awkcell_t *p, *prev = NULL;
	int h;
	
	arr = (awkarray_t *) ap->sval;
	h = awklib_hash(s, arr->size);
	for (p = arr->tab[h]; p != NULL; prev = p, p = p->cnext)
		if (strcmp(s, p->nval) == 0) {
			if (prev == NULL)	/* 1st one */
				arr->tab[h] = p->cnext;
			else			/* middle somewhere */
				prev->cnext = p->cnext;
			if (FREEABLE(p))
				XFREE(p->sval);
			free(p->nval);
			free(p);
			arr->nelem--;
			return;
		}
}

awkcell_t *
setawkvar(awk_t *awkp, const char *n, const char *s, awkfloat_t f, unsigned t, awkarray_t *arr)
{
	int h;
	awkcell_t *p;

	if (n == NULL)
		n = "";

	if ((p = awklib_lookup(n, arr)) != NULL) {
		DPRINTF(awkp, ("setawkvar found %p: n=%s s=\"%s\" f=%g t=%o\n",
			p, NN(p->nval), NN(p->sval), p->fval, p->tval) );
		return(p);
	}
	p = malloc(sizeof(awkcell_t));
	if (p == NULL)
		FATAL(awkp, "out of space for symbol table at %s", n);
	p->nval = awklib_tostring(awkp, n);
	p->sval = awklib_tostring(awkp, (s) ? s : "");
	p->fval = f;
	p->tval = t;
	p->csub = CUNK;
	p->ctype = OCELL;
	arr->nelem++;
	if (arr->nelem > FULLTAB * arr->size)
		rehash(arr);
	h = awklib_hash(n, arr->size);
	p->cnext = arr->tab[h];
	arr->tab[h] = p;
	DPRINTF(awkp, ("setawkvar set %p: n=%s s=\"%s\" f=%g t=%o\n",
		p, p->nval, p->sval, p->fval, p->tval) );
	return(p);
}

/* form hash value for string s */
int
awklib_hash(const char *s, int n)
{
	unsigned hashval;

	for (hashval = 0; *s != '\0'; s++)
		hashval = (*s + 31 * hashval);
	return hashval % n;
}

/* rehash items in small table into big one */
static void
rehash(awkarray_t *arr)
{
	int i, nh, nsz;
	awkcell_t *cp, *op, **np;

	nsz = GROWTAB * arr->size;
	np = calloc(nsz, sizeof(awkcell_t *));
	if (np == NULL)		/* can't do it, but can keep running. */
		return;		/* someone else will run out later. */
	for (i = 0; i < arr->size; i++) {
		for (cp = arr->tab[i]; cp; cp = op) {
			op = cp->cnext;
			nh = awklib_hash(cp->nval, nsz);
			cp->cnext = np[nh];
			np[nh] = cp;
		}
	}
	free(arr->tab);
	arr->tab = np;
	arr->size = nsz;
}

/* look for s in arr */
awkcell_t *
awklib_lookup(const char *s, awkarray_t *arr)
{
	awkcell_t *p;
	int h;

	h = awklib_hash(s, arr->size);
	for (p = arr->tab[h]; p != NULL; p = p->cnext)
		if (strcmp(s, p->nval) == 0)
			return(p);	/* found it */
	return(NULL);			/* not found */
}

/* set float val of a awkcell_t */
awkfloat_t
awklib_setfval(awk_t *awkp, awkcell_t *vp, awkfloat_t f)
{
	int fldno;

	f += 0.0;		/* normalise negative zero to positive zero */
	if ((vp->tval & (NUM | STR)) == 0) 
		awklib_funnyvar(awkp, vp, "assign to");
	if (ISFLD(vp)) {
		awkp->donerec = 0;	/* mark $0 invalid */
		fldno = atoi(vp->nval);
		if (fldno > *awkp->NF)
			awklib_newfld(awkp, fldno);
		DPRINTF(awkp, ("setting field %d to %g\n", fldno, f) );
	} else if (ISREC(vp)) {
		awkp->donefld = 0;	/* mark $1... invalid */
		awkp->donerec = 1;
	}
	if (FREEABLE(vp))
		XFREE(vp->sval); /* free any previous string */
	vp->tval &= ~STR;	/* mark string invalid */
	vp->tval |= NUM;	/* mark number ok */
	DPRINTF(awkp, ("awklib_setfval %p: %s = %g, t=%o\n", vp, NN(vp->nval), f, vp->tval) );
	return vp->fval = f;
}

void
awklib_funnyvar(awk_t *awkp, awkcell_t *vp, const char *rw)
{
	if (ISARR(vp))
		FATAL(awkp, "can't %s %s; it's an array name.", rw, vp->nval);
	if (vp->tval & FCN)
		FATAL(awkp, "can't %s %s; it's a function.", rw, vp->nval);
	WARNING(awkp, "funny variable %p: n=%s s=\"%s\" f=%g t=%o",
		vp, vp->nval, vp->sval, vp->fval, vp->tval);
}

char *
awklib_setsval(awk_t *awkp, awkcell_t *vp, const char *s)	/* set string val of a awkcell_t */
{
	char *t;
	int fldno;

	DPRINTF(awkp, ("starting awklib_setsval %p: %s = \"%s\", t=%o, r,f=%d,%d\n", 
		vp, NN(vp->nval), s, vp->tval, awkp->donerec, awkp->donefld) );
	if ((vp->tval & (NUM | STR)) == 0)
		awklib_funnyvar(awkp, vp, "assign to");
	if (ISFLD(vp)) {
		awkp->donerec = 0;	/* mark $0 invalid */
		fldno = atoi(vp->nval);
		if (fldno > *awkp->NF)
			awklib_newfld(awkp, fldno);
		DPRINTF(awkp, ("setting field %d to %s (%p)\n", fldno, s, s) );
	} else if (ISREC(vp)) {
		awkp->donefld = 0;	/* mark $1... invalid */
		awkp->donerec = 1;
	}
	t = awklib_tostring(awkp, s);	/* in case it's self-assign */
	vp->tval &= ~NUM;
	vp->tval |= STR;
	if (FREEABLE(vp))
		XFREE(vp->sval);
	vp->tval &= ~DONTFREE;
	DPRINTF(awkp, ("awklib_setsval %p: %s = \"%s (%p) \", t=%o r,f=%d,%d\n", 
		vp, NN(vp->nval), t,t, vp->tval, awkp->donerec, awkp->donefld) );
	return(vp->sval = t);
}

awkfloat_t
awklib_getfval(awk_t *awkp, awkcell_t *vp)	/* get float val of a awkcell_t */
{
	if ((vp->tval & (NUM | STR)) == 0)
		awklib_funnyvar(awkp, vp, "read value of");
	if (ISFLD(vp) && awkp->donefld == 0)
		awklib_fldbld(awkp);
	else if (ISREC(vp) && awkp->donerec == 0)
		awklib_recbld(awkp);
	if (!ISNUM(vp)) {	/* not a number */
		vp->fval = atof(vp->sval);	/* best guess */
		if (awklib_is_number(vp->sval) && !(vp->tval&CON))
			vp->tval |= NUM;	/* make NUM only sparingly */
	}
	DPRINTF(awkp, ("awklib_getfval %p: %s = %g, t=%o\n", vp, NN(vp->nval), vp->fval, vp->tval) );
	return(vp->fval);
}

/* get string val of a awkcell_t */
static char *
get_str_val(awk_t *awkp, awkcell_t *vp, char **fmt)
{
	char s[100];
	double dtemp;

	if ((vp->tval & (NUM | STR)) == 0)
		awklib_funnyvar(awkp, vp, "read value of");
	if (ISFLD(vp) && awkp->donefld == 0)
		awklib_fldbld(awkp);
	else if (ISREC(vp) && awkp->donerec == 0)
		awklib_recbld(awkp);
	if (ISSTR(vp) == 0) {
		if (FREEABLE(vp))
			XFREE(vp->sval);
		if (modf(vp->fval, &dtemp) == 0)	/* it's integral */
			snprintf(s, sizeof(s), "%.30g", vp->fval);
		else
			snprintf(s, sizeof(s), *fmt, vp->fval);
		vp->sval = awklib_tostring(awkp, s);
		vp->tval &= ~DONTFREE;
		vp->tval |= STR;
	}
	DPRINTF(awkp, ("awklib_getsval %p: %s = \"%s (%p)\", t=%o\n", vp, NN(vp->nval), vp->sval, vp->sval, vp->tval) );
	return(vp->sval);
}

/* get string val of a awkcell_t */
char *
awklib_getsval(awk_t *awkp, awkcell_t *vp)
{
      return get_str_val(awkp, vp, awkp->CONVFMT);
}

/* get string val of a awkcell_t for print */
char *
awklib_getpssval(awk_t *awkp, awkcell_t *vp)
{
      return get_str_val(awkp, vp, awkp->OFMT);
}


/* make a copy of string s */
char *
awklib_tostring(awk_t *awkp, const char *s)
{
	char *p;

	p = strdup(s);
	if (p == NULL)
		FATAL(awkp, "out of space in awklib_tostring on %s", s);
	return(p);
}

/* make a copy of string s */
char *
awklib_tostringN(awk_t *awkp, const char *s, size_t n)
{
	char *p;

	p = malloc(n + 1);
	if (p == NULL)
		FATAL(awkp, "out of space in awklib_tostring on %s", s);
	(void) memcpy(p, s, n);
	p[n] = 0x0;
	return(p);
}

/* collect string up to next delim */
char *
awklib_qstring(awk_t *awkp, const char *is, int delim)
{
	const char *os = is;
	int c, n;
	uint8_t *s = (uint8_t *) is;
	uint8_t *buf, *bp;

	if ((buf = malloc(strlen(is)+3)) == NULL)
		FATAL(awkp,  "out of space in awklib_qstring(%s)", s);
	for (bp = buf; (c = *s) != delim; s++) {
		if (c == '\n')
			SYNTAX(awkp,  "newline in string %.20s...", os );
		else if (c != '\\')
			*bp++ = c;
		else {	/* \something */
			c = *++s;
			if (c == 0) {	/* \ at end */
				*bp++ = '\\';
				break;	/* for loop */
			}	
			switch (c) {
			case '\\':	*bp++ = '\\'; break;
			case 'n':	*bp++ = '\n'; break;
			case 't':	*bp++ = '\t'; break;
			case 'b':	*bp++ = '\b'; break;
			case 'f':	*bp++ = '\f'; break;
			case 'r':	*bp++ = '\r'; break;
			default:
				if (!isdigit(c)) {
					*bp++ = c;
					break;
				}
				n = c - '0';
				if (isdigit(s[1])) {
					n = 8 * n + *++s - '0';
					if (isdigit(s[1]))
						n = 8 * n + *++s - '0';
				}
				*bp++ = n;
				break;
			}
		}
	}
	*bp++ = 0;
	return (char *) buf;
}
