/*
 * This is the testing module -- type "h" at the prompt
 * If you want to test the setuid/setgid stuff, you
 * should make this setuid to root
 *
 * Author information:
 * Matt Bishop
 * Department of Computer Science
 * University of California at Davis
 * Davis, CA  95616-8562
 * phone (916) 752-8060
 * email bishop@cs.ucdavis.edu
 *
 * This code is placed in the public domain.  I do ask that
 * you keep my name associated with it, that you not represent
 * it as written by you, and that you preserve these comments.
 * This software is provided "as is" and without any guarantees
 * of any sort.
 *
 * Version information:
 * 1.0		May 25, 1994		Matt Bishop
 */
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <signal.h>
#include "env.h"

/*
 * this is the signal handler
 *
 * Why is this here?  You really don't need it, but during
 * testing I got a SIGPIPE that I couldn't explain; turned
 * out there was a problem with dup'ing a file descriptor
 * in the child creation routine.  So, I fixed it, but just
 * in case there's a problem with porting this thing, I've
 * left the code in place for future use.
 *
 * If you get a signal, quickout is set to 1 so the mpopen
 * loops terminate
 */
int quickout = 0;		/* drop out of loop */

/*
 * primitive signal handler
 */
void subsig(int signo)
{
	/*
	 * be informative on SIGPIPE or child
	 * termination, and cryptic on everything else
	 */
	if (signo == SIGPIPE || signo == SIGCHLD){
		printf("child died; signal ");
		if (signo == SIGCHLD)
			printf("SIGCHLD");
		else if (signo == SIGPIPE)
			printf("SIGPIPE");
		else
			printf("%d", signo);
		printf(" caught\n");
	}
	/* say you got something */
	quickout = 1;
	/*
	 * reset these (needed on some systems,
	 * unnecessary on others)
	 */
	(void) signal(SIGPIPE, subsig);
	(void) signal(SIGCHLD, subsig);
}

/*
 * discard leading whitespace
 */
char *eatblanks(char *x)
{
	while(isspace(*x)) x++;
	return(x);
}

/*
 * error handler
 * just say what happened
 */
void oops(int code)
{
	switch(code){
	case SE_NONE: 			/* no error */
		return;
	case SE_NOMEM: 			/* no memory */
		printf("ran out of memory\n");
		return;
	case SE_NOPIPE: 		/* no pipes */
		printf("can't create another pipe\n");
		return;
	case SE_NOVAR: 			/* no variable */
		printf("unknown environment variable\n");
		return;
	case SE_BADFD: 		/* no file descriptor */
		printf("unknown file descriptor\n");
		return;
	default:			/* no idea! */
		printf("unknown error %d\n", code);
		return;
	}
}

/*
 * the start of the problem
 */
void main(void)
{
	char buf[1024];		/* input buffer */
	char cmd[1024];		/* command buffer */
	char *p;		/* used to walk command buffer */
	int um;			/* temporary used for a multitude of sins */
	FILE *fp;		/* used for mpopen/mpclose */

	/*
	 * may be paranoia, but catch these
	 * in case there are kiddie problems
	 */
	(void) signal(SIGPIPE, subsig);
	(void) signal(SIGCHLD, subsig);

	/*
	 * do it!
	 */
	while(printf("> "), gets(buf) != NULL){
		/* eat leading blanks */
		p = eatblanks(buf);
		/* do the command */
		switch(buf[0]){
		case 'i':		/* initialize */
			le_clobber();
			break;
		case 'u':		/* set effective user id */
			p = eatblanks(++p);
			if (sscanf(p, "%d", &um) != 1)
				printf("need numeric uid\n");
			else
				oops(le_euid(um));
			break;
		case 'g':		/* set effective group id */
			p = eatblanks(++p);
			if (sscanf(p, "%d", &um) != 1)
				printf("need numeric gid\n");
			else
				oops(le_egid(um));
			break;
		case 'a':		/* add environment variable */
			p = eatblanks(++p);
			oops(le_set(p));
			break;
		case 'd':		/* delete environment variable */
			p = eatblanks(++p);
			oops(le_unset(p));
			break;
		case 'm':		/* set file creation mask */
			p = eatblanks(++p);
			if (sscanf(p, "%o", &um) != 1)
				printf("need octal umask\n");
			else
				oops(le_umask(um));
			break;
		case 'c':		/* set command to run */
			p = eatblanks(++p);
			(void) strcpy(cmd, p);
			break;
		case 'r':		/* run command using msystem */
			printf("status is %d\n", msystem(cmd));
			quickout = 0;
			break;
		case 'P':		/* run command, read from pipe */
			if ((fp = mpopen(cmd, "r")) == NULL)
				printf("%s could not be executed\n", cmd);
			else{
				while(!quickout && 
					   fgets(buf, sizeof(buf), fp) != NULL)
					printf("%s", buf);
				quickout = 0;
				printf("status is %d\n", mpclose(fp));
			}
			break;
		case 'p':		/* run command, write to pipe */
			if ((fp = mpopen(cmd, "w")) == NULL)
				printf("%s could not be executed\n", cmd);
			else{
				while(printf("input (. to stop)> "),
				  (!quickout &&
				    fgets(buf, 1024, stdin) != NULL &&
						strcmp(buf, ".\n") != 0)){
					fprintf(fp, "%s", buf);
					fflush(fp);
				}
				quickout = 0;
				printf("status is %d\n", mpclose(fp));
			}
			break;
		case '-':		/* fd to be kept open */
			if (scanf("%d", &um) == EOF)
				exit(1);
			oops(le_openfd(um));
			break;
		case '|':		/* fd to be kept closed */
			if (scanf("%d", &um) == EOF)
				exit(1);
			oops(le_closefd(um));
			break;
		case 'q':		/* exit */
			exit(1);
		default:		/* help message */
			printf("- nn\tkeep file descriptor nn open\n");
			printf("| nn\tkeep file descriptor nn closed\n");
			printf("a arg\tset environment variable\n");
			printf("c line\trest of line is command\n");
			printf("d arg\tunset environment variable\n");
			printf("g nnn\tset (real & effective) gid to nnn\n");
			printf("i\tinitialize environment\n");
			printf("m nnn\tset umask to nnn (octal)\n");
			printf("P\texecute command, pipe output\n");
			printf("p\texecute command, pipe from stdin\n");
			printf("q\tquit\n");
			printf("r\texecute command\n");
			printf("u nnn\tset (real & effective) uid to nnn\n");
			break;
		}
	}

	/*
	 * say good night, Dick!
	 */
	exit(0);
}
