/*
 *	  Copyright 1998-2000 by the Purdue Research Foundation for CERIAS (the
 *	  Center for Education and Research in Information Assurance and
 *	  Security).  All rights reserved.  This work may be used for
 *	  non-profit educational and research purposes only.  Any copies made
 *	  of this file or portions of its contents must include this copyright
 *	  statement.  For information on reuse, licensing, or copying, contact
 *	  <cerias-info@cerias.purdue.edu>.
 *
 *	  This software is experimental in nature and is provided without any
 *	  express or implied warranties, including, without limitation, the
 *	  implied warranties of merchantability and fitness for any particular
 *	  purpose.
 *
 *  $Id: BSMTokenStream.c,v 1.4 2000/01/31 03:56:00 flack Exp $
 *
 *  BSMTokenStream.c - Chapman Flack, flack@cs.purdue.edu
 *
 *  Some Options:       Defined                      Not Defined
 *
 *  EXPAND_IN_ADDR    Fields containing IP       Fields containing IP
 *                    addresses are returned     addresses are returned
 *                    in BSMBlocks in string     as BSMLongs in network
 *                    (dotted quad) notation     byte order
 *
 *  EXPAND_IP_TOS     IP types of service are    Raw numeric IP types of
 *                    returned in BSMBlocks      service are returned as
 *                    as strings IPTOS_LOWDELAY, BSMLongs.
 *                    IPTOS_THROUGHPUT, or
 *                    IPTOS_RELIABILITY
 *
 *  EXPAND_IP_PROTO   Protocol names (from       Raw protocol numbers are
 *                    getprotobynumber) are      returned as BSMLongs.
 *                    returned in BSMBlocks.
 *
 *  This module does not yet support every token type declared in
 *  /usr/include/bsm/audit_record.h. It supports those that appeared in a
 *  set of sample files available in the lab during development. There
 *  are two degrees of "non-support" possible for a BSM token:
 *
 *  1)  It can fall through to the default case in getNextToken(), throwing a
 *      BSMScannerException.  If you run this module against an input source
 *      that elicits this exception, you are out of luck until you write a
 *      TOKFN for that token type and add the appropriate call to its
 *      label in the switch statement. Until then, this module will not
 *      know how many bytes to read to reach the next token boundary, and
 *      will not be able to continue reading.  However, if the token is
 *      one you are not otherwise interested in, you need do no more than
 *	bring it to the second degree of non-support:
 *
 *  2)  It can have a TOKFN defined and properly called from the switch
 *      statement, but the TOKFN simply contains PUNT, which produces a
 *    	Token for the token type but nothing else.  (You should adjust the
 *      grammar so this is all it expects, or a parse exception will result.)
 *      In this case the input file is advanced the proper amount so the
 *      remainder of the file can still be read; so, PUNTing need not
 *      be of concern until you really need the information conveyed in that
 *      type of token. However, for those tokens whose sizes are variable,
 *      even a minimal TOKFN must calculate the correct size and read the
 *      remainder of the token before PUNTing.
 *
 *  Adding support for new token types involves the macros TOKFN, TOK, TOKTILL,
 *  EMIT, and others, whose descriptions are below at their definitions.
 *  The big switch() on token IDs is at the very end of this file.
 *  
 */
static char const RCSid[] = "$Id: BSMTokenStream.c,v 1.4 2000/01/31 03:56:00 flack Exp $";

#include "BSMTokenStream.h"

#include <assert.h>
#include <sys/types.h>
#include <bsm/audit.h>

#pragma pack(1)
#include <bsm/audit_record.h>
typedef struct au_newgroups_tok { /* if this is ever added to audit_record.h */
  u_short count;      	      	  /* just eliminate this stand-in definition */
  u_long groups[1];   	      	  /* and rework the TOKFN if need be */
  }
  au_newgroups_tok_t;
typedef struct au_exec_env_tok {  /* ditto this */
  u_long count;
  char env_args[1];
  }
  au_exec_env_tok_t, au_exec_args_tok_t;
#pragma pack()

#include "eventcache.h"

#include <sys/param.h>
#include <sys/sysmacros.h>

#include <bsm/libbsm.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#define FQJNICALL(name) \
  JNICALL Java_edu_purdue_cerias_projects_BSMParser_BSMTokenStream_##name

#define FQSIGCLASS(name) "Ledu/purdue/cerias/projects/BSMParser/" #name ";"

#if !defined(AUT_ARG)  &&  defined(AUT_ARG32)
/*
 *  Thank you, Solaris 2.7, for changing existing names and structures ... !
 *
 *  The conditional code here simply allows a 2.6-and-previous scanner to be
 *  built on a 2.7 platform.  The scanner and parser have not been updated to
 *  handle the new 2.7 log elements.
 */
#define AUT_ARG AUT_ARG32
typedef au_arg32_tok_t au_arg_tok_t;

#define AUT_RETURN AUT_RETURN32
typedef au_ret32_tok_t au_ret_tok_t;

#define AUT_HEADER AUT_HEADER32
typedef au_header32_tok_t au_header_tok_t;

#define AUT_SUBJECT AUT_SUBJECT32
typedef au_subj32_tok_t au_subj_tok_t;

#define AUT_SERVER AUT_SERVER32
/* but they didn't change au_server_tok_t! */

#define AUT_PROCESS AUT_PROCESS32
typedef au_proc32_tok_t au_proc_tok_t;

#define AUT_OTHER_FILE AUT_OTHER_FILE32
typedef au_file32_tok_t au_file_tok_t;

#define AUR_LONG AUR_INT64
#endif

static struct ClassMemberIDs {	/* cached to reduce JNI overhead */
  struct {
    jclass    class;
    jmethodID read_1;
    jmethodID read_n;
  } InputStream;
  
  struct {
    jclass    class;
    jfieldID  record;
  } BSMTokenStream;
  
  struct {
    jclass    class;
    jfieldID  next;
    jfieldID  record;
  } BSMToken;
  
  struct {
    jclass    class;
    jmethodID eof;
    jmethodID mysterytok;
  } BSMScannerException;
  
  struct {    /* extends BSMToken */
    jclass    class;
    jfieldID  value;
    jmethodID init;
  } BSMBlock, BSMLong, BSMDouble;
  
  struct {    /* extends BSMToken */
    jclass    class;
    jfieldID  length;
    jfieldID  version;
    jfieldID  emod;
    jfieldID  time;
    jfieldID  msec;
    jmethodID init;
  } BSMHeader;
  
  struct {    /* extends BSMToken */
    jclass    class;
    jmethodID init;
  } BSMProto;
} ID;

/*
 * Class:     BSMTokenStream
 * Method:    initialize
 */
JNIEXPORT void FQJNICALL(initialize)
  (JNIEnv *env, jclass class, jclass error, jclass input, jclass token,
   jclass lexerr, jclass blockCL, jclass longCL, jclass doubleCL,
   jclass headerCL, jclass protoCL, jthrowable memerr)
{
  if ( sizeof (jbyte) != 1 )
    (*env)->ThrowNew( env, error, "jbyte is not 8 bits");
    
#define CLASS(cls,arg) \
  if ( ID.cls.class = (*env)->NewGlobalRef( env, arg) ); else goto nomem  
#define FIELD(cls,name,sig) \
  if ( ID.cls.name = (*env)->GetFieldID ( env, ID.cls.class, #name, sig) ) \
    ; else return
#define METHOD(cls,name,sig,mem) \
  if ( ID.cls.mem = (*env)->GetMethodID( env, ID.cls.class, name, sig) ) \
    ; else return
#define CONSTRUCT(cls,mem,sig) METHOD(cls,"<init>",sig,mem)

  CLASS(  InputStream, input);
  METHOD( InputStream, "read", "()I"    , read_1);
  METHOD( InputStream, "read", "([BII)I", read_n);
    
  CLASS( BSMTokenStream, class);
  FIELD( BSMTokenStream, record, "I");
  
  CLASS( BSMToken, token);
  FIELD( BSMToken, next, FQSIGCLASS(BSMToken));
  FIELD( BSMToken, record, "I");
  
  CLASS( BSMScannerException, lexerr);
  CONSTRUCT( BSMScannerException, eof, "()V");
  CONSTRUCT( BSMScannerException, mysterytok, "(B)V");
  
  CLASS( BSMBlock, blockCL);
  FIELD( BSMBlock, value, "[B");
  CONSTRUCT( BSMBlock, init, "(I)V");
  
  CLASS( BSMLong, longCL);
  FIELD( BSMLong, value, "J");
  CONSTRUCT( BSMLong, init, "(I)V");
  
  CLASS( BSMDouble, doubleCL);
  FIELD( BSMDouble, value, "D");
  CONSTRUCT( BSMDouble, init, "(I)V");
  
  CLASS( BSMHeader, headerCL);
  FIELD( BSMHeader, length, "J");
  FIELD( BSMHeader, version, "S");
  FIELD( BSMHeader, emod, "I");
  FIELD( BSMHeader, time, "J");
  FIELD( BSMHeader, msec, "J");
  CONSTRUCT( BSMHeader, init, "(I)V");
  
  CLASS( BSMProto, protoCL);
  CONSTRUCT( BSMProto, init, "(I)V");
  
  return;

nomem:
  if ( NULL == (*env)->ExceptionOccurred( env) )
    (*env)->Throw( env, memerr);
}

typedef struct context {
  char	    alignhack [ 8 ];  /* must be first. See ALIGNED comments below. */
  JNIEnv    *env;
  jobject   this;
  jint      recordNo;
  jobject   input;
  jobject   token;
  jbyteArray buffer;
  jbyte     *pinned;
  jboolean  isCopy;
  jsize     bufferSize;       /* filled in by bufxtend thus by KREAD & EXTEND */
  jthrowable memoryER;
  int       suppressName;
} *context;
/*
 *  Because the BSM token structures are packed with no padding, odd address
 *  exceptions may be generated by direct reference to certain members.
 *  So, instead of t->member, use ALIGNED(member) which expands to an expression
 *  of the same type as t->member but aligned on an 8-byte boundary.
 *  (If sizeof t->member exceeds 8, t->member itself is returned, on the
 *  assumption that larger members are probably strings or similar composites
 *  which will be operated on by byte instructions that will not generate
 *  odd address exceptions.  Should this be a problem, it is necessary only
 *  to change size of alignhack (in struct context) from 8 to a larger value.)
 *  ALIGNED is used automatically by the EMIT macro.
 */
#define ALIGNED(member) (((tp)align(k,&(t->member), \
  (size_t)&(((tp)0)->member),sizeof t->member))->member)

static void *
align( context k, void *nonalignedbuf, size_t off, size_t size)
{
  if ( size > sizeof k->alignhack )
    return ((char *)nonalignedbuf) - off;	/* at least it might work */
  memcpy( k->alignhack, nonalignedbuf, size);
  return ((char *)(k->alignhack)) - off;
}
/*
 *  bufxtend ensures the buffer is large enough to contain the entire token
 *  to be read.  It is called directly by readstruct(); elsewhere it is used
 *  only via the EXTEND macro (described below near TOKTILL).
 *
 *  Such painstaking defensive coding was more necessary in the earliest
 *  versions of this module (else it would have been susceptible to graceless
 *  failure due to buffer overruns).  Now that Java I/O is used, a buffer of
 *  fixed generous size could be used blindly, as Java would throw an exception
 *  rather than overrunning it.  But the painstaking part was already done.
 */
void *
bufxtend( context k, size_t from, size_t nbyte, void *t)
{
  jsize old_size = (*(k->env))->GetArrayLength( k->env, k->buffer);

  register jsize
    buf_size = old_size,
    adjust = 0;
  
  nbyte += from;

  while ( buf_size < nbyte )
  {
    buf_size <<= 1;
    adjust = 1;
  }

  if ( adjust ) {
    jbyteArray obuf = k->buffer;
    k->buffer = (*(k->env))->NewByteArray( k->env, buf_size);
    if ( k->buffer == NULL ) {
      if ( NULL == (*(k->env))->ExceptionOccurred( k->env) )
      	(*(k->env))->Throw( k->env, k->memoryER);
      return NULL;
    }
    if ( k->pinned != NULL ) {
      (*(k->env))->SetByteArrayRegion( k->env, k->buffer, 0, old_size, k->pinned);
      (*(k->env))->ReleaseByteArrayElements( k->env, obuf, k->pinned, JNI_ABORT);
      k->pinned = NULL;
    }
  }
  
  if ( k->pinned == NULL ) {
    k->pinned = (*(k->env))->GetByteArrayElements( k->env, k->buffer, &k->isCopy);
    if ( k->pinned == NULL ) {
      if ( NULL == (*(k->env))->ExceptionOccurred( k->env) )
      	(*(k->env))->Throw( k->env, k->memoryER);
      return NULL;
    }
  }

  if ( t != NULL )
    *(void **)t = k->pinned;
  k->bufferSize = buf_size;
  return k->pinned + from;
}
/*
 *  KREAD is used to read nbyte bytes into space pointed to by buffer.
 *  It is assumed and required that buffer points somewhere within the
 *  already-allocated token-buffer space whose base address is in t.
 *  Note that (in a Java implementation that does copying rather than pinning)
 *  the entire buffer may be relocated as a side effect of KREAD (i.e. a new
 *  base address may be written into t).
 */
#define KREAD(buffer,nbyte) if (buffer=kread(k,buffer,nbyte,&t));else return

void *
kread( context k, void *buffer, size_t nbyte, void *t)
{
  size_t offset = 0;
  jint where, got;
  
  if ( k->pinned != NULL ) {
    offset = ((jbyte *)buffer) - k->pinned;
    if ( k->isCopy ) {
      (*(k->env))->ReleaseByteArrayElements( k->env,
      	      	      	      	      	     k->buffer, k->pinned, JNI_ABORT);
      k->pinned = NULL;
    }
  }
  
  where = offset;
  
  while ( nbyte > 0 ) {
    got = (*(k->env))->CallIntMethod( k->env, k->input, ID.InputStream.read_n,
      	      	      	      	      k->buffer, where, nbyte);
    if ( got == -1 ) {
      jobject o;
      o = (*(k->env))->NewObject( k->env,
    		      ID.BSMScannerException.class, ID.BSMScannerException.eof);
      if ( o == NULL ) return NULL;
      (*(k->env))->Throw( k->env, (jthrowable)o);
      return NULL;
    }
    if ( (*(k->env))->ExceptionOccurred( k->env) )
      return NULL;
    where += got;
    nbyte -= got;
  }
  
  if ( k->pinned == NULL ) {
    k->pinned = (*(k->env))->GetByteArrayElements(k->env,k->buffer, &k->isCopy);
    if ( k->pinned == NULL ) {
      if ( NULL == (*(k->env))->ExceptionOccurred( k->env) )
      	(*(k->env))->Throw( k->env, k->memoryER);
      return NULL;
    }
    buffer = k->pinned + offset;
    if ( t != NULL )
      *(void **)t = k->pinned;
  }
  
  return buffer;
}
/*
 *  readstruct reads a structure of size nbyte and returns the address of the
 *  buffer into which it was read.  It returns NULL if something went wrong,
 *  whereupon the caller should also immediately return (a Java exception has
 *  probably been thrown).  readstruct is not called directly, but via the TOK
 *  and TOKTILL macros.
 */
void *
readstruct( context k, size_t nbyte)
{
  jbyte
    *buffer;

  if ( NULL == bufxtend( k, 0, nbyte, &buffer) )
    return NULL;

  return kread( k, buffer, nbyte, &buffer);
}
/*
 *  makeTok constructs a new object of the specified class (whatclass)
 *  which must be a subclass of BSMToken.  whatkind is a caller-provided
 *  jcache structure whose 'name' member is the symbolic name (a string) of the
 *  value to be stored in the 'type' field of the new token. The jcache
 *  structure also contains space to store a jfieldID (obtained by looking up
 *  the name string) and a jint (the actual 'type' value, obtained by using
 *  the fieldID).  If the fieldID is NULL, makeTok will look it up and retrieve
 *  the int value, saving both in the jcache structure.  It is expected that
 *  'whatkind' points to static storage in the caller so the lookups will be
 *  avoided next time.
 */
jobject
makeTok( context k, jclass whatclass, jcache *whatkind, jmethodID construct)
{
  jobject tok;
  if ( whatkind->kindID == NULL ) {
    whatkind->kindID = (*(k->env))->GetStaticFieldID( k->env,
      	      	      	      	 ID.BSMTokenStream.class, whatkind->name, "I");
    if ( whatkind->kindID == NULL ) return NULL;
    whatkind->kind = (*(k->env))->GetStaticIntField( k->env,
      	      	      	      	    ID.BSMTokenStream.class, whatkind->kindID);
  }
  tok = (*(k->env))->NewObject( k->env, whatclass, construct, whatkind->kind);
  if ( tok == NULL ) {
    if ( NULL == (*(k->env))->ExceptionOccurred( k->env) )
      (*(k->env))->Throw( k->env, k->memoryER);
    return NULL;
  }
  (*(k->env))->SetIntField( k->env, tok, ID.BSMToken.record, k->recordNo);
  (*(k->env))->SetObjectField( k->env, k->token, ID.BSMToken.next, tok);
  k->token = tok;
  return tok;
}

int
tokByName( context k, jcache *whatkind) {
  if ( ! k->suppressName )
    return makeTok( k, ID.BSMProto.class, whatkind, ID.BSMProto.init)
           ? JNI_OK : JNI_ERR;
  k->suppressName = 0;
  return JNI_OK;
}
#define EMITNAME(kind) if ( JNI_OK == tokByName( k, kind) ); else return;

#define JCACHE(str) static jcache BSM_jcache[] = { { NULL, 0, str } };

int
headerTok( context k, jcache *whatkind, jlong length, jshort version,
           jint emod, jlong time, jlong msec) {
  jobject tok;
  tok = makeTok( k, ID.BSMHeader.class, whatkind, ID.BSMHeader.init);
  if ( tok == NULL ) return JNI_ERR;
  (*(k->env))->SetLongField ( k->env, tok, ID.BSMHeader.length,  length);
  (*(k->env))->SetShortField( k->env, tok, ID.BSMHeader.version, version);
  (*(k->env))->SetIntField  ( k->env, tok, ID.BSMHeader.emod,    emod);
  (*(k->env))->SetLongField ( k->env, tok, ID.BSMHeader.time,    time);
  (*(k->env))->SetLongField ( k->env, tok, ID.BSMHeader.msec,    msec);
  return JNI_OK;
}

int
blockTok( context k, jcache *whatkind, jbyte *base, jsize length) {
  jobject tok;
  jbyteArray value = (*(k->env))->NewByteArray( k->env, length);
  if ( value == NULL ) {
    if ( NULL == (*(k->env))->ExceptionOccurred( k->env) )
      (*(k->env))->Throw( k->env, k->memoryER);
    return JNI_ERR;
  }
  (*(k->env))->SetByteArrayRegion( k->env, value, 0, length, base);
  tok = makeTok( k, ID.BSMBlock.class, whatkind, ID.BSMBlock.init);
  if ( tok == NULL ) return JNI_ERR;
  (*(k->env))->SetObjectField( k->env, tok, ID.BSMBlock.value, value);
  return JNI_OK;
}
#define EMITBLOCK(kind,base,length) { JCACHE(kind) \
  if ( JNI_ERR == blockTok( k, BSM_jcache, (jbyte *)base, length) ) return; }

int
stringTok( context k, jcache *whatkind, char *base) {
  register char *p = base;
  register jsize length = 0;
  while ( *(p++) != '\0' )
    ++ length;
  return blockTok( k, whatkind, (jbyte *)base, length);
}
#define EMITSTRING(kind,base) { JCACHE(kind) \
  if ( JNI_ERR == stringTok( k, BSM_jcache, base) ) return; }

int
longTok( context k, jcache *whatkind, jlong value) {
  jobject tok = makeTok( k, ID.BSMLong.class, whatkind, ID.BSMLong.init);
  if ( tok == NULL ) return JNI_ERR;
  (*(k->env))->SetLongField( k->env, tok, ID.BSMLong.value, value);
  return JNI_OK;
}
#define EMITLONG( kind, value) { JCACHE(kind) \
  if ( JNI_ERR == longTok( k, BSM_jcache, value) ) return; }

int
doubleTok( context k, jcache *whatkind, jdouble value) {
  jobject tok = makeTok( k, ID.BSMDouble.class, whatkind, ID.BSMDouble.init);
  if ( tok == NULL ) return JNI_ERR;
  (*(k->env))->SetDoubleField( k->env, tok, ID.BSMDouble.value, value);
  return JNI_OK;
}
#define EMITDOUBLE( kind, value) { JCACHE(kind) \
  if ( JNI_ERR == doubleTok( k, BSM_jcache, value) ) return; }
/*
 *  The macro TOK(handle) emits code to read a structure t of type
 *  au_<handle>_tok_t and then pass it to <handle>_fn().
 *
 *  TOKTILL(handle,member) does the same thing, but reads only the portion of
 *  the structure up to (not including) t.<member>.  The function <handle>_fn
 *  is responsible for reading the remainder of the structure (whose size,
 *  presumably, can be determined from something in the part of the structure
 *  already read); indeed <handle>_fn *must* determine the proper size and
 *  read the remainder of the structure, to preserve the invariant that the
 *  file pointer is at a token boundary when a token is to be read.
 */
#define TOK(handle) {\
  register au_##handle##_tok_t *t = readstruct( k, sizeof *t); \
  if ( t == NULL ) return; \
  handle##_fn( t, k); } break

#define TOKTILL(handle,member) {\
  typedef au_##handle##_tok_t *tp; \
  register tp t =\
  readstruct( k, ((size_t)&(((tp)0)->member))); \
  if ( t == NULL ) return; \
  handle##_fn( t, k); } break
/*
 *  The macro TOKFN(handle) declares the function <handle>_fn(t,k)
 *  which will be called whenever an au_<handle>_tok_t is read by TOK(handle)
 *  or TOKTILL(handle).  Within the function, t points to the struct that was
 *  read (partially read in the case of TOKTILL),  type->name is a char const *
 *  containing the token name "au_<handle>_tok_t", and tp is a typedef for
 *  pointer to au_<handle>_tok_t, i.e. the type of t.  The first statement of
 *  the function is produced automatically.  The function is declared with
 *  void return type.
 */
#define TOKFN(handle) static void \
  handle##_fn( au_##handle##_tok_t *t, context k) { \
  static jcache type = { NULL, 0, "au_" #handle "_tok_t" }; \
  typedef au_##handle##_tok_t *tp; \
  EMITNAME(&type)
/*
 *  When TOKTILL is used to read a token, the TOKFN is responsible for reading
 *  the rest of it.  Before doing so, the function should use the macro
 *  EXTEND(member,nbyte) to ensure that the token buffer is large enough to
 *  read nbyte bytes starting at &(t->member).
 */
#define EXTEND(member,nbyte) \
  bufxtend( k, (char *)&(t->member) - (char *)t, (nbyte), &t)

#define EMIT(type,element) { JCACHE(#element) \
  if ( JNI_ERR == type##Tok(k,BSM_jcache,ALIGNED(element)) ) return; }

#define PUNT }
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                  TOKEN PROCESSING FUNCTIONS BEGIN HERE                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
TOKFN(attr)
  EMIT(long,mode)
  EMIT(long,uid)
  EMIT(long,gid)
  EMITLONG( "fs$major", getemajor( ALIGNED(fs)))
  EMITLONG( "fs$minor", geteminor( ALIGNED(fs)))
  EMIT(long,node)
  EMITLONG( "dev$major", getemajor( ALIGNED(dev)))
  EMITLONG( "dev$minor", geteminor( ALIGNED(dev)))
}

TOKFN(exit)
  EMIT(long,status)
  EMIT(long,retval)
}

TOKFN(groups) PUNT

TOKFN(header)
  (*(k->env))->SetIntField( k->env, k->this,
      	      	      	    ID.BSMTokenStream.record, ++(k->recordNo));
  headerTok( k, BSM_eventname( ALIGNED(event)),
             ALIGNED(length), ALIGNED(version), ALIGNED(emod),
	     ALIGNED(time), ALIGNED(msec));
}

TOKFN(inaddr)
#if defined(EXPAND_IN_ADDR)
  EMITSTRING( "ia", inet_ntoa( ALIGNED(ia)))
#else
  EMITBLOCK( "ia", &(t->ia), sizeof t->ia)
#endif
}

TOKFN(ip)
  EMIT(long,version)
  EMITLONG( "ip_v", t->ip.ip_v)
  EMITLONG( "ip_hl", t->ip.ip_hl)
#if defined(EXPAND_IP_TOS)
  {
    register char *tos;
    switch ( t->ip.ip_tos )
    {
      case IPTOS_LOWDELAY   : tos = "IPTOS_LOWDELAY"; break;
      case IPTOS_THROUGHPUT : tos = "IPTOS_THROUGHPUT"; break;
      case IPTOS_RELIABILITY: tos = "IPTOS_RELIABILITY"; break;
      default: tos = NULL;
    }

    EMITSTRING( "ip_tos", tos)
  }
#else
    EMITLONG( "ip_tos", t->ip.ip_tos)
#endif
  EMITLONG( "ip_len", ALIGNED(ip.ip_len))
  EMITLONG( "ip_id",  ALIGNED(ip.ip_id))
  EMITLONG( "ip_off", ALIGNED(ip.ip_off))
  EMITLONG( "ip_ttl", t->ip.ip_ttl)
#if defined(EXPAND_IP_PROTO)
  EMITSTRING( "ip_p", getprotobynumber( t->ip.ip_p)->p_name)
#else
  EMITLONG( "ip_p", t->ip.ip_p)
#endif
  EMITLONG( "ip_sum", ALIGNED(ip.ip_sum))
#if defined(EXPAND_IN_ADDR)
  EMITSTRING( "ip_src", inet_ntoa( ALIGNED(ip.ip_src)))
  EMITSTRING( "ip_dst", inet_ntoa( ALIGNED(ip.ip_dst)))
#else
  EMITBLOCK( "ip_src", &(t->ip.ip_src), sizeof t->ip.ip_src)
  EMITBLOCK( "ip_dst", &(t->ip.ip_dst), sizeof t->ip.ip_dst)
#endif
}

TOKFN(ipc)
  EMIT(long,id)
}

TOKFN(ipc_perm)
  EMITLONG( "uid" , ALIGNED(ipc_perm.uid ))
  EMITLONG( "gid" , ALIGNED(ipc_perm.gid ))
  EMITLONG( "cuid", ALIGNED(ipc_perm.cuid))
  EMITLONG( "cgid", ALIGNED(ipc_perm.cgid))
  EMITLONG( "mode", ALIGNED(ipc_perm.mode))
  EMITLONG( "seq" , ALIGNED(ipc_perm.seq ))
  EMITLONG( "key" , ALIGNED(ipc_perm.key ))
}

TOKFN(iport)
  EMIT(long,iport)
}

TOKFN(exec_args)
  EMIT(long,count)
  {
    register void *buffer;
    register jsize size;
    register u_long
      count = ALIGNED(count);
    while ( count-- ) {
      size = k->bufferSize;
      for ( buffer = (void *)t ;; buffer = (char *)buffer + 1 ) {
        if ( size <= (char *)buffer - (char *)t ) {
	  buffer = bufxtend( k, (char *)buffer - (char *)t, 1, &t);
	  if ( buffer == NULL ) return;
	  size = k->bufferSize;
	}
	KREAD( buffer, 1);
	if ( '\0' == *(char *)buffer )
	  break;
      }
      EMITBLOCK( "env_args", t, (char *)buffer - (char *)t)
    }
  }
}

TOKFN(exec_env)
  EMIT(long,count)
  {
    register void *buffer;
    register jsize size;
    register u_long
      count = ALIGNED(count);
    while ( count-- ) {
      size = k->bufferSize;
      for ( buffer = (void *)t ;; buffer = (char *)buffer + 1 ) {
        if ( size <= (char *)buffer - (char *)t ) {
	  buffer = bufxtend( k, (char *)buffer - (char *)t, 1, &t);
	  if ( buffer == NULL ) return;
	  size = k->bufferSize;
	}
	KREAD( buffer, 1);
	if ( '\0' == *(char *)buffer )
	  break;
      }
      EMITBLOCK( "env_args", t, (char *)buffer - (char *)t)
    }
  }
}

TOKFN(newgroups)
  EMIT(long,count)
  {
    register int
      idx = 0;
    register u_short
      count = ALIGNED(count);
    register u_short
      len = count * sizeof t->groups;
    register char
      *buf;
    buf = EXTEND( groups, len);
    KREAD( buf, len);
    while( 0 < count-- )
      EMITLONG( "groups", t->groups[idx++])
  }
}

TOKFN(proc)
  EMIT(long,auid)
  EMIT(long,euid)
  EMIT(long,egid)
  EMIT(long,ruid)
  EMIT(long,rgid)
  EMIT(long,pid)
  EMIT(long,sid)
  EMITLONG( "tid$port$major", getemajor( ALIGNED(tid.port)))
  EMITLONG( "tid$port$minor", geteminor( ALIGNED(tid.port)))
#if defined(EXPAND_IN_ADDR)
  EMITSTRING( "tid$machine",
      	      	 inet_ntoa( *(struct in_addr *)&ALIGNED(tid.machine)))
#else
  EMITBLOCK( "tid$machine", &(t->tid.machine), sizeof t->tid.machine)
#endif
}

TOKFN(ret)
  EMIT(long,error)    /* && should be the manifest constant; a project! */
  EMIT(long,retval)
}

TOKFN(seq)
  EMIT(long,num)
}

TOKFN(socket)
  EMIT(long,type)	/* && expand this */
  EMIT(long,lport)
#if defined(EXPAND_IN_ADDR)
  EMITSTRING( "laddr", inet_ntoa( ALIGNED(laddr)))
#else
  EMITBLOCK( "laddr", &(t->laddr), sizeof t->laddr)
#endif
return;		/* socket-inet does not include next two members! */
  EMIT(long,fport)
  EMITSTRING( "faddr", inet_ntoa( ALIGNED(faddr)))
}

TOKFN(subj)
  EMIT(long,auid)
  EMIT(long,euid)
  EMIT(long,egid)
  EMIT(long,ruid)
  EMIT(long,rgid)
  EMIT(long,pid)
  EMIT(long,sid)
  EMITLONG( "tid$port$major", getemajor( ALIGNED(tid.port)))
  EMITLONG( "tid$port$minor", geteminor( ALIGNED(tid.port)))
#if defined(EXPAND_IN_ADDR)
  EMITSTRING( "tid$machine",
      	      	 inet_ntoa( *(struct in_addr *)&ALIGNED(tid.machine)))
#else
  EMITBLOCK( "tid$machine", &(t->tid.machine), sizeof t->tid.machine)
#endif
}

TOKFN(server)
  EMIT(long,auid)
  EMIT(long,euid)
  EMIT(long,ruid)
  EMIT(long,egid)
  EMIT(long,pid)
}

TOKFN(data)
  {
    register char
      *bufp,
      *xbufp,
      *xbuf,
      *pf,
      *sz,
      *buf;
    register size_t
      bytesper;
    switch ( ALIGNED(pfmt) )
    {
      case AUP_BINARY : pf = "AUP_BINARY" ; break;
      case AUP_OCTAL  : pf = "AUP_OCTAL"  ; break;
      case AUP_DECIMAL: pf = "AUP_DECIMAL"; break;
      case AUP_HEX    : pf = "AUP_HEX"    ; break;
      case AUP_STRING : pf = "AUP_STRING" ; break;
               default: pf = NULL;
    }
    switch ( ALIGNED(size) )
    {
      case AUR_BYTE :	/* == AUR_CHAR */
	bytesper = sizeof (char);
	sz = "AUR_BYTE"; break;
      case AUR_SHORT:
	bytesper = sizeof (short);
	sz = "AUR_SHORT"; break;
      case AUR_INT  :
	bytesper = sizeof (int);
	sz = "AUR_INT"; break;
      case AUR_LONG :
	bytesper = sizeof (long);
	sz = "AUR_LONG"; break;
      default:
	bytesper = 0;
	sz = NULL;
    }
    EMITSTRING( "pfmt", pf)
    EMITSTRING( "size", sz)
    EMIT(long,number)
    buf = EXTEND(data, ALIGNED(number) * bytesper);
    KREAD( buf, bytesper * ALIGNED(number));
    EMITBLOCK( "data", buf, bytesper * ALIGNED(number))
  }
}

TOKFN(file)
  EMIT(long,time)
  EMIT(long,msec)
  EMIT(long,length)
  {
    register char
      *buf;
    register u_short
      len = ALIGNED(length);

    buf = EXTEND( fname, len);
    KREAD( buf, len);
    EMITBLOCK( "fname", buf, len - 1)
  }
}

TOKFN(text)
  EMIT(long,length)
  {
    register char
      *buf;
    register u_short
      len = ALIGNED(length);

    buf = EXTEND( data, len);
    KREAD( buf, len);
    EMITBLOCK( "data", buf, len - 1)
  }
}

TOKFN(path)
  EMIT(long,length)
  {
    register char
      *buf;
    register u_short
      len = ALIGNED(length);

    buf = EXTEND( name, len);
    KREAD( buf, len);
    EMITBLOCK( "name", buf, len - 1)
  }
}

TOKFN(arg)
  EMIT(long,num)
  EMIT(long,val)
  EMIT(long,length)
  {
    register char
      *buf;
    register u_short
      len = ALIGNED(length);
    
    buf = EXTEND( data, len);
    KREAD( buf, len);
    EMITBLOCK( "data", buf, len - 1)
  }
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    getNextToken ROUTINE (AND MAIN TOKEN-TYPE switch() STATEMENT) BELOW       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*
 * Class:     BSMTokenStream
 * Method:    getNextToken
 */
JNIEXPORT void FQJNICALL(getNextToken)
  (JNIEnv *env, jobject this, jobject input, jobject token, jbyteArray buf,
   jthrowable oome)
{
  register int
    tok_id;

#pragma align 8 (ctx)
  struct context ctx; /*first member is alignhack (see ALIGN comments above)*/
  context k = &ctx;
  
  ctx.env      = env;
  ctx.this     = this;
  ctx.recordNo = (*env)->GetIntField( env, this, ID.BSMTokenStream.record);
  ctx.input    = input;
  ctx.token    = token;
  ctx.buffer   = buf;
  ctx.pinned   = NULL;
  ctx.memoryER = oome;
  ctx.suppressName = 0;
  
  tok_id = (*env)->CallIntMethod( env, input, ID.InputStream.read_1);
  if ( tok_id == -1 )
    return;

  switch ( tok_id )   	/* The big switch */
  {
    case AUT_HEADER:  	ctx.suppressName = 1; TOK(header);
    case AUT_ATTR:	TOK(attr);
    case AUT_EXIT:	TOK(exit);
    case AUT_GROUPS:	TOK(groups);
    case AUT_IN_ADDR:	TOK(inaddr);
    case AUT_IP:	TOK(ip);
    case AUT_IPC:	TOK(ipc);
    case AUT_IPC_PERM:	TOK(ipc_perm);
    case AUT_IPORT:	TOK(iport);
    case AUT_PROCESS:	TOK(proc);
    case AUT_RETURN:	TOK(ret);
    case AUT_SEQ:	TOK(seq);
    case AUT_SOCKET:	TOKTILL(socket,fport); /* file !match declaration */
    case AUT_SUBJECT:	TOK(subj);
    case AUT_SERVER:	TOK(server);
    case AUT_OTHER_FILE:TOKTILL(file,fname);
    case AUT_TEXT:	TOKTILL(text,data);
    case AUT_DATA:	TOKTILL(data,data);
    case AUT_PATH:	TOKTILL(path,name);
    case AUT_ARG:	TOKTILL(arg,data);
    case AUT_NEWGROUPS: TOKTILL(newgroups,groups);
    case AUT_EXEC_ARGS: TOKTILL(exec_args,env_args);
    case AUT_EXEC_ENV:	TOKTILL(exec_env,env_args);

    case AUT_INVALID:
    case AUT_OHEADER:
    case AUT_TRAILER:
    case AUT_TRAILER_MAGIC:

    case AUT_OPAQUE:

    case AUT_LABEL:
    case AUT_ILABEL:
    case AUT_SLABEL:
    case AUT_CLEAR:
    case AUT_PRIV:
    case AUT_UPRIV:
    case AUT_LIAISON:
    case AUT_XATOM:
    case AUT_XOBJ:
    case AUT_XPROTO:
    case AUT_XSELECT:
    case AUT_CMD:
    case AUT_ACL:
    default: {
      jobject o;
      o = (*env)->NewObject( env,
      	        ID.BSMScannerException.class, ID.BSMScannerException.mysterytok,
      	      	(jbyte)tok_id);
      if ( o == NULL ) return;
      (*env)->Throw( env, (jthrowable)o);
    }
  }
}

#include <bsm/libbsm.h>

/*
 * Class:     BSMTokenStream
 * Method:    setEventFileName
 * Signature: ([B)V
 */
JNIEXPORT void FQJNICALL(setEventFileName)
  (JNIEnv *env, jclass me, jbyteArray efn)
{
  jbyte *cp = (*env)->GetByteArrayElements( env, efn, NULL);
  if ( cp == NULL ) return;
  setaueventfile( (char *)cp);
  (*env)->ReleaseByteArrayElements( env, efn, cp, JNI_ABORT);
}
