/* commands.c -- handle commands from stdin				*/
/*
 * Copyright (c) 1994  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */
#include "xdatplot.h"
#include "rfiles.h"
#include <fcntl.h>
#define	NONBLOCK	FNONBIO		/* non-blocking I/O flag	*/
#define	BLOCKERR	EAGAIN		/* error if read would block	*/

typedef	struct	COMMAND {
    String	c_cmd;
    Bool	(*c_cb)();
}		COMMAND;

#ifdef	__STDC__
static	void	cmd_in(XtPointer, int *, XtInputId *);
static	int	readline(void);
static	String	cgets(String, int, FILE *);
static	void	parse_line(String, int *, COMMAND **, String *);
static	void	respond(int, int, String);
static	Bool	error_cmd(String, String *);
static	Bool	quit_cmd(String, String *);
static	Bool	dissociate_cmd(String, String *);
static	Bool	new_file_cmd(String, String *);
static	Bool	resource_cmd(String, String *);
static	Bool	scroll_t_cmd(String, String *);
static	Bool	size_t_cmd(String, String *);
static	Bool	zoom_t_cmd(String, String *);
static	Bool	size_zoom_scroll_t_cmd(String, String *);
static	Bool	set_cursor_cmd(String, String *);
static	Bool	clear_cursor_cmd(String, String *);
static	Bool	region_cmd(String, String *);
#else	/* __STDC__ */
static	void	cmd_in();
static	int	readline();
static	String	cgets();
static	void	parse_line();
static	void	respond();
static	Bool	error_cmd();
static	Bool	quit_cmd();
static	Bool	dissociate_cmd();
static	Bool	new_file_cmd();
static	Bool	resource_cmd();
static	Bool	scroll_t_cmd();
static	Bool	size_t_cmd();
static	Bool	zoom_t_cmd();
static	Bool	size_zoom_scroll_t_cmd();
static	Bool	set_cursor_cmd();
static	Bool	clear_cursor_cmd();
static	Bool	region_cmd();
#endif	/* __STDC__ */

static	COMMAND	cmds[] = {
    {"Quit", quit_cmd},
    {"Dissociate", dissociate_cmd},
    {"NewFile", new_file_cmd},
    {"Resource", resource_cmd},
    {"ScrollT", scroll_t_cmd},
    {"SizeT", size_t_cmd},
    {"ZoomT", zoom_t_cmd},
    {"SizeZoomScrollT", size_zoom_scroll_t_cmd},
    {"SetCursor", set_cursor_cmd},
    {"ClearCursor", clear_cursor_cmd},
    {"Region", region_cmd},
};
#define	NCMDS	(XtNumber(cmds))

static	XtInputId	cmdi = NO_INPUTID; /* for command input		*/
static	char		*lbuf = NULL;	/* line buffer			*/
static	int		bufsiz = 0;	/* its current size		*/
static	String		err_msg = NULL;
static	Bool		response = False; /* True if response needed	*/
static	int		cid = EOF;	/* current command ID		*/

/* command_input -- initialize command input				*/
void
command_input()
{
    int			flags;

    if (!app_data.read_commands) return;
    if (NO_INPUTID != cmdi) error("command_input: can't happen");
    cmdi = AppAddInput(app, 0, XtInputReadMask, cmd_in, NULL);
    flags = fcntl(0, F_GETFL, 0);
    fcntl(0, F_SETFL, flags | NONBLOCK); /* set non-blocking input	*/
}

void
end_command_input()
{
    int			flags;

    if (!app_data.read_commands) return;
    if (NO_INPUTID != cmdi) XtRemoveInput(cmdi);
    cmdi = NO_INPUTID;
    flags = fcntl(0, F_GETFL, 0);
    fcntl(0, F_SETFL, flags & ~NONBLOCK);
    close(0);
}

#define	MINBUF	64

void
cmd_in(data, source, id)
XtPointer	data;
int		*source;
XtInputId	*id;
{
    int			ret;
    COMMAND		*cmd;
    String		args;
    String		msg;

    while(0 < (ret = readline())) {
	dprintf(lbuf);
	parse_line(lbuf, &cid, &cmd, &args);
	if (NULL == cmd) return;
	msg = NULL;
	response = True;
	ret = (*cmd->c_cb)(args, &msg);
	if (response) respond(ret, cid, msg);
    }
    if (EOF == ret) {
	dprintf("EOF on stdin\n");
	XtRemoveInput(cmdi);
	cmdi = NO_INPUTID;
	exit_program();
    }
}

static int
readline()
{
    int			len = 0;

    if (NULL == lbuf) {
	lbuf = (String) XtMalloc(MINBUF);
	bufsiz = MINBUF;
    }
    lbuf[0] = '\0';
    errno = 0;
    if (NULL == cgets(lbuf, bufsiz, stdin)) {
	return((BLOCKERR == errno) ? 0 : EOF);
    }
    while(NULL == strchr(lbuf, '\n')) {
	len = strlen(lbuf);
	if (len >= bufsiz-1) {
	    bufsiz += MINBUF;
	    lbuf = (String) Realloc(lbuf, bufsiz);
	}
	if (NULL == cgets(lbuf + len, bufsiz - len, stdin)) break;
    }
    if (NULL == strchr(lbuf, '\n')) {
	lbuf[len] = '\n';
	lbuf[len+1] = '\0';
    }
    return(strlen(lbuf));
}

/* cgets -- like fgets, but handle line continuation			*/
static String
cgets(buf, n, f)
String	buf;
int	n;
FILE	*f;
{
    register String	s;

    if (NULL == fgets(buf, n, f)) return(NULL);
    if (
	(NULL != (s = STRCHR(buf, '\n'))) &&
	('\\' == s[-1])
    ) s[-1] = '\0';
    return(buf);
}

static void
parse_line(line, idp, cmdp, argsp)
String	line;
int	*idp;
COMMAND	**cmdp;
String	*argsp;
{
    register int	i;
    String		s;
    static COMMAND	ecmd = {"Error", error_cmd};

    *idp = EOF;
    *cmdp = NULL;
    *argsp = NULL;
    while(isspace(*line)) line++;
    if ('\0' == *line) return;
    if (isdigit(*line)) {
	*idp = strtol(line, &s, 10);
	if ((s == line) || (*idp < 0)) {
	    err_msg = "Invalid ID";
	    *idp = EOF;
	    *cmdp = &ecmd;
	    return;
	}
	line = s;
	while(isspace(*line)) line++;
    }
    for(s=line; ('\0' != *s) && !isspace(*s); s++);
    for(i=0; i<NCMDS; i++) {
	if (0 == strncmp(cmds[i].c_cmd, line, s-line)) break;
    }
    if (i >= NCMDS) {
	err_msg = "Unknown command";
	*cmdp = &ecmd;
	return;
    }
    while(isspace(*s)) s++;
    *cmdp = &cmds[i];
    *argsp = s;
    if (NULL != (s = STRCHR(s, '\n'))) *s = '\0';
}

static void
respond(ret, cid, msg)
Bool	ret;
int	cid;
String	msg;
{
    if (EOF != cid) printf("%d ", cid);
    printf("%s", (ret ? "ERR" : "OK"));
    if (NULL != msg) printf(" %s", msg);
    putchar('\n');
    fflush(stdout);
    response = False;
}

static Bool
error_cmd(args, msgp)
String	args;
String	*msgp;
{
    *msgp = err_msg;
    return(True);
}

static Bool
quit_cmd(args, msgp)
String	args;
String	*msgp;
{
    if ('\0' != *args) {
	*msgp = "no arguments allowed";
	return(True);
    }
    respond(False, cid, NULL);
    exit_program();
    return(False);			/* not reached			*/
}

static Bool
dissociate_cmd(args, msgp)
String	args;
String	*msgp;
{
    if ('\0' != *args) {
	*msgp = "no arguments allowed";
	return(True);
    }
    respond(False, cid, NULL);
    XtRemoveInput(cmdi);
    cmdi = NO_INPUTID;
    fclose(stdin);
    fclose(stdout);
    return(False);
}

static Bool
new_file_cmd(args, msgp)
String	args;
String	*msgp;
{
    FILTER		*olddst = destination_filter(&DF_FIL);

    *msgp = new_file(args, False);
    return(NULL != *msgp);
}

static Bool
resource_cmd(args, msgp)
String	args;
String	*msgp;
{
    register String	s = args;
    String		t = args;
    register int	c;
    String		wid;
    String		res;
    String		val;
    Widget		widget;

    while('\0' != (c = *t++)) {
	if ('\\' == c) c = escape(&t);
	if (EOF != c) *s++ = c;
    }
    *s = '\0';
    if (
	(parse_resource_line(args, &wid, &res, &val)) ||
	(NULL == wid)
    ) {
	*msgp = "Resource parse error";
	return(True);
    }
    if (strlen(wid) == strspn(wid, "*.")) {
	widget = toplevel;
    }
    else {
	if (NO_WIDGET == (widget = XtNameToWidget(toplevel, wid))) {
	    *msgp = "Unknown widget";
	    return(True);
	}
    }
    if (
	(widget != toplevel) ||
	(set_application_resource(res, val))
    ) {
	XtVaSetValues(widget,
	    XtVaTypedArg, res, XtRString,
	    val, strlen(val)+1,
	NULL);
    }
    return(False);
}

static Bool
scroll_t_cmd(args, msgp)
String	args;
String	*msgp;
{
    double		time;
    String		s;
    TIME		ntl;
    TIME		ntr;

    time = strtod(args, &s);
    if ((s == args) || ('\0' != *s)) {
	*msgp = "invalid number";
	return(True);
    }
    ntl = Map_t_to_TIME(time);
    ntr = ntl + TR - TL;
    FORCE_RANGE(ntl, ntr, TMIN, TMAX);
    scroll_plot(ntl);
    adjust_sb();
    return(False);
}

static Bool
size_t_cmd(args, msgp)
String	args;
String	*msgp;
{
    Dimension		w;
    String		s;

    w = strtol(args, &s, 0);
    if ((s == args) || ('\0' != *s)) {
	*msgp = "invalid number";
	return(True);
    }
    XtVaSetValues(toplevel, XtNallowShellResize, True, NULL);
    XtVaSetValues(plot, XmNwidth, w, NULL);
    XtVaSetValues(toplevel, XtNallowShellResize, False, NULL);
    return(False);
}

static Bool
zoom_t_cmd(args, msgp)
String	args;
String	*msgp;
{
    String		s;
    double		zoom;
    double		delta;

    zoom = strtod(args, &s);
    if ((s == args) || ('\0' != *s)) {
	*msgp = "invalid number";
	return(True);
    }
    if (0.0 >= zoom) {
	*msgp = "invalid zoom";
	return(True);
    }
    zoom /= t_to_Dim_sf;
    delta = (zoom - 1.0) * (app_data.tr - app_data.tl);
    if (delta > 0.5) t_zoom_in(zoom);
    else if (delta < -0.5) t_zoom_out(1.0/zoom);
    else {}
    return(False);
}

#define	COPYARG \
    {						\
	String		t;			\
						\
	while(isspace(*s)) s++;			\
	for(					\
	    t=abuf;				\
	    ('\0' != *s) && !isspace(*s);	\
	    s++, t++				\
	) {					\
	    if (t - abuf >= LLEN-1) break;	\
	    *t = *s;				\
	}					\
	*t = '\0';				\
    }

static Bool
size_zoom_scroll_t_cmd(args, msgp)
String	args;
String	*msgp;
{
    char		abuf[LLEN];
    String		s = args;

    COPYARG;
    if (size_t_cmd(abuf, msgp)) return(True);
    COPYARG;
    if (zoom_t_cmd(abuf, msgp)) return(True);
    COPYARG;
    if (scroll_t_cmd(abuf, msgp)) return(True);
    return(False);
}

static Bool
set_cursor_cmd(args, msgp)
String	args;
String	*msgp;
{
    double		time;
    String		s;

    time = strtod(args, &s);
    if ((s == args) || ('\0' != *s)) {
	*msgp = "invalid number";
	return(True);
    }
    set_cursor(Map_t_to_TIME(time));
    return(False);
}

static Bool
clear_cursor_cmd(args, msgp)
String	args;
String	*msgp;
{
    if ('\0' != *args) {
	*msgp = "no arguments allowed";
	return(True);
    }
    reset_cursor();
    return(False);
}

static Bool
region_cmd(args, msgp)
String	args;
String	*msgp;
{
    double		time;
    String		s;

    time = strtod(args, &s);
    if ((s == args) || ('\0' != *s)) {
	*msgp = "invalid number";
	return(True);
    }
    set_cursor_end(Map_t_to_TIME(time));
    return(False);
}
