/* filter.c -- data filtering package					*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */
#include "filter.h"
#include "template.h"

#define	DF_BUFSIZ	(0x10000)
#define	Nfree(p)	((NULL == (p)) || (free(p), (p) = NULL))

typedef	struct	TP_PRIV {		/* private data for template	*/
    TEMPL	tpl;			/* template			*/
    FILTER_VAL	*vals;			/* cooked values		*/
    int		nvals;			/* # values			*/
    int		tzero;			/* time offset			*/
    double	nfac;			/* normalization factor		*/
    FILTER_VAL	nmin;			/* normalized min		*/
    FILTER_VAL	nmax;			/* normalized max		*/
}		TP_PRIV;

typedef	struct	CV_PRIV {		/* private data for convolution	*/
    TEMPL	tpl;			/* template			*/
    FILTER_VAL	*vals;			/* cooked values		*/
    int		nvals;			/* # values			*/
    int		tzero;			/* time offset			*/
}		CV_PRIV;

#ifdef	__STDC__
caddr_t		estrdup(caddr_t);
static	void	kill_cache(FILTER_CACHE *);
static	void	clear_cache(FILTER_CACHE *);
static	FILTER_VAL empty(FILTER *, TIME);
static	Bool	do_empty();
static	FILTER_VAL df(FILTER *, TIME);
static	Bool	make_df(FILTER_TYPE, DF_DATA *, FILTER *);
static	Bool	chg_df(FILTER *, DF_DATA *);
static	Bool	nsrc_df(FILTER *);
static	Bool	kill_df(FILTER *);
static	void	df_data_free(DF_DATA *);
static	FILTER_VAL hp(FILTER *, TIME);
static	Bool	make_hp_tau(FILTER_TYPE, HP_TAU_DATA *, FILTER *);
static	Bool	make_hp_f(FILTER_TYPE, HP_F_DATA *, FILTER *);
static	Bool	chg_hp_tau(FILTER *, HP_TAU_DATA *);
static	Bool	chg_hp_f(FILTER *, HP_F_DATA *);
static	FILTER_VAL lp(FILTER *, TIME);
static	void	lhp_seed(FILTER *, TIME, TIME *, FILTER_VAL *);
static	Bool	make_lp_tau(FILTER_TYPE, LP_TAU_DATA *, FILTER *);
static	Bool	make_lp_f(FILTER_TYPE, LP_F_DATA *, FILTER *);
static	Bool	chg_lp_tau(FILTER *, LP_TAU_DATA *);
static	Bool	chg_lp_f(FILTER *, LP_F_DATA *);
static	Bool	nsrc_generic(FILTER *);
static	Bool	kill_generic(FILTER *);
static	void	shift_cache(FILTER *, TIME, FILTER_VAL **, TIME *,
			    TIME *, TIME *);
static	void	clear_caches(FILTER *);
static	FILTER_VAL mm(FILTER *, TIME);
static	Bool	make_mm(FILTER_TYPE, MM_DATA *, FILTER *);
static	Bool	chg_mm(FILTER *, MM_DATA *);
static	Bool	nsrc_mm(FILTER *);
static	FILTER_VAL pk(FILTER *, TIME);
static	Bool	make_pk(FILTER_TYPE, PK_DATA *, FILTER *);
static	Bool	chg_pk(FILTER *, PK_DATA *);
static	Bool	nsrc_pk(FILTER *);
static	FILTER_VAL scp(FILTER *, TIME);	/* scale with positive factor	*/
static	FILTER_VAL scn(FILTER *, TIME);	/* scale with negative factor	*/
static	Bool	make_sc(FILTER_TYPE, SC_DATA *, FILTER *);
static	Bool	chg_sc(FILTER *, SC_DATA *);
static	Bool	nsrc_sc(FILTER *);
static	FILTER_VAL tp(FILTER *, TIME);
static	Bool	make_tp(FILTER_TYPE, TP_DATA *, FILTER *);
static	Bool	chg_tp(FILTER *, TP_DATA *);
static	Bool	nsrc_tp(FILTER *);
static	Bool	kill_tp(FILTER *);
static	void	destroy_tp_priv(TP_PRIV *);
static	FILTER_VAL cv(FILTER *, TIME);
static	Bool	make_cv(FILTER_TYPE, CV_DATA *, FILTER *);
static	Bool	chg_cv(FILTER *, CV_DATA *);
static	Bool	nsrc_cv(FILTER *);
static	Bool	kill_cv(FILTER *);
static	void	destroy_cv_priv(CV_PRIV *);
#else	/* __STDC__ */
caddr_t		estrdup();
static	void	kill_cache();
static	void	clear_cache();
static	FILTER_VAL empty(FILTER *, TIME);
static	Bool	do_empty();
static	FILTER_VAL df();
static	Bool	make_df();
static	Bool	chg_df();
static	Bool	nsrc_df()
static	Bool	kill_df();
static	void	df_data_free();
static	FILTER_VAL hp();
static	Bool	make_hp_tau();
static	Bool	make_hp_f();
static	Bool	chg_hp_tau();
static	Bool	chg_hp_f();
static	FILTER_VAL lp();
static	void	lhp_seed();
static	Bool	make_lp_tau();
static	Bool	make_lp_f();
static	Bool	chg_lp_tau();
static	Bool	chg_lp_f();
static	Bool	nsrc_generic();
static	Bool	kill_generic();
static	void	shift_cache();
static	void	clear_caches();
static	FILTER_VAL mm();
static	Bool	make_mm();
static	Bool	chg_mm();
static	Bool	nsrc_mm();
static	FILTER_VAL pk();
static	Bool	make_pk();
static	Bool	chg_pk();
static	Bool	nsrc_pk();
static	FILTER_VAL scp();		/* scale with positive factor	*/
static	FILTER_VAL scn();		/* scale with negative factor	*/
static	Bool	make_sc();
static	Bool	chg_sc();
static	Bool	nsrc_sc();
static	FILTER_VAL tp();
static	Bool	make_tp();
static	Bool	chg_tp();
static	Bool	nsrc_tp();
static	Bool	kill_tp();
static	void	destroy_tp_priv();
static	FILTER_VAL cv();
static	Bool	make_cv();
static	Bool	chg_cv();
static	Bool	nsrc_cv();
static	Bool	kill_cv();
static	void	destroy_cv_priv();
#endif	/* __STDC__ */

static	FILTER	fils[F_NUMBER] = {	/* empty filters of all types	*/
    {					/* The empty filter		*/
	NULL, NULL, F_EMPTY, "Empty", NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
        {NULL, 0, 0, 1, 0, NULL},
        {NULL, 0, 0, 1, 0, NULL},
        empty, do_empty, do_empty, do_empty, kill_generic
    },
    {					/* DF = datafile filter		*/
	NULL, NULL, F_DF, NULL, NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
	{NULL, 0, 0, 1, 0, NULL},
	{NULL, 0, 0, 1, 0, NULL},
	df, make_df, chg_df, nsrc_df, kill_df
    },
    {					/* HIPASS_TAU = hi-pass filter	*/
	NULL, NULL, F_HIPASS_TAU, NULL, NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
	{NULL, 0, 0, 1, 0, NULL},
	{NULL, 0, 0, 1, 0, NULL},
	hp, make_hp_tau, chg_hp_tau, nsrc_generic, kill_generic
    },
    {					/* HIPASS_f = hi-pass filter	*/
	NULL, NULL, F_HIPASS_F, NULL, NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
	{NULL, 0, 0, 1, 0, NULL},
	{NULL, 0, 0, 1, 0, NULL},
	hp, make_hp_f, chg_hp_f, nsrc_generic, kill_generic
    },
    {					/* LOPASS_TAU = lo-pass		*/
	NULL, NULL, F_LOPASS_TAU, NULL, NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
	{NULL, 0, 0, 1, 0, NULL},
	{NULL, 0, 0, 1, 0, NULL},
	lp, make_lp_tau, chg_lp_tau, nsrc_generic, kill_generic
    },
    {					/* LOPASS_F = lo-pass		*/
	NULL, NULL, F_LOPASS_F, NULL, NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
	{NULL, 0, 0, 1, 0, NULL},
	{NULL, 0, 0, 1, 0, NULL},
	lp, make_lp_f, chg_lp_f, nsrc_generic, kill_generic
    },
    {					/* MINIMAX = minimax compress	*/
	NULL, NULL, F_MINIMAX, NULL, NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
	{NULL, 0, 0, 1, 0, NULL},
	{NULL, 0, 0, 1, 0, NULL},
	mm, make_mm, chg_mm, nsrc_mm, kill_generic
    },
    {					/* SCALE			*/
	NULL, NULL, F_SCALE, NULL, NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
	{NULL, 0, 0, 1, 0, NULL},
	{NULL, 0, 0, 1, 0, NULL},
	scp, make_sc, chg_sc, nsrc_sc, kill_generic
    },
    {					/* PICK = pick spaced values	*/
	NULL, NULL, F_PICK, NULL, NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
	{NULL, 0, 0, 1, 0, NULL},
	{NULL, 0, 0, 1, 0, NULL},
	pk, make_pk, chg_pk, nsrc_pk, kill_generic
    },
    {					/* TEMPL = match template	*/
	NULL, NULL, F_TEMPL, NULL, NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
	{NULL, 0, 0, 1, 0, NULL},
	{NULL, 0, 0, 1, 0, NULL},
	tp, make_tp, chg_tp, nsrc_tp, kill_tp
    },
    {					/* CNVLV = convolution		*/
	NULL, NULL, F_CNVLV, NULL, NULL,
	0, 0, 0.0, 0.0, 1.0, 1.0,
	NULL,
	{NULL, 0, 0, 1, 0, NULL},
	{NULL, 0, 0, 1, 0, NULL},
	cv, make_cv, chg_cv, nsrc_cv, kill_cv
    }
};

/*
 * publicly accessible filter names
 */
const	char	*filter_names[F_NUMBER] = {
    "Empty", "Data File",
    "Hi Pass (tau specified)", "Hi Pass (frequency specified)",
    "Lo Pass (tau specified)", "Lo Pass (frequency specified)",
    "Minimax compression", "Scale", "Pick", "Template match",
    "Convolution"
};

Bool
make_filter(type, data, fp)
FILTER_TYPE	type;
FILTER_DATA	*data;
FILTER		*fp;
{
    if ((0 > type) || (F_NUMBER <= type)) return(TRUE);
    bcopy(&fils[type], fp, sizeof(FILTER));
    fp->f_fmax = infinity();
    fp->f_fmin = -infinity();
    return((*fp->f_make)(type, data, fp));
}

static FILTER_VAL
empty(fp, t)
FILTER		*fp;
TIME		t;
{
    return(quiet_nan(0));
}

static Bool
do_empty()
{
    return(FALSE);
}

/* connect_filter -- put a new filter between two others
 *
 * FILTER	fps, fpd;
 * FILTER	fpm;
 * if (connect_filters(&fps, &fpm, &fpd) error...
 *
 * On call the output of fps must be connected to the input of fpd.
 * (One of the two arguments may be NULL if the other isn't currently
 * writing to or reading from a filter, respectively.  If both are
 * NULL connect_filters does nothing.)  fpm must be disconnected from
 * all other filters on call.  (It may be returned from make_filter,
 * or may have been explicitly disconnected with delete_filter.)
 * On return, fpm is connected between fps and fpd: fps -> fpm -> fpd.
 */
Bool
connect_filters(fps, fpm, fpd)
FILTER		*fps;
FILTER		*fpm;
FILTER		*fpd;
{
    if ((NULL != fpm->f_src) || (NULL != fpm->f_dst)) return(TRUE);
    if (NULL != fps) {
	if (fpd != fps->f_dst) return(TRUE);
	fpm->f_src = fps;
	fps->f_dst = fpm;
    }
    if (NULL != fpd) {
	if (fps != fpd->f_src) return(TRUE);
	fpm->f_dst = fpd;
	fpd->f_src = fpm;
    }
    clear_caches(fpm);
    return(FALSE);
}

/* delete_filter -- disconnect filter from up and downstream stuff	*/
Bool
delete_filter(fp)
FILTER		*fp;
{
    FILTER		*gp;

    if (NULL == fp) return(TRUE);
    gp = fp->f_dst;
    if (NULL != fp->f_src) fp->f_src->f_dst = fp->f_dst;
    if (NULL != fp->f_dst) fp->f_dst->f_src = fp->f_src;
    fp->f_src = NULL;
    fp->f_dst = NULL;
    clear_caches(fp);
    clear_caches(gp);
    return(FALSE);
}

/* disconnect_filter -- disconnect filter from downstream stuff		*/
Bool
disconnect_filter(fp)
FILTER		*fp;
{
    FILTER		*gp;

    if (NULL == fp) return(TRUE);
    if (NULL == (gp = fp->f_dst)) return(FALSE);
    fp->f_dst = NULL;
    gp->f_src = NULL;
    clear_caches(gp);
    return(FALSE);
}

Bool
force_filter_tmul(fp, tmul)
FILTER		*fp;
double		tmul;
{
    fp->f_tmul = tmul;
    clear_caches(fp);
}

Bool
force_filter_fmul(fp, fmul)
FILTER		*fp;
double		fmul;
{
    fp->f_fmul = fmul;
    clear_caches(fp);
}

/* clear_caches -- invalidate downstream caches				*/
static void
clear_caches(fp)
FILTER		*fp;
{
    for(; NULL != fp; fp = fp->f_dst) {
	(*fp->f_nsrc)(fp);
    }
}

/* nsrc_generic -- clear caches and propagate bounds			*/
static Bool
nsrc_generic(fp)
FILTER		*fp;
{
    if ((NULL != fp->f_cv.f_nxt) || (1 != fp->f_cv.f_sp))
        error("nsrc_generic: can't happen");
    fp->f_cv.f_t0 = 0;
    fp->f_cv.f_n = 0;
    kill_cache(&fp->f_cm);
    if (NULL == fp->f_src) {
	fp->f_tmin = fp->f_tmax = 0;
	fp->f_fmax = infinity();
	fp->f_fmin = -infinity();
	fp->f_tmul = fp->f_fmul = 1.0;
    }
    else {
	fp->f_tmin = fp->f_src->f_tmin;
	fp->f_tmax = fp->f_src->f_tmax;
	fp->f_fmin = fp->f_src->f_fmin;
	fp->f_fmax = fp->f_src->f_fmax;
	fp->f_tmul = fp->f_src->f_tmul;
	fp->f_fmul = fp->f_src->f_fmul;
    }
    return(FALSE);
}

static void
kill_cache(cp)
FILTER_CACHE	*cp;
{
    FILTER_CACHE	*dp;

    if (NULL == cp) return;
    clear_cache(cp);
    dp = cp->f_nxt;
    cp->f_nxt = NULL;
    while(NULL != dp) {
	cp = dp->f_nxt;
	clear_cache(dp);
	free(dp);
	dp = cp;
    }
}

static void
clear_cache(cp)
FILTER_CACHE	*cp;
{
    if (NULL != cp->f_vals) free(cp->f_vals);
    cp->f_vals = NULL;
    cp->f_t0 = 0;
    cp->f_n = 0;
    cp->f_sp = 1;
    cp->f_nv = 0;
}

/*
 * This is where the template filter spends its days.  Anything you
 * can do to optimize it will pay.
 */
INLINE double
dot_product(v1, v2, n)
double	*v1;
double	*v2;
int	n;
{
    register double	sum = 0.0;

    while(n >= 8) {
	sum += *v1++ * *v2++;
	sum += *v1++ * *v2++;
	sum += *v1++ * *v2++;
	sum += *v1++ * *v2++;
	sum += *v1++ * *v2++;
	sum += *v1++ * *v2++;
	sum += *v1++ * *v2++;
	sum += *v1++ * *v2++;
	n -= 8;
    }
    while(n-- > 0) {
	sum += *v1++ * *v2++;
    }
    return(sum);
}

/* DF filter: read values from a data file				*/
static	Bool	df_open = FALSE;
static	V_INFO	df_info;

static FILTER_VAL
df(fp, t)
FILTER		*fp;
TIME		t;
{
    register int	i;
    TIME		u;
    FILTER_VAL		*where;
    TIME		when;
    TIME		howmany;
    TIME		append;
    FILTER_VAL		*vp;

    if ((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n))
        return(fp->f_cv.f_vals[t - fp->f_cv.f_t0]);
    shift_cache(fp, t, &where, &when, &howmany, &append);
    for(
	i = 0, u = when, vp = where;
	(i < howmany) && (VEOF != V(u));
	i++, u++, vp++
    ) *vp = V(u);
    fp->f_cv.f_n += i;
    if (howmany == i) fp->f_cv.f_n += append;
    return(
	((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n)) ?
	fp->f_cv.f_vals[t - fp->f_cv.f_t0] :
	INVALID_FILTER_VAL
    );
}

static Bool
make_df(type, data, fp)
FILTER_TYPE	type;
DF_DATA		*data;
FILTER		*fp;
{
    char		lbuf[LLEN];
    DF_DATA		*dp;

    if (df_open) return(TRUE);		/* only one at a time		*/
    if (NULL == fp->f_data)
        fp->f_data = (FILTER_DATA *) ealloc(sizeof(FILTER_DATA));
    else
        Nfree(fp->f_data->df_data.fname);
    dp = &fp->f_data->df_data;
    bcopy(data, dp, sizeof(DF_DATA));
    dp->type = type;
    dp->fname = estrdup(data->fname);
    sprintf(lbuf, "Data File:\n%s", dp->fname);
    fp->f_name = estrdup(lbuf);
    if (EOF == Vstream(dp->stream, dp->fname, dp->bufsiz)) {
	kill_df(fp);
	return(TRUE);
    }
    bcopy(Vinfo(), &df_info, sizeof(V_INFO));
    df_open = TRUE;
    fp->f_tmin = df_info.V_tlo;
    fp->f_tmax = df_info.V_thi;
    fp->f_fmin = df_info.V_vlo;
    fp->f_fmax = df_info.V_vhi;
    fp->f_tmul = df_info.V_tscale;
    fp->f_fmul = df_info.V_vscale;
    fp->f_cv.f_nxt = NULL;
    fp->f_cv.f_t0 = df_info.V_tlo;
    fp->f_cv.f_n = 0;
    fp->f_cv.f_sp = 1;
    if (NULL == fp->f_cv.f_vals) {
	fp->f_cv.f_vals = (FILTER_VAL *) ealloc(DF_BUFSIZ * sizeof(FILTER_VAL));
	fp->f_cv.f_nv = DF_BUFSIZ;
    }
    return(FALSE);
}

static Bool
chg_df(fp, data)
FILTER		*fp;
DF_DATA		*data;
{
    if (!df_open) return(TRUE);
    df_data_free(&fp->f_data->df_data);
    fp->f_data = NULL;
    Vend();
    df_open = FALSE;
    if (make_df(F_DF, data, fp)) return(TRUE);
    clear_caches(fp);
    return(FALSE);
}

/* nsrc_df -- since df independent of upstream filters, do nothing	*/
static Bool
nsrc_df(fp)
FILTER		*fp;
{
    return(FALSE);
}

static Bool
kill_df(fp)
FILTER		*fp;
{
    if (df_open) Vend();
    df_open = FALSE;
    delete_filter(fp);
    Nfree(fp->f_name);
    df_data_free(&fp->f_data->df_data);
    fp->f_data = NULL;
    kill_cache(&fp->f_cv);
    kill_cache(&fp->f_cm);
    fp->f_type = F_INACTIVE;
    return(FALSE);
}

static void
df_data_free(dp)
DF_DATA		*dp;
{
    if (NULL != dp) {
	if (NULL != dp->fname) free(dp->fname);
	free(dp);
    }
}

typedef	struct	LHP_PRIV {		/* private data for bandpass	*/
    TIME	back;			/* how far to backtrack		*/
    double	fac;			/* coefficients			*/
    double	fm1;
}		LHP_PRIV;

static FILTER_VAL
hp(fp, t)
FILTER		*fp;
TIME		t;
{
    TIME		tseed;
    FILTER_VAL		seed;
    FILTER_VAL		*where;
    TIME		when;
    TIME		howmany;
    TIME		append;
    LHP_PRIV		*pp = (LHP_PRIV *) fp->f_priv;
    FILTER_VAL		val;

    if ((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n))
        return(fp->f_cv.f_vals[t - fp->f_cv.f_t0]);
    if ((t < fp->f_tmin) || (t >= fp->f_tmax))
	return(INVALID_FILTER_VAL);
    if (fp->f_cv.f_n > 0) {
	tseed = fp->f_cv.f_t0 + fp->f_cv.f_n - 1;
	seed = F(fp->f_src, tseed) - fp->f_cv.f_vals[fp->f_cv.f_n - 1];
    }
    else {
	seed = INVALID_FILTER_VAL;
    }
    shift_cache(fp, t, &where, &when, &howmany, &append);
    lhp_seed(fp, when, &tseed, &seed);
    while(howmany > 0) {
	val = F(fp->f_src, when);
	seed = pp->fac * val + pp->fm1 * seed;
	*where++ = val - seed;
	when++;
	howmany--;
    }
    fp->f_cv.f_n = where - fp->f_cv.f_vals;
    if (howmany <= 0) fp->f_cv.f_n += append;
    return(
	((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n)) ?
	fp->f_cv.f_vals[t - fp->f_cv.f_t0] :
	INVALID_FILTER_VAL
    );
}

static Bool
make_hp_tau(type, data, fp)
FILTER_TYPE	type;
HP_TAU_DATA	*data;
FILTER		*fp;
{
    fp->f_data = (FILTER_DATA *) ealloc(sizeof(FILTER_DATA));
    fp->f_priv = NULL;
    fp->f_cv.f_vals = (FILTER_VAL *) ealloc(DF_BUFSIZ * sizeof(FILTER_VAL));
    fp->f_cv.f_nv = DF_BUFSIZ;
    if (chg_hp_tau(fp, data)) {
	kill_filter(fp);
	return(TRUE);
    }
    return(FALSE);
}

static Bool
make_hp_f(type, data, fp)
FILTER_TYPE	type;
HP_F_DATA	*data;
FILTER		*fp;
{
    fp->f_data = (FILTER_DATA *) ealloc(sizeof(FILTER_DATA));
    fp->f_priv = NULL;
    fp->f_cv.f_vals = (FILTER_VAL *) ealloc(DF_BUFSIZ * sizeof(FILTER_VAL));
    fp->f_cv.f_nv = DF_BUFSIZ;
    if (chg_hp_f(fp, data)) {
	kill_filter(fp);
	return(TRUE);
    }
    return(FALSE);
}

static Bool
chg_hp_tau(fp, data)
FILTER		*fp;
HP_TAU_DATA	*data;
{
    char		lbuf[LLEN];
    HP_TAU_DATA		*dp;
    LHP_PRIV		*pp;

    if ((0.0 > data->tol) || (0.0 > data->tau)) return(TRUE);
    dp = &fp->f_data->hp_tau_data;
    bcopy(data, dp, sizeof(HP_TAU_DATA));
    dp->type = fp->f_type;
    pp = (LHP_PRIV *) ealloc(sizeof(LHP_PRIV));
    if (0.0 == dp->tau) {
	pp->back = 0;
	pp->fac = 1.0;
	pp->fm1 = 0.0;
    }
    else {
	if (0.0 == dp->tol) pp->back = MAXLONG/2;
	else pp->back = ceil(-log(dp->tol) * dp->tau);
	pp->fac = -expm1(-1.0/dp->tau);
	pp->fm1 = 1.0 - pp->fac;
    }
    Nfree(fp->f_priv);
    fp->f_priv = (VOIDST) pp;
    clear_caches(fp);
    Nfree(fp->f_name);
    sprintf(lbuf, "Hi Pass:\ntau = %.15g", data->tau);
    fp->f_name = estrdup(lbuf);
    return(FALSE);
}

static Bool
chg_hp_f(fp, data)
FILTER		*fp;
HP_F_DATA	*data;
{
    char		lbuf[LLEN];
    HP_F_DATA		*dp;
    LHP_PRIV		*pp;
    double		tau = 0.0;

    if ((0.0 > data->tol) || (0.0 > data->f)) return(TRUE);
    dp = &fp->f_data->hp_f_data;
    bcopy(data, dp, sizeof(HP_F_DATA));
    dp->type = fp->f_type;
    pp = (LHP_PRIV *) ealloc(sizeof(LHP_PRIV));
    if (0.0 < dp->f) {
					/* tau = 1/(2 Pi f Sqrt[3])	*/
	tau = 0.0918881492369653415852210830521615735662 / dp->f;
	pp->fac = -expm1(-1.0/tau);
	pp->fm1 = 1.0 - pp->fac;
    }
    else {
	pp->fac = 0.0;
	pp->fm1 = 1.0;
    }
    if ((0.0 == dp->f) || (0.0 == dp->tol)) pp->back = MAXLONG/2;
    else pp->back = ceil(-log(dp->tol) * tau);
    Nfree(fp->f_priv);
    fp->f_priv = (VOIDST) pp;
    clear_caches(fp);
    Nfree(fp->f_name);
    sprintf(lbuf, "Hi Pass:\n-3 dB frequency = %.15g", data->f);
    fp->f_name = estrdup(lbuf);
    return(FALSE);
}

static FILTER_VAL
lp(fp, t)
FILTER		*fp;
TIME		t;
{
    TIME		tseed;
    FILTER_VAL		seed;
    FILTER_VAL		*where;
    TIME		when;
    TIME		howmany;
    TIME		append;
    LHP_PRIV		*pp = (LHP_PRIV *) fp->f_priv;
    FILTER_VAL		val;

    if ((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n))
        return(fp->f_cv.f_vals[t - fp->f_cv.f_t0]);
    if ((t < fp->f_tmin) || (t >= fp->f_tmax))
	return(INVALID_FILTER_VAL);
    if (fp->f_cv.f_n > 0) {
	seed = fp->f_cv.f_vals[fp->f_cv.f_n - 1];
	tseed = fp->f_cv.f_t0 + fp->f_cv.f_n - 1;
    }
    else {
	seed = INVALID_FILTER_VAL;
    }
    shift_cache(fp, t, &where, &when, &howmany, &append);
    lhp_seed(fp, when, &tseed, &seed);
    while(howmany > 0) {
	val = F(fp->f_src, when);
	seed = pp->fac * val + pp->fm1 * seed;
	*where++ = seed;
	when++;
	howmany--;
    }
    fp->f_cv.f_n = where - fp->f_cv.f_vals;
    if (howmany <= 0) fp->f_cv.f_n += append;
    return(
	((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n)) ?
	fp->f_cv.f_vals[t - fp->f_cv.f_t0] :
	INVALID_FILTER_VAL
    );
}

/* lhp_seed -- compute a seed for low or hipass filter
 *
 * FILTER	*fp
 * TIME		tseed;
 * FILTER_VAL	seed;
 * lhp_seed(fp, when, &tseed, &seed);
 *
 * On call, tseed and seed should be the time and seed value
 * corresponding to the last value that was in the original cache.
 * when is the first time point that needs to be computed after
 * shifting the cache.  On return tseed will be when-1 and seed the
 * corresponding seed value.
 */
static void
lhp_seed(fp, when, tseed, seed)
FILTER		*fp;
TIME		when;
TIME		*tseed;
FILTER_VAL	*seed;
{
    LHP_PRIV	*pp = (LHP_PRIV *) fp->f_priv;

    if (
	IS_INVALID_FILTER_VAL(*seed) ||
	(*tseed < when - pp->back) ||
	(*tseed >= when)
    ) {
	*tseed = when - pp->back;
	if (*tseed <= fp->f_tmin) {
	    *tseed = fp->f_tmin - 1;
	    *seed = F(fp->f_src, fp->f_tmin);
	}
	else {
	    *seed = F(fp->f_src, *tseed);
	}
    }
    while(*tseed < when-2) {
	(*tseed)++;
	*seed = pp->fac * F(fp->f_src, *tseed) + pp->fm1 * *seed;
    }
}

static Bool
make_lp_tau(type, data, fp)
FILTER_TYPE	type;
LP_TAU_DATA	*data;
FILTER		*fp;
{
    fp->f_data = (FILTER_DATA *) ealloc(sizeof(FILTER_DATA));
    fp->f_priv = NULL;
    fp->f_cv.f_vals = (FILTER_VAL *) ealloc(DF_BUFSIZ * sizeof(FILTER_VAL));
    fp->f_cv.f_nv = DF_BUFSIZ;
    if (chg_lp_tau(fp, data)) {
	kill_filter(fp);
	return(TRUE);
    }
    return(FALSE);
}

static Bool
make_lp_f(type, data, fp)
FILTER_TYPE	type;
LP_F_DATA	*data;
FILTER		*fp;
{
    fp->f_data = (FILTER_DATA *) ealloc(sizeof(FILTER_DATA));
    fp->f_priv = NULL;
    fp->f_cv.f_vals = (FILTER_VAL *) ealloc(DF_BUFSIZ * sizeof(FILTER_VAL));
    fp->f_cv.f_nv = DF_BUFSIZ;
    if (chg_lp_f(fp, data)) {
	kill_filter(fp);
	return(TRUE);
    }
    return(FALSE);
}

static Bool
chg_lp_tau(fp, data)
FILTER		*fp;
LP_TAU_DATA	*data;
{
    char		lbuf[LLEN];
    LP_TAU_DATA		*dp;
    LHP_PRIV		*pp;

    if ((0.0 > data->tol) || (0.0 > data->tau)) return(TRUE);
    dp = &fp->f_data->lp_tau_data;
    bcopy(data, dp, sizeof(LP_TAU_DATA));
    dp->type = fp->f_type;
    pp = (LHP_PRIV *) ealloc(sizeof(LHP_PRIV));
    if (0.0 == dp->tau) {
	pp->back = 0;
	pp->fac = 1.0;
	pp->fm1 = 0.0;
    }
    else {
	if (0.0 == dp->tol) pp->back = MAXLONG/2;
	else pp->back = ceil(-log(dp->tol) * dp->tau);
	pp->fac = -expm1(-1.0/dp->tau);
	pp->fm1 = 1.0 - pp->fac;
    }
    Nfree(fp->f_priv);
    fp->f_priv = (VOIDST) pp;
    clear_caches(fp);
    Nfree(fp->f_name);
    sprintf(lbuf, "Lo Pass:\ntau = %.15g", data->tau);
    fp->f_name = estrdup(lbuf);
    return(FALSE);
}

static Bool
chg_lp_f(fp, data)
FILTER		*fp;
LP_F_DATA	*data;
{
    char		lbuf[LLEN];
    LP_F_DATA		*dp;
    LHP_PRIV		*pp;
    double		tau = 0.0;

    if ((0.0 > data->tol) || (0.0 > data->f)) return(TRUE);
    dp = &fp->f_data->lp_f_data;
    bcopy(data, dp, sizeof(LP_F_DATA));
    dp->type = fp->f_type;
    pp = (LHP_PRIV *) ealloc(sizeof(LHP_PRIV));
    if (0.0 < dp->f) {
					/* tau = Sqrt[3]/(2 Pi f)	*/
	tau = 0.2756644477108960247556632491564847206987 / dp->f;
	pp->fac = -expm1(-1.0/tau);
	pp->fm1 = 1.0 - pp->fac;
    }
    else {
	pp->fac = 0.0;
	pp->fm1 = 1.0;
    }
    if ((0.0 == dp->f) || (0.0 == dp->tol)) pp->back = MAXLONG/2;
    else pp->back = ceil(-log(dp->tol) * tau);
    Nfree(fp->f_priv);
    fp->f_priv = (VOIDST) pp;
    clear_caches(fp);
    Nfree(fp->f_name);
    sprintf(lbuf, "Lo Pass:\n-3 dB frequency = %.15g", data->f);
    fp->f_name = estrdup(lbuf);
    return(FALSE);
}

/* shift_cache -- center new t value in cache
 *
 * FILTER_CACHE	cache;
 * TIME		t;
 * FILTER_VAL	*where;
 * TIME		when;
 * TIME		howmany;
 * TIME		append;
 * shift_cache(&cache, t, &where, &when, &howmany, &append);
 *
 * shift_cache adjusts cache so that the TIME t is centered.  On return
 * where, when, and howmany are set to indicate what new values have
 * to be read to fill up the cache.  If the new values will be
 * appended to values already in the cache, shift_cache returns with
 * cache.f_n indicating the number of valid values already in the
 * cache, and append set to zero.  If the new values will be prepended
 * to values already in the cache, shift_cache returns with cache.f_n
 * == 0 and append equal to the number of values in the cache at
 * position where + howmany.
 */
static void
shift_cache(fp, t, where, when, howmany, append)
FILTER		*fp;
TIME		t;
FILTER_VAL	**where;
TIME		*when;
TIME		*howmany;
TIME		*append;
{
    TIME		t0;
    FILTER_CACHE	*cp = &fp->f_cv;
    TIME		nv = fp->f_cv.f_nv;

    t0 = t - nv / 2;
    if (t0 < fp->f_tmin) t0 = fp->f_tmin;
    if (t0 + nv > fp->f_tmax) t0 = fp->f_tmax - nv;
    if (t0 < fp->f_tmin) {
	t0 = fp->f_tmin;
	nv = fp->f_tmax - fp->f_tmin;
    }
    /*
     * old range: |----------|
     * new range:     |----------|
     */
    if ((t0 >= cp->f_t0) && (t0 < cp->f_t0 + cp->f_n)) {
	if (t0 > cp->f_t0) {
	    bcopy(
	        &cp->f_vals[t0 - cp->f_t0],
	        cp->f_vals,
	        (cp->f_t0 + cp->f_n - t0) * sizeof(FILTER_VAL)
	    );
	    cp->f_n -= t0 - cp->f_t0;
	    cp->f_t0 = t0;
	}
	*where = &cp->f_vals[cp->f_n];
	*when = cp->f_t0 + cp->f_n;
	*howmany = nv - cp->f_n;
	*append = 0;
    }
    /*
     * old range:      |----------|
     * new range: |----------|
     */
    else if (
	(t0 + nv > cp->f_t0) &&
	(t0 + nv < cp->f_t0 + cp->f_n)
    ) {
	bcopy(
	    cp->f_vals,
	    &cp->f_vals[cp->f_t0 - t0],
	    (t0 + nv - cp->f_t0) * sizeof(FILTER_VAL)
	);
	*where = cp->f_vals;
	*when = t0;
	*howmany = cp->f_t0 - t0;
	*append = t0 + nv - cp->f_t0;
	cp->f_t0 = t0;
	cp->f_n = 0;
    }
    /*
     * old range:   !------|
     * new range: |----------|
     */
    else if ((t0 < cp->f_t0) && (t0 + nv >= cp->f_t0 + cp->f_n)) {
	bcopy(
	    cp->f_vals,
	    &cp->f_vals[cp->f_t0 - t0],
	    cp->f_n * sizeof(FILTER_VAL)
	);
	*where = cp->f_vals;
	*when = t0;
	*howmany = cp->f_t0 - t0;
	*append = cp->f_n;
	cp->f_t0 = t0;
	cp->f_n = 0;
    }
    /*
     * old range: |----------|
     * new range:              |----------|
     */
    else {
	*where = cp->f_vals;
	*when = t0;
	*howmany = nv;
	*append = 0;
	cp->f_t0 = t0;
	cp->f_n = 0;
    }
}

static Bool
kill_generic(fp)
FILTER		*fp;
{
    delete_filter(fp);
    Nfree(fp->f_name);
    Nfree(fp->f_data);
    Nfree(fp->f_priv);
    kill_cache(&fp->f_cv);
    kill_cache(&fp->f_cm);
    fp->f_type = F_INACTIVE;
    return(FALSE);
}

static FILTER_VAL
mm(fp, t)
FILTER		*fp;
TIME		t;
{
    int			i;
    MM_DATA		*dp = &fp->f_data->mm_data;
    FILTER_VAL		*where;
    TIME		when;
    TIME		howmany;
    TIME			append;
    TIME		tin;
    FILTER_VAL		vin;
    TIME		tn;
    FILTER_VAL		vn;
    TIME		tx;
    FILTER_VAL		vx;

    if ((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n))
        return(fp->f_cv.f_vals[t - fp->f_cv.f_t0]);
    if ((t < fp->f_tmin) || (t >= fp->f_tmax))
	return(INVALID_FILTER_VAL);
    shift_cache(fp, t, &where, &when, &howmany, &append);
    tin = (when >> 1);
    tin *= dp->n;
    if ((0 != (when & 1)) && (howmany > 0)) {
	tn = tx = 0;
	vn = vx = F(fp->f_src, tin);
	for(i=1; i < dp->n; i++) {
	    vin = F(fp->f_src, tin + i);
	    if (vin < vn) {
		tn = i;
		vn = vin;
	    }
	    if (vin > vx) {
		tx = i;
		vx = vin;
	    }
	}
	*where++ = ((tx > tn) ? vx : vn);
	tin += dp->n;
	howmany--;
    }
    while(howmany >= 2) {
	tn = tx = 0;
	vn = vx = F(fp->f_src, tin);
	for(i=1; i<dp->n; i++) {
	    vin = F(fp->f_src, tin + i);
	    if (vin < vn) {
		tn = i;
		vn = vin;
	    }
	    if (vin > vx) {
		tx = i;
		vx = vin;
	    }
	}
	if (tn < tx) {
	    *where++ = vn;
	    *where++ = vx;
	}
	else {
	    *where++ = vx;
	    *where++ = vn;
	}
	tin += dp->n;
	howmany -= 2;
    }
    if (howmany > 0) {
	tn = tx = 0;
	vn = vx = F(fp->f_src, tin);
	for(i=1; i<dp->n; i++) {
	    vin = F(fp->f_src, tin + i);
	    if (vin < vn) {
		tn = i;
		vn = vin;
	    }
	    if (vin > vx) {
		tx = i;
		vx = vin;
	    }
	}
	*where++ = ((tx > tn) ? vn : vx);
	tin += dp->n;
	howmany--;
    }
    fp->f_cv.f_n = where - fp->f_cv.f_vals;
    if (howmany <= 0) fp->f_cv.f_n += append;
    return(
	((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n)) ?
	fp->f_cv.f_vals[t - fp->f_cv.f_t0] :
	INVALID_FILTER_VAL
    );
}

static Bool
make_mm(type, data, fp)
FILTER_TYPE	type;
MM_DATA		*data;
FILTER		*fp;
{
    fp->f_data = (FILTER_DATA *) ealloc(sizeof(FILTER_DATA));
    fp->f_cv.f_vals = (FILTER_VAL *) ealloc(DF_BUFSIZ * sizeof(FILTER_VAL));
    fp->f_cv.f_nv = DF_BUFSIZ;
    if (chg_mm(fp, data)) {
	kill_filter(fp);
	return(TRUE);
    }
    return(FALSE);
}

static Bool
chg_mm(fp, data)
FILTER		*fp;
MM_DATA		*data;
{
    char		lbuf[LLEN];
    MM_DATA		*dp;

    if (0 >= data->n) return(TRUE);
    dp = &fp->f_data->mm_data;
    bcopy(data, dp, sizeof(MM_DATA));
    dp->type = fp->f_type;
    clear_caches(fp);
    Nfree(fp->f_name);
    sprintf(lbuf, "Minimax:\nn = %d", data->n);
    fp->f_name = estrdup(lbuf);
    return(FALSE);
}

static Bool
nsrc_mm(fp)
FILTER		*fp;
{
    MM_DATA		*dp = &fp->f_data->mm_data;
    double		tm;
    TIME		np;

    if ((NULL != fp->f_cv.f_nxt) || (1 != fp->f_cv.f_sp))
        error("nsrc_mm: can't happen");
    fp->f_cv.f_t0 = 0;
    fp->f_cv.f_n = 0;
    kill_cache(&fp->f_cm);
    if (NULL == fp->f_src) {
	fp->f_tmin = fp->f_tmax = 0;
	fp->f_fmax = infinity();
	fp->f_fmin = -infinity();
	fp->f_tmul = dp->n / 2.0;
	fp->f_fmul = 1.0;
    }
    else {
	tm = fp->f_src->f_tmin;
	tm /= dp->n;
	tm = ceil(tm - 1.0e-7);
	fp->f_tmin = tm;
	fp->f_tmin *= 2;
	tm = fp->f_src->f_tmax;
	tm /= dp->n;
	tm = floor(tm + 1.0e-7);
	fp->f_tmax = tm;
	fp->f_tmax *= 2;
	fp->f_fmin = fp->f_src->f_fmin;
	fp->f_fmax = fp->f_src->f_fmax;
	fp->f_tmul = fp->f_src->f_tmul * dp->n / 2.0;
	fp->f_fmul = fp->f_src->f_fmul;
    }
    return(FALSE);
}

static FILTER_VAL
scp(fp, t)
FILTER		*fp;
TIME		t;
{
    return(F(fp->f_src, t));
}

static FILTER_VAL
scn(fp, t)
FILTER		*fp;
TIME		t;
{
    return(-F(fp->f_src, t));
}

static Bool
make_sc(type, data, fp)
FILTER_TYPE	type;
SC_DATA		*data;
FILTER		*fp;
{
    fp->f_data = (FILTER_DATA *) ealloc(sizeof(FILTER_DATA));
    if (chg_sc(fp, data)) {
	kill_filter(fp);
	return(TRUE);
    }
    return(FALSE);
}

static Bool
chg_sc(fp, data)
FILTER		*fp;
SC_DATA		*data;
{
    char		lbuf[LLEN];
    SC_DATA		*dp;

    if (0.0 == data->factor) return(TRUE);
    dp = &fp->f_data->sc_data;
    bcopy(data, dp, sizeof(SC_DATA));
    dp->type = fp->f_type;
    Nfree(fp->f_name);
    sprintf(lbuf, "Scale:\nfactor = %.15g", data->factor);
    fp->f_name = estrdup(lbuf);
    if (data->factor < 0.0)
	fp->f_func = &scn;
    else
	fp->f_func = &scp;
    clear_caches(fp);
    return(FALSE);
}

static Bool
nsrc_sc(fp)
FILTER		*fp;
{
    SC_DATA		*dp = &fp->f_data->sc_data;

    if (NULL == fp->f_src) {
	fp->f_tmin = fp->f_tmax = 0;
	fp->f_fmax = infinity();
	fp->f_fmin = -infinity();
	fp->f_tmul = 1.0;
	fp->f_fmul = fabs(dp->factor);
    }
    else {
	fp->f_tmin = fp->f_src->f_tmin;
	fp->f_tmax = fp->f_src->f_tmax;
	fp->f_tmul = fp->f_src->f_tmul;
	if (dp->factor < 0.0) {
	    fp->f_fmin = -fp->f_src->f_fmax;
	    fp->f_fmax = -fp->f_src->f_fmin;
	    fp->f_fmul = -dp->factor * fp->f_src->f_fmul;
	}
	else {
	    fp->f_fmin = fp->f_src->f_fmin;
	    fp->f_fmax = fp->f_src->f_fmax;
	    fp->f_fmul = dp->factor * fp->f_src->f_fmul;
	}
    }
    return(FALSE);
}

static FILTER_VAL
pk(fp, t)
FILTER		*fp;
TIME		t;
{
    int			i;
    PK_DATA		*dp = &fp->f_data->pk_data;
    FILTER_VAL		*where;
    TIME		when;
    TIME		howmany;
    TIME			append;
    TIME		tin;
    FILTER_VAL		vin;

    if ((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n))
        return(fp->f_cv.f_vals[t - fp->f_cv.f_t0]);
    if ((t < fp->f_tmin) || (t >= fp->f_tmax))
	return(INVALID_FILTER_VAL);
    shift_cache(fp, t, &where, &when, &howmany, &append);
    tin = (when * dp->n) + dp->st;
    while(howmany-- > 0) {
	*where++ = F(fp->f_src, tin);
	tin += dp->n;
    }
    fp->f_cv.f_n = where - fp->f_cv.f_vals;
    if (howmany <= 0) fp->f_cv.f_n += append;
    return(
	((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n)) ?
	fp->f_cv.f_vals[t - fp->f_cv.f_t0] :
	INVALID_FILTER_VAL
    );
}

static Bool
make_pk(type, data, fp)
FILTER_TYPE	type;
PK_DATA		*data;
FILTER		*fp;
{
    fp->f_data = (FILTER_DATA *) ealloc(sizeof(FILTER_DATA));
    fp->f_cv.f_vals = (FILTER_VAL *) ealloc(DF_BUFSIZ * sizeof(FILTER_VAL));
    fp->f_cv.f_nv = DF_BUFSIZ;
    if (chg_pk(fp, data)) {
	kill_filter(fp);
	return(TRUE);
    }
    return(FALSE);
}

static Bool
chg_pk(fp, data)
FILTER		*fp;
PK_DATA		*data;
{
    char		lbuf[LLEN];
    PK_DATA		*dp;

    if ((0 > data->st) || (0 >= data->n)) return(TRUE);
    dp = &fp->f_data->pk_data;
    bcopy(data, dp, sizeof(PK_DATA));
    dp->type = fp->f_type;
    clear_caches(fp);
    Nfree(fp->f_name);
    sprintf(lbuf, "Pick:\nstart = %d, n = %d", data->st, data->n);
    fp->f_name = estrdup(lbuf);
    return(FALSE);
}

static Bool
nsrc_pk(fp)
FILTER		*fp;
{
    PK_DATA		*dp = &fp->f_data->pk_data;
    double		tm;
    TIME		np;

    if ((NULL != fp->f_cv.f_nxt) || (1 != fp->f_cv.f_sp))
        error("nsrc_pk: can't happen");
    fp->f_cv.f_t0 = 0;
    fp->f_cv.f_n = 0;
    kill_cache(&fp->f_cm);
    if (NULL == fp->f_src) {
	fp->f_tmin = fp->f_tmax = 0;
	fp->f_fmax = infinity();
	fp->f_fmin = -infinity();
	fp->f_tmul = dp->n;
	fp->f_fmul = 1.0;
    }
    else {
	tm = fp->f_src->f_tmin;
	tm /= dp->n;
	tm = ceil(tm - 1.0e-7);
	fp->f_tmin = tm;
	tm = fp->f_src->f_tmax;
	tm /= dp->n;
	tm = floor(tm + 1.0e-7);
	fp->f_tmax = tm;
	fp->f_fmin = fp->f_src->f_fmin;
	fp->f_fmax = fp->f_src->f_fmax;
	fp->f_tmul = fp->f_src->f_tmul * dp->n;
	fp->f_fmul = fp->f_src->f_fmul;
    }
    return(FALSE);
}

static FILTER_VAL
tp(fp, t)
FILTER		*fp;
TIME		t;
{
    FILTER_VAL		*where;
    TIME		when;
    TIME		howmany;
    TIME		append;
    FILTER_VAL		val;
    TP_DATA		*dp = &fp->f_data->tp_data;
    TP_PRIV		*pp = (TP_PRIV *) fp->f_priv;

    if ((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n))
        return(fp->f_cv.f_vals[t - fp->f_cv.f_t0]);
    if ((t < fp->f_tmin) || (t >= fp->f_tmax))
	return(INVALID_FILTER_VAL);
    shift_cache(fp, t, &where, &when, &howmany, &append);
    if (INVALID == pp->tpl.type) {
	while(howmany > 0) {
	    *where++ = F(fp->f_src, when);
	    when++;
	    howmany--;
	}
    }
    else {
	register int	i;
	FILTER_VAL	*v;		/* vector			*/
	FILTER_VAL	sv = 0.0;	/* sum of values		*/
	FILTER_VAL	ssv = 0.0;	/* sum of squared values	*/
	FILTER_VAL	stv;		/* sum of vals * template	*/
	FILTER_VAL	ssr;		/* sum of squared residuals	*/
	FILTER_VAL	s2n;		/* s^2 * (n-1)			*/
	FILTER_VAL	b;		/* fitted size factor		*/
	double		dfr;		/* ratio of degrees of freedom	*/
	int		nbc;		/* # bytes to copy		*/

	/*
	 * initialize: set up vector and initial sums
	 */
	v = (FILTER_VAL *) ealloc(pp->nvals * sizeof(FILTER_VAL));
	when -= pp->tzero;
	for(i=0; i<pp->nvals; i++) {
	    v[i] = F(fp->f_src, when); when++;
	    sv += v[i];
	}
	ssv = dot_product(v, v, pp->nvals);
	dfr = pp->nvals - 2;
	dfr /= pp->nvals - 1;
	nbc = (pp->nvals - 1) * sizeof(FILTER_VAL);
	/*
	 * calculate values
	 */
	while(howmany > 0) {
	    s2n = ssv - sv * sv / pp->nvals;
	    stv = dot_product(v, pp->vals, pp->nvals);
	    if ((stv < pp->nmin) || (stv > pp->nmax)) {
		if (stv < pp->nmin) b = pp->nmin;
		else b = pp->nmax;
		ssr = s2n + b * b - 2 * b * stv;
	    }
	    else {
		ssr = s2n - stv * stv;
	    }
	    if (0.0 >= ssr) {
		val = dp->maxout;
	    }
	    else {
		val = dfr * s2n / ssr;
	    }
	    *where++ = val;
	    howmany--;
	    if (howmany <= 0) break;
	    /*
	     * shift to next point
	     */
	    sv -= v[0];
	    ssv -= v[0] * v[0];
	    bcopy(&v[1], &v[0], nbc);
	    v[pp->nvals - 1] = val = F(fp->f_src, when); when++;
	    sv += val;
	    ssv += val * val;
	}
	free(v);
    }
    fp->f_cv.f_n = where - fp->f_cv.f_vals;
    if (howmany <= 0) fp->f_cv.f_n += append;
    return(
	   ((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n)) ?
	   fp->f_cv.f_vals[t - fp->f_cv.f_t0] :
	   INVALID_FILTER_VAL
	   );
}

static Bool
make_tp(type, data, fp)
FILTER_TYPE	type;
TP_DATA		*data;
FILTER		*fp;
{
    int			bufsiz = max(DF_BUFSIZ/4, BUFSIZ);

    fp->f_data = (FILTER_DATA *) ealloc(sizeof(FILTER_DATA));
    bzero(fp->f_data, sizeof(*fp->f_data));
    fp->f_priv = NULL;
    fp->f_cv.f_vals = (FILTER_VAL *) ealloc(bufsiz * sizeof(FILTER_VAL));
    fp->f_cv.f_nv = bufsiz;
    if (chg_tp(fp, data)) {
	kill_filter(fp);
	return(TRUE);
    }
    return(FALSE);
}

static Bool
chg_tp(fp, data)
FILTER		*fp;
TP_DATA		*data;
{
    char		lbuf[LLEN];
    TP_DATA		*dp;
    TP_PRIV		*pp;
    char		*s;

    if (0.0 == data->tscale) return(TRUE);
    dp = &fp->f_data->tp_data;
    Nfree(dp->fname);
    bcopy(data, dp, sizeof(TP_DATA));
    dp->fname = estrdup(data->fname);
    dp->type = fp->f_type;
    pp = (TP_PRIV *) ealloc(sizeof(TP_PRIV));
    pp->vals = NULL;
    if ('\0' == *dp->fname) {
	pp->tpl.type = INVALID;
    }
    else {
	register int	i;
	register double	sum;

	if (
	    (NULL != (s = read_template(dp->fname, &pp->tpl))) ||
	    (NULL != (s = cook_template(&pp->tpl, dp->tscale,
					&pp->vals, &pp->nvals, &pp->tzero))) ||
	    (0 >= pp->nvals)
	) {
	    Nfree(s);
	    destroy_tp_priv(pp);
	    return(TRUE);
	}
	if (2 >= pp->nvals) {
	    pp->tpl.type = INVALID;
	}
	else {
	    sum = 0.0;
	    for(i=0; i<pp->nvals; i++) {
		sum += pp->vals[i];
	    }
	    sum /= pp->nvals;
	    for(i=0; i<pp->nvals; i++) {
		pp->vals[i] -= sum;
	    }
	    pp->nfac = sqrt(dot_product(pp->vals, pp->vals, pp->nvals));
	    if (0.0 < pp->nfac) {
		for(i=0; i<pp->nvals; i++) {
		    pp->vals[i] /= pp->nfac;
		}
		pp->nmin = dp->min * pp->nfac * dp->fscale;
		pp->nmax = dp->max * pp->nfac * dp->fscale;
	    }
	    else {
		pp->tpl.type = INVALID;
	    }
	}
    }
    if (NULL != fp->f_priv) destroy_tp_priv((TP_PRIV *) fp->f_priv);
    fp->f_priv = (VOIDST) pp;
    Nfree(fp->f_name);
    sprintf(lbuf,
	    "Template:\ntemplate file = \"%s\", tscale = %.15g, fscale = %.15g, maxout = %.15g, bounds = %.15g, %.15g",
	    data->fname, data->tscale, data->fscale, data->maxout,
	    data->min, data->max);
    fp->f_name = estrdup(lbuf);
    clear_caches(fp);
    return(FALSE);
}

static Bool
nsrc_tp(fp)
FILTER		*fp;
{
    TP_DATA		*dp = &fp->f_data->tp_data;
    TP_PRIV		*pp = (TP_PRIV *) fp->f_priv;

    nsrc_generic(fp);
    if ((NULL == pp) || (INVALID == pp->tpl.type)) return(FALSE);
    if (NULL == fp->f_src) {
	fp->f_tmin = fp->f_tmax = 0;
	fp->f_fmax = dp->maxout;
	fp->f_fmin = 0.0;
	fp->f_tmul = 1.0;
	fp->f_fmul = dp->fscale;
    }
    else {
	fp->f_tmin = fp->f_src->f_tmin + pp->tzero;
	fp->f_tmax = fp->f_src->f_tmax - pp->nvals + pp->tzero;
	fp->f_tmul = fp->f_src->f_tmul;
	fp->f_fmin = 0.0;
	fp->f_fmax = dp->maxout;
	fp->f_fmul = dp->fscale * fp->f_src->f_fmul;
    }
    return(FALSE);
}

static Bool
kill_tp(fp)
FILTER		*fp;
{
    TP_DATA		*dp = &fp->f_data->tp_data;

    delete_filter(fp);
    Nfree(fp->f_name);
    Nfree(dp->fname);
    Nfree(fp->f_data);
    destroy_tp_priv((TP_PRIV *) fp->f_priv);
    fp->f_priv = NULL;
    kill_cache(&fp->f_cv);
    kill_cache(&fp->f_cm);
    fp->f_type = F_INACTIVE;
    return(FALSE);
}

static void
destroy_tp_priv(pp)
TP_PRIV		*pp;
{
    if (NULL == pp) return;
    switch(pp->tpl.type) {
    case INTERPOLATED:
	Nfree(pp->tpl.templ_inter.points);
	break;

    default:
	break;
    }
    Nfree(pp->vals);
    free(pp);
}

static FILTER_VAL
cv(fp, t)
FILTER		*fp;
TIME		t;
{
    FILTER_VAL		*where;
    TIME		when;
    TIME		howmany;
    TIME		append;
    FILTER_VAL		val;
    CV_DATA		*dp = &fp->f_data->cv_data;
    CV_PRIV		*pp = (CV_PRIV *) fp->f_priv;

    if ((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n))
        return(fp->f_cv.f_vals[t - fp->f_cv.f_t0]);
    if ((t < fp->f_tmin) || (t >= fp->f_tmax))
	return(INVALID_FILTER_VAL);
    shift_cache(fp, t, &where, &when, &howmany, &append);
    if (INVALID == pp->tpl.type) {
	while(howmany > 0) {
	    *where++ = F(fp->f_src, when);
	    when++;
	    howmany--;
	}
    }
    else {
	register int	i;
	FILTER_VAL	*v;		/* vector			*/
	FILTER_VAL	val;		/* sum of vals * template	*/
	int		nbc;		/* # bytes to copy		*/

	/*
	 * initialize: set up vector and initial sums
	 */
	v = (FILTER_VAL *) ealloc(pp->nvals * sizeof(FILTER_VAL));
	when -= pp->tzero;
	for(i=0; i<pp->nvals; i++) {
	    v[i] = F(fp->f_src, when); when++;
	}
	nbc = (pp->nvals - 1) * sizeof(FILTER_VAL);
	/*
	 * calculate values
	 */
	while(howmany > 0) {
	    val = dot_product(v, pp->vals, pp->nvals);
	    *where++ = val;
	    howmany--;
	    if (howmany <= 0) break;
	    /*
	     * shift to next point
	     */
	    bcopy(&v[1], &v[0], nbc);
	    v[pp->nvals - 1] = F(fp->f_src, when); when++;
	}
	free(v);
    }
    fp->f_cv.f_n = where - fp->f_cv.f_vals;
    if (howmany <= 0) fp->f_cv.f_n += append;
    return(
	   ((t >= fp->f_cv.f_t0) && (t - fp->f_cv.f_t0 < fp->f_cv.f_n)) ?
	   fp->f_cv.f_vals[t - fp->f_cv.f_t0] :
	   INVALID_FILTER_VAL
	   );
}

static Bool
make_cv(type, data, fp)
FILTER_TYPE	type;
CV_DATA		*data;
FILTER		*fp;
{
    int			bufsiz = max(DF_BUFSIZ/4, BUFSIZ);

    fp->f_data = (FILTER_DATA *) ealloc(sizeof(FILTER_DATA));
    bzero(fp->f_data, sizeof(*fp->f_data));
    fp->f_priv = NULL;
    fp->f_cv.f_vals = (FILTER_VAL *) ealloc(bufsiz * sizeof(FILTER_VAL));
    fp->f_cv.f_nv = bufsiz;
    if (chg_cv(fp, data)) {
	kill_filter(fp);
	return(TRUE);
    }
    return(FALSE);
}

static Bool
chg_cv(fp, data)
FILTER		*fp;
CV_DATA		*data;
{
    char		lbuf[LLEN];
    CV_DATA		*dp;
    CV_PRIV		*pp;
    char		*s;

    if (0.0 == data->tscale) return(TRUE);
    dp = &fp->f_data->cv_data;
    Nfree(dp->fname);
    bcopy(data, dp, sizeof(CV_DATA));
    dp->fname = estrdup(data->fname);
    dp->type = fp->f_type;
    pp = (CV_PRIV *) ealloc(sizeof(CV_PRIV));
    pp->vals = NULL;
    if ('\0' == *dp->fname) {
	pp->tpl.type = INVALID;
    }
    else {
	register int	i;
	register int	j;
	FILTER_VAL	val;

	if (
	    (NULL != (s = read_template(dp->fname, &pp->tpl))) ||
	    (NULL != (s = cook_template(&pp->tpl, dp->tscale,
					&pp->vals, &pp->nvals, &pp->tzero))) ||
	    (0 >= pp->nvals)
	) {
	    Nfree(s);
	    destroy_cv_priv(pp);
	    return(TRUE);
	}
	if (0 >= pp->nvals) {
	    pp->tpl.type = INVALID;
	}
	else {
	    /*
	     * divide by tscale (to keep integral constant)
	     */
	    for(i=0; i<pp->nvals; i++) {
		pp->vals[i] /= dp->tscale;
	    }
	    /*
	     * flip the template (so we can use ordinary dot product)
	     */
	    for(
		i=0, j=pp->nvals-1;
		i < j;
		i++, j--
	    ) {
		val = pp->vals[i];
		pp->vals[i] = pp->vals[j];
		pp->vals[j] = val;
	    }
	    pp->tzero = pp->nvals - pp->tzero - 1;
	}
    }
    if (NULL != fp->f_priv) destroy_cv_priv((CV_PRIV *) fp->f_priv);
    fp->f_priv = (VOIDST) pp;
    Nfree(fp->f_name);
    sprintf(lbuf,
	    "Convolution:\ntemplate file = \"%s\", tscale = %.15g",
	    data->fname, data->tscale);
    fp->f_name = estrdup(lbuf);
    clear_caches(fp);
    return(FALSE);
}

static Bool
nsrc_cv(fp)
FILTER		*fp;
{
    CV_DATA		*dp = &fp->f_data->cv_data;
    CV_PRIV		*pp = (CV_PRIV *) fp->f_priv;

    nsrc_generic(fp);
    if ((NULL == pp) || (INVALID == pp->tpl.type)) return(FALSE);
    if (NULL != fp->f_src) {
	fp->f_tmin = fp->f_src->f_tmin + pp->tzero;
	fp->f_tmax = fp->f_src->f_tmax - pp->nvals + pp->tzero;
    }
    return(FALSE);
}

static Bool
kill_cv(fp)
FILTER		*fp;
{
    CV_DATA		*dp = &fp->f_data->cv_data;

    delete_filter(fp);
    Nfree(fp->f_name);
    Nfree(dp->fname);
    Nfree(fp->f_data);
    destroy_cv_priv((CV_PRIV *) fp->f_priv);
    fp->f_priv = NULL;
    kill_cache(&fp->f_cv);
    kill_cache(&fp->f_cm);
    fp->f_type = F_INACTIVE;
    return(FALSE);
}

static void
destroy_cv_priv(pp)
CV_PRIV		*pp;
{
    if (NULL == pp) return;
    switch(pp->tpl.type) {
    case INTERPOLATED:
	Nfree(pp->tpl.templ_inter.points);
	break;

    default:
	break;
    }
    Nfree(pp->vals);
    free(pp);
}
