/* lalib.c -- various utilities from L Avery's library			*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

/* filelen.c -- return length of an open file in bytes
 *
 * long	nbytes;
 * FILE	*file;
 * nbytes = filelen(file);
 *
 * file points to an open stream.  It returns the size of the file in
 * bytes, or (long) EOF if there is an error.
 */
#include <stdio.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/stat.h>
#include "config.h"
#include "config.h"
#include "readline.h"

char	*fgets();
int	fclose();
FILE	*fopen();
int	strlen();

int	errno;
extern	char	*PROGNAME;
extern	int	debug;			/* debugging on?		*/
extern	int	indent;			/* indent lines by		*/

long
filelen(f)
FILE	*f;
{
    struct stat	buf;

    if (EOF == fstat(fileno(f), &buf)) return((long) EOF);
    return(buf.st_size);
}

/* error.c -- report an error, then die
 * 
 * char	*string;
 * error(string)
 *
 * error writes the specified error message on stderr, preceded by the
 * program name (obtained from the global char *PROGNAME) and followed
 * by a newline, then exits with status ERROR.
 */
void
error(f)
char	*f;
{
    fprintf(stderr, "%s: %s\n", PROGNAME, f);
    if (0 != errno) perror(PROGNAME);
    exit(ERROR);
}

/* error1.c -- report an error, then die
 * 
 * char	*format;
 * char *string;
 * error(format, string)
 *
 * error1 is called like printf, but the format must have only one %s
 * in it.  (Actually, it will work on most systems with any single
 * reference.)  It writes the specified error message on stderr,
 * preceded by the program name (obtained from the global char
 * *PROGNAME) and followed by a newline, then exits with status ERROR.
*/
void
error1(f, s)
char	*f;
char	*s;
{
    char	lbuf[LLEN];

    sprintf(lbuf, f, s);
    fprintf(stderr, "%s: %s\n", PROGNAME, lbuf);
    if (0 != errno) perror(PROGNAME);
    exit(ERROR);
}

/* dprintf.c -- print out debugging info
 *
 * char	*message;
 * char	*arg;
 * dprintf(message);
 * dprintf1(message, arg);
 *
 * dprintf and dprintf1 first check whether the global debug is TRUE;
 * if not, they exit without doing anything.  If it is true, 2*indent
 * blanks are written to stderr (indent is another global int), and
 * then printf is called with the arguments that were passed to
 * dprintf or dprintf1.  (The varargs version is nicer, but harder to
 * port.) 
 */
void
dprintf(f)
char	*f;
{
    register int	i;

    if (!debug) return;
    for(i=0; i<indent; i++) {
	fprintf(stderr, "  ");
    }
    fprintf(stderr, f);
}

void
dprintf1(f, a)
char	*f;
char	*a;
{
    register int	i;

    if (!debug) return;
    for(i=0; i<indent; i++) {
	fprintf(stderr, "  ");
    }
    fprintf(stderr, f, a);
}

/* ealloc.c -- allocate memory, die if error
 * 
 * unsigned	size;
 * char		*space;
 * space = ealloc(size);
 * 
 * ealloc allocates memory just like the library routine malloc,
 * except that if there is no memory, ealloc executes error("no
 * memory").
 */
VOIDST
ealloc(s)
unsigned	s;
{
    register VOIDST	a;

    if (NULL == (a = (VOIDST) malloc(s))) error("no memory");
    return(a);
}

/* erealloc.c -- reallocate memory, die if error
 *
 * char		*oldbuf;
 * char		*newbuf;
 * unsigned	size;
 * newbuf = erealloc(oldbuf, size);
 */
VOIDST
erealloc(b, s)
VOIDST		b;
unsigned	s;
{
    if (NULL == (b = (VOIDST) realloc(b, s)))
	error("no memory");
    return(b);
}

/* estrdup.c -- duplicate a string, die if error
 * 
 * char	*string;
 * char	*newstring;
 * newstring = estrdup(string);
 * 
 * estrdup returns a copy of its argument, located in memory
 * allocated from the heap.  If it is unable to allocate the
 * necessary memory, estrdup executes error("no memory"). 
 */
char *
estrdup(s)
char	*s;
{
    register char	*t;

    t = ealloc(strlen(s)+1);
    strcpy(t, s);
    return(t);
}

/* igets.c -- read a line of input
 * 
 * char	lbuf[LLEN];
 * FILE	*current;
 * FNL	*names;
 * if (NULL == igets(lbuf, LLEN, &current, &names)) error...
 * 
 * current points to a currently open file.  (If no file
 * is open, current may be NULL.)  names points to a list
 * of names of files.  The first name in the list in
 * the name of the currently open file.  Subsequent names
 * identify files from which input is to be read when
 * current is exhausted.  The name "-" is taken to mean
 * stdin.  igets reads a line, if necessary
 * closing current and opening the next file in the list.
 * current and names are updated if this is necessary.
 * igets returns NULL if there is an error or endfile on
 * current and there are no more files, or the next file
 * can't be opened.
 */
char *
igets(s, n, f, fl)
char	*s;
int	n;
FILE	**f;
FNL	**fl;
{
    FNL			*fp;

    if (NULL == *f) {
	if (NULL == *fl) return(NULL);
	if (0 == strcmp("-", (*fl)->f_n)) {
	    *f = stdin;
	}
	else {
	    if (NULL == (*f = fopen((*fl)->f_n, "r")))
		error1("unable to open %s", (*fl)->f_n);
	}
    }
    while(NULL == fgets(s, n, *f)) {
	if (EOF == fclose(*f))
	    error1("unable to close %s", (*fl)->f_n);
	fp = *fl;
	*fl = fp->f_nxt;
	free(fp->f_n);
	free(fp);
	*f = NULL;
	if (NULL == *fl) return(NULL);
	if (NULL == (*f = fopen((*fl)->f_n, "r")))
	    error1("unable to open %s", (*fl)->f_n);
    }
    return(s);
}

/* ieof -- check for endfile on input
 * 
 * FILE	*current;
 * FNL	*names;
 * int	atend;
 * atend = ieof(current, names);
 * 
 * current and names are as for igets (but note that ieof
 * expects current and names, not their addresses).  ieof
 * returns TRUE iff the end of file current has been reached
 * and there are no more files.
 * 
 * ieof uses the cumbersome method of attempting to read
 * the next char, because feof doesn't work properly on
 * the VAX.  On other machines, simpler methods should
 * work.
 */
int
ieof(f, fl)
FILE	*f;
FNL	*fl;
{
    register int	c;

    if (NULL == f) {
	return(NULL == fl);
    }
    else {
#ifdef	vaxc
	if (
	    ((NULL == fl) || (NULL == fl->f_nxt)) &&
	    (EOF == (c = getc(f)))
	) return(TRUE);
	ungetc(c, f);
#else	/* vaxc */
	if (
	    ((NULL == fl) || (NULL == fl->f_nxt)) &&
	    feof(f)
	) return(TRUE);
#endif	/* vaxc */
	return(FALSE);
    }
}


/* readline.c -- read a line of unlimited length
 * 
 * char	*line;
 * FILE	*cif;
 * FNL	*inf;
 * int	size;
 * size = readline(&line, &cif, &inf);
 * 
 * cif points to a currently open file (or is NULL if there
 * is none).  inf is a list of names of files to be read
 * when cif is exhausted.  readline reads a line, putting
 * it in space allocated from the heap.  A pointer to the
 * line is placed in line.  The terminal newline is replaced
 * with a '\0'.  The number of chars in the line (not including
 * the terminator) is returned.
 */
int
readline(lp, ci, il)
char		**lp;
FILE		**ci;
FNL		**il;
{
    char		*b;		/* buffer			*/
    int			bs;		/* buffer size			*/
    char		*s;		/* current char pointer		*/
    int			sp;		/* space left			*/
    int			sl;

    b = (char *) ealloc(MINBUF);
    bs = MINBUF;
    s = b;
    sp = bs;
    for(;;) {
	if (NULL == igets(s, sp, ci, il)) return(EOF);
	sl = strlen(s);
	s += sl;
	sp -= sl;
	if ('\n' == s[-1]) break;
	if (1 >= sp) {
	    bs += MINBUF;
	    b = (char *) erealloc(b, bs);
	    sp += MINBUF;
	    s = b + bs - sp;
	}
    }
    s[-1] = '\0';
    *lp = b;
    return(bs - sp - 1);
}

#ifdef	HAVE_SYS_TIME_H
#include <sys/time.h>
/* has_input -- test if input available on stream
 *
 * int	fd;
 * int	input;
 * input = has_input(stream);
 *
 * has_input return TRUE if input is available on file descriptor fd
 * (meaning read will not block), FALSE otherwise.
 */
int
has_input(fd)
int	fd;
{
    fd_set		fz;
    fd_set		fs;
    struct timeval	tout;

    tout.tv_sec = 0;
    tout.tv_usec = 0;
    FD_ZERO(&fz);
    FD_ZERO(&fs);
    FD_SET(fd, &fs);
    return(0 < sselect(fd+1, &fs, &fz, &fz, &tout));
}

/* sselect -- like select, but repeat if interrupted			*/
int
sselect(width, rs, ws, xs, tv)
int		width;
fd_set		*rs, *ws, *xs;
struct timeval	*tv;
{
    register int	ret;

    do {
	errno = 0;
	ret = select(width, rs, ws, xs, tv);
    } while((EOF == ret) && (EINTR == errno));
    return(ret);
}
#endif	/* HAVE_SYS_TIME_H */

/* append_string -- append to a string, using erealloc			*/
char *
append_string(base, cat, len)
char		*base;
char		*cat;
int		*len;
{
    char		*s;

    s = (char *) erealloc(base, strlen(cat) + *len + 1);
    strcpy(s + *len, cat);
    *len += strlen(cat);
    return(s);
}
