/* run.c -- run a program
 *
 * int		pipes[3];
 * char		*command;
 * pid_t	pid;
 * pid = run(command, pipes);
 *
 * run execs the command in the string command.  If successful, it
 * returns the pid of the new process, otherwise it returns EOF.
 * pipes[0] is a file descriptor open for output, connected to the
 * standard input of the new process.  pipes[1] and pipes[2] are file
 * descriptors open for input connected to the standard output and the
 * standard error of the new process.
 */
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include "config.h"

#ifdef	__STDC__
caddr_t	getenv(caddr_t);
#else	/* __STDC__ */
caddr_t	getenv();
#endif	/* __STDC__ */

#define	DEF_SHELL	"/bin/sh"	/* default shell		*/
#define	IN	0			/* which pipe end is which	*/
#define	OUT	1

pid_t
run(cmd, pipes)
char		*cmd;
int		pipes[];
{
    char		lbuf[LLEN];
    int			ifd[2];		/* pipes			*/
    int			ofd[2];
    int			efd[2];
    pid_t		rpid;

    if (EOF == pipe(ifd)) return(EOF);
    if (EOF == pipe(ofd)) {
	close(ifd[0]);
	close(ifd[1]);
	return(EOF);
    }
    if (EOF == pipe(efd)) {
	close(ifd[0]);
	close(ifd[1]);
	close(ofd[0]);
	close(ofd[1]);
	return(EOF);
    }
    if (0 == (rpid = VFORK())) {
	char		*shell;

	/*
	 * child process: set up file descriptors and exec
	 */
	close(ifd[OUT]); dup2(ifd[IN], 0); close(ifd[IN]);
	close(ofd[IN]); dup2(ofd[OUT], 1); close(ofd[OUT]);
	close(efd[IN]); dup2(efd[OUT], 2); close(efd[OUT]);
	if (NULL == (shell = getenv("SHELL"))) shell = DEF_SHELL;
	if (EOF == execl(shell, shell, "-c", cmd, NULL)) {
	    sprintf(lbuf, "unable to exec %s -c \"%s\"", shell, cmd);
	    error(lbuf);
	}
    }
    /*
     * parent process
     */
    if (EOF == rpid) {
	close(ifd[0]); close(ifd[1]);
	close(ofd[0]); close(ofd[1]);
	close(efd[0]); close(efd[1]);
	return(rpid);
    }
    close(ifd[IN]); pipes[0] = ifd[OUT];
    close(ofd[OUT]); pipes[1] = ofd[IN];
    close(efd[OUT]); pipes[2] = efd[IN];
    return(rpid);
}

/* run1 -- run a program
 *
 * int		*pipe;
 * char		*command;
 * pid_t	pid;
 * pid = run1(command, pipe);
 *
 * run1 is just like run, except that only stdin of the new process is
 * attached to a pipe: stdout and stderr are left unchanged.
 */
pid_t
run1(cmd, pipes)
char		*cmd;
int		*pipes;
{
    char		lbuf[LLEN];
    int			ifd[2];		/* pipes			*/
    pid_t		rpid;

    if (EOF == pipe(ifd)) return(EOF);
    if (0 == (rpid = VFORK())) {
	char		*shell;

	/*
	 * child process: set up file descriptors and exec
	 */
	close(ifd[OUT]); dup2(ifd[IN], 0); close(ifd[IN]);
	if (NULL == (shell = getenv("SHELL"))) shell = DEF_SHELL;
	if (EOF == execl(shell, shell, "-c", cmd, NULL)) {
	    sprintf(lbuf, "unable to exec %s -c \"%s\"", shell, cmd);
	    error(lbuf);
	}
    }
    /*
     * parent process
     */
    if (EOF == rpid) {
	close(ifd[0]); close(ifd[1]);
	return(rpid);
    }
    close(ifd[IN]);
    pipes[0] = ifd[OUT];
    return(rpid);
}

/* run2 -- run a program
 *
 * int		pipes[2]
 * char		*path;
 * char		**argv;
 * pid_t	pid;
 * pid = run2(command, argv, pipes);
 *
 * run2 is identical to run, except for two things.  First, only stdin
 * and stdout of the new process are redirected: stderr is left
 * unchanged.  Second, the command to execute is passed as a path and
 * argument list.  These arguments are passed directly to execvp.
 */
pid_t
run2(path, argv, pipes)
char		*path;
char		**argv;
int		pipes[];
{
    char		lbuf[LLEN];
    int			ifd[2];		/* pipes			*/
    int			ofd[2];
    pid_t		rpid;

    if (EOF == pipe(ifd)) return(EOF);
    if (EOF == pipe(ofd)) {
	close(ifd[0]);
	close(ifd[1]);
	return(EOF);
    }
    if (0 == (rpid = VFORK())) {
	/*
	 * child process: set up file descriptors and exec
	 */
	close(ifd[OUT]); dup2(ifd[IN], 0); close(ifd[IN]);
	close(ofd[IN]); dup2(ofd[OUT], 1); close(ofd[OUT]);
	if (EOF == execvp(path, argv)) {
	    sprintf(lbuf, "unable to exec %s", path);
	    error(lbuf);
	}
    }
    /*
     * parent process
     */
    if (EOF == rpid) {
	close(ifd[0]); close(ifd[1]);
	close(ofd[0]); close(ofd[1]);
	return(rpid);
    }
    close(ifd[IN]); pipes[0] = ifd[OUT];
    close(ofd[OUT]); pipes[1] = ofd[IN];
    return(rpid);
}
