/********************************************************************
 *         This example code is from the book:
 *
 *           Motif Debuggin and Performance Tuning
 *            ISBN 0-13-147984-9
 *         by
 *           Douglas Young
 *           Prentice Hall, 1995
 *
 *         Copyright 1994 by Prentice Hall
 *         All Rights Reserved
 *
 *  Permission to use, copy, modify, and distribute this software for 
 *  any purpose except publication and without fee is hereby granted, provided 
 *  that the above copyright notice appear in all copies of the software.
 * *****************************************************************************/

/***************************************************
 * formattachments.c: Tests for form attachments
 ****************************************************/
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <stdio.h>
#include "wlintP.h"

static void TestPosition ( Widget        w, 
                           int           fractionBase, 
                           unsigned char side, 
                           char         *sideName, 
                           int           position )
{
   /*
    * Make sure any positional attachment is within
    * a reasonable range.
    */

    if ( side  == XmATTACH_POSITION && 
         position > fractionBase )     
        fprintf ( stderr,              
                  "%ss %s position is too large\n",
                  XtName ( w ), sideName );
}

static void TestOppositeAttachment( Widget        w,
                                    unsigned char side,
                                    char         *sideName,
                                    Widget        widget )
{
   /*
    * If attached to a widget, make sure the widget
    * has been specified properly.
    */

    if ( side == XmATTACH_OPPOSITE_WIDGET )           
    {                                                
        if ( widget == NULL )                        
            fprintf ( stderr,                        
                      "%s attached to NULL %s widget\n", 
                      XtName ( w ), sideName );      
    }
}

static void TestAttachment ( Widget        w,
                             unsigned char attachment,
                             char         *sideName,
                             Widget        widget,
                             char         *otherside ) 
{
   /*
    * If attached to a widget, make sure the widget
    * has been specified properly, and that the attachment
    * point is not also attached.
    */

    if ( attachment == XmATTACH_WIDGET )    
    {   
        unsigned char attach;

        if ( widget == NULL ) 
            fprintf ( stderr,
                      "%s attached to a NULL %s widget\n",
                      XtName ( w ), sideName );           
        else      
        {       
            XtVaGetValues ( widget,
                            otherside, &attach,
                            NULL ); 
            if ( attach != XmATTACH_NONE )   
            {             
                fprintf ( stderr,                         
                         "%s is attached on the %s to %s ",
                         XtName ( w ), sideName, 
                         XtName ( widget ) );
               fprintf ( stderr,
               "whose %s resource is not set to attach_none\n",
                        otherside ); 
            }
        }   
    }                                               
}

/*
 * The WL test function.
 */

void FormAttachmentsTest ( Widget w, ActionType action )
{
    unsigned char top, bottom, left, right;
    int           topPosition,  bottomPosition, 
                  leftPosition, rightPosition;
    int           fractionBase;
    Widget        topWidget,  bottomWidget,
                  leftWidget, rightWidget;

    if ( action == INIT ||
         action == POSTOP )
        return;

   /*
    * Be sure this widget has a parent. No shells.
    */

    if ( !XtParent ( w ) || XtIsShell ( w ) )
        return;

   /*
    * See if the parent is a form widget.
    */

    if ( !XmIsForm ( XtParent ( w ) ) )
        return;

   /*
    * Get all attachment values from the widget and parent.
    */

    XtVaGetValues ( XtParent ( w ),
                    XmNfractionBase, &fractionBase,
                    NULL );
  
    XtVaGetValues ( w,
                    XmNleftAttachment,   &left,
                    XmNrightAttachment,  &right,
                    XmNtopAttachment,    &top,
                    XmNbottomAttachment, &bottom,
                    XmNtopPosition,      &topPosition,
                    XmNbottomPosition,   &bottomPosition,
                    XmNleftPosition,     &leftPosition,
                    XmNrightPosition,    &rightPosition,
                    XmNtopWidget,        &topWidget,
                    XmNbottomWidget,     &bottomWidget,
                    XmNleftWidget,       &leftWidget,
                    XmNrightWidget,      &rightWidget, 
                    NULL );

   /*
    * Check position attachments
    */

    TestPosition ( w, fractionBase, left,
                   "left", leftPosition );
    TestPosition ( w, fractionBase, right, 
                  "right", rightPosition );
    TestPosition ( w, fractionBase, top, 
                   "top", topPosition );
    TestPosition ( w, fractionBase, bottom, 
                   "bottom", bottomPosition );
   /*
    * Check that left/top positions are less than 
    * right/bottom positions.
    */

    if ( left  == XmATTACH_POSITION && 
         right == XmATTACH_POSITION &&
         leftPosition >= rightPosition )
        fprintf ( stderr, 
                  "invalid left/right positions: %s\n",
                  XtName ( w ) );

    if ( top    == XmATTACH_POSITION &&
         bottom == XmATTACH_POSITION &&
         bottomPosition <= topPosition )
       fprintf ( stderr, 
                 "Invalid top/bottom positions: %s\n",
                 XtName (  w ) );

   /*
    * Test each widget attachment
    */

    TestAttachment ( w, top,     "top", 
                     topWidget,    XmNbottomAttachment );
    TestAttachment ( w, bottom,  "bottom",
                     bottomWidget, XmNtopAttachment );
    TestAttachment ( w, left,   "left",
                     leftWidget,   XmNrightAttachment );
    TestAttachment ( w, right,  "right",
                     rightWidget,  XmNleftAttachment );
   /*
    * Test each opposite widget attachment
    */

    TestOppositeAttachment ( w, top,   
                             "top",    topWidget );
    TestOppositeAttachment ( w, bottom, 
                             "bottom", bottomWidget );
    TestOppositeAttachment ( w, left, 
                             "left",   leftWidget );
    TestOppositeAttachment ( w, right, 
                             "right",  rightWidget );
}

