/********************************************************************
 *         This example code is from the book:
 *
 *           Motif Debuggin and Performance Tuning
 *            ISBN 0-13-147984-9
 *         by
 *           Douglas Young
 *           Prentice Hall, 1995
 *
 *         Copyright 1994 by Prentice Hall
 *         All Rights Reserved
 *
 *  Permission to use, copy, modify, and distribute this software for 
 *  any purpose except publication and without fee is hereby granted, provided 
 *  that the above copyright notice appear in all copies of the software.
 * *****************************************************************************/

/**************************************
 * wlint.c: Driver for WL framework.
 *************************************/
#include "wlintP.h"
#include <X11/IntrinsicP.h>
#include <Xm/Xm.h>
#include <Xm/MenuShell.h>
#include <Xm/Display.h>
#include <stdio.h>

extern void WLRegisterTests ( void );
static void RunTest ( Widget w,
                      TestType type,
                      WLTestProc func );
static void InitializeTests ( void );

static TestData *testList = NULL;
static int       numTests = 0;

void _WLRunTests ( Widget w, const char *str )
{
    int      i;
    char    *opt = NULL;
    Boolean  verbose = FALSE;
    char    *name, *appClass;
  
   /*
    * Only alow this test to be run for shell widgets.
    * Disallow the XmDisplay shell used for drag and drop
    * and menu shells as starting points.
    */

    if ( !XtIsShell ( w ) || 
         XmIsDisplay ( w ) ||
         XmIsMenuShell ( w) )
        	return;

   /*
    * Get the name of the application to allow resources
    * to be retrieved.
    */

    XtGetApplicationNameAndClass ( XtDisplay ( w ),
                                   &name, &appClass );

   /*
    * Find out if the tests should be run, and if verbose
    * mode should be turned on.
    */

    opt = XGetDefault ( XtDisplay ( w ), name, "WLint" );

    if ( opt == NULL ||
         strcasecmp ( opt, "on" ) != 0 )
      return;

    opt = XGetDefault ( XtDisplay ( w ), name, "WLverbose" );

    if ( opt != NULL &&
         strcasecmp ( opt, "true" ) == 0 )
       	verbose = TRUE;

    if ( verbose )
    {
        	/*
        	 * Set up the tests. Print any message provided.
	         */
     
        	fprintf ( stderr, "\n\n" );
        	fprintf ( stderr, "*****************************\n" );
        	if ( str )            
	            fprintf ( stderr, "Starting test %s\n", str );
        	fprintf ( stderr, "*****************************\n" );
    }

   /*
    * Initialize data structures, and install functions
    * to be executed.
    */

    InitializeTests();
    WLRegisterTests();

   /*
    * Report each test as it begins, and fire off the test.
    */

    for ( i = 0; i < numTests; i++ )
    {
        ( *testList[i].func ) ( NULL, INIT );

        	if ( verbose )
        	{
	             fprintf ( stderr, "\n\n" );
	             fprintf ( stderr, "**********************\n" );
	             if ( testList[i].name )  
                 		fprintf ( stderr, "TEST: %s\n",
                           testList[i].name );
	             fprintf ( stderr, "***********************\n" );
        	}

        RunTest ( w, testList[i].type, testList[i].func );

       /*
        * Run each test function one last time. 
        */

        ( *testList[i].func ) ( NULL, POSTOP );
    }
}

static void RunTest ( Widget     w,
                      TestType   type,
                      WLTestProc func)
{
    unsigned int i;
        
    if ( !w )  /* End the recursive test */
        return;
    /*
     * Compare the type for which each test has been 
     * with the type of the current widget, and execute
     * the test function, as appropriate.
     */

    switch ( type )
    {
      case SHELL:
        if ( XtIsShell ( w ) )
            ( *func )( w, RUNTEST );
        break;
      case MANAGER:
        if ( XmIsManager ( w ) )
            ( *func )( w, RUNTEST );        
        break;        
      case PRIMITIVE:
        if ( XmIsPrimitive ( w ) || XmIsGadget ( w ) )
            ( *func )( w, RUNTEST );                
        break;    
      case GADGET:
        if ( XmIsGadget ( w ) )
            ( *func )( w, RUNTEST );                
        break;        
      case ALL:
        ( *func )( w, RUNTEST );                        
        break;        
      case JUSTONCE:
        ( *func )( w, RUNTEST );
        return;
    }

   /*
    * If this widget can have children, call RunTest()
    * recursively for each child.
    */

    if ( XtIsComposite ( w ) ) 
    {
	        CompositeWidget cw = ( CompositeWidget ) w;

        for ( i = 0; i < cw->composite.num_children; i++ ) 
            RunTest ( cw->composite.children[i], type, func );
    }

   /*
    * Any widget (but not gadgets) can have popup
    * children. Run the test for these as well.
    */

    if ( XtIsWidget ( w ) )
        for ( i = 0; i < w->core.num_popups; i++ ) 
        {
            Widget child = w->core.popup_list[i];
            RunTest ( child, type, func );        
        }
}

static void InitializeTests (void )
{
    numTests = 0;

    if ( testList )
        XtFree ( ( char * ) testList );

    testList = NULL;
}


void WLAddTest ( char *name, TestType type, WLTestProc func )
{
    numTests++;
    testList = ( TestData* ) 
                   XtRealloc ( ( char * ) testList,
                                  sizeof ( TestData ) * numTests );
    if ( name )
        testList[numTests-1].name = XtNewString ( name );
    else
        testList[numTests-1].name = NULL;

    testList[numTests-1].type = type;
    testList[numTests-1].func = func;
}

