/********************************************************************
 *         This example code is from the book:
 *
 *           Motif Debuggin and Performance Tuning
 *            ISBN 0-13-147984-9
 *         by
 *           Douglas Young
 *           Prentice Hall, 1995
 *
 *         Copyright 1994 by Prentice Hall
 *         All Rights Reserved
 *
 *  Permission to use, copy, modify, and distribute this software for 
 *  any purpose except publication and without fee is hereby granted, provided 
 *  that the above copyright notice appear in all copies of the software.
 * *****************************************************************************/

/*********************************************************
 * fractal.c: Main window and rendering code for fractal.
 *********************************************************/
#include <Xm/Xm.h> 
#include <Xm/DrawingA.h> 
#include <Xm/MainW.h> 
#include <Xm/RowColumn.h>
#include <Xm/CascadeB.h>
#include <Xm/PushB.h>
#include <stdlib.h>
#include "fractal.h"
#include "TimerTools.h"

extern ShowPreferences ( Widget parent, ImageData *data );

static void InitData ( Widget w, ImageData *data );
static void ResizeCallback ( Widget    w, 
                             XtPointer clientData,
                             XtPointer callData );
static void RedisplayCallback ( Widget    w, 
                                XtPointer clientData,
                                XtPointer callData );
static void ShowPreferencesCallback ( Widget    w, 
                                      XtPointer clientData,
                                      XtPointer callData );
static void QuitCallback ( Widget    w, 
                           XtPointer clientData,
                           XtPointer callData );
static void FlushBuffer ( Widget w, Pixmap pix, GC gc, int size );
static void Render ( Widget  w, Pixmap pix, GC gc, 
                     XPoint *points, 
                     int     numPoints , int size);
static void BufferPoint ( Widget w, Pixmap pix, 
                          GC gc,   Pixel color, 
                          int x,   int y , int size);
static void InitBuffer();

/*
 * Function called from preference module
 */

void CreateImage ( ImageData *data );

/*
 * Functions that support rubberband selection of a region.
 */

static void StartRubberBand ( Widget    w,
                              XtPointer clientData,
                              XEvent   *event,
                              Boolean  *flag );

static void TrackRubberBand ( Widget    w, 
                              XtPointer clientData,
                              XEvent   *event,
                              Boolean  *flag );

static void EndRubberBand ( Widget    w,
                            XtPointer clientData,
                            XEvent   *event,
                            Boolean   *flag );

static int startX, startY, lastX, lastY;

Widget CreateMenu ( Widget parent, ImageData *data ) 
{
    Widget menu, pref, quit;
    Widget cascade, submenu;    
    int i;

    menu = XmCreateMenuBar ( parent, "menu", NULL, 0 );
    XtManageChild ( menu );

    submenu = XmCreatePulldownMenu ( menu, 
                                     "Application", 
                                     NULL, 0 );

    cascade = XtVaCreateManagedWidget ( "Application",
                                  xmCascadeButtonWidgetClass,
                                  menu, 
                                  XmNsubMenuId, submenu,
                                  NULL  );

    pref = XtCreateManagedWidget ( "Preferences",
                                   xmPushButtonWidgetClass,
                                   submenu, NULL, 0 );

    quit = XtCreateManagedWidget ( "Quit",
                                   xmPushButtonWidgetClass,
                                   submenu, NULL, 0 );

    XtAddCallback ( pref, XmNactivateCallback,
                    ShowPreferencesCallback, 
                    (XtPointer) data);
    XtAddCallback ( quit, XmNactivateCallback, 
                    QuitCallback, NULL );    
    
    return ( menu );
}

void main ( int argc, char **argv ) 
{
    Widget       shell, canvas, mainWindow, menu;
    XtAppContext app;
    ImageData    data;

    ReportTime ( "Starting", TRUE );
    shell = XtAppInitialize ( &app, "Fractal", NULL, 0, 
                              &argc, argv, NULL, NULL, 0 );

    mainWindow =
             XtCreateManagedWidget ( "mainWindow", 
                                     xmMainWindowWidgetClass,
                                     shell, NULL, 0 );
   /*
    * Create the widget in which to display the fractal.
    */
    
    canvas = XtCreateManagedWidget ( "canvas", 
                                     xmDrawingAreaWidgetClass,
                                     mainWindow, NULL, 0 );
   /*
    * Create the GCs needed by the fractal program.
    */

    InitData ( canvas, &data );  

   /*
    * Create the menu bar and set up the window layout.
    */

    menu = CreateMenu ( mainWindow, &data );

    XtVaSetValues ( mainWindow,
                    XmNmenuBar,    menu,
                    XmNworkWindow, canvas, 
                    NULL );

   /*
    * Add callbacks to handle resize and exposures.
    */

    XtAddCallback ( canvas, XmNexposeCallback,
                    RedisplayCallback, ( XtPointer ) &data ); 
    XtAddCallback ( canvas, XmNresizeCallback,
                    ResizeCallback, ( XtPointer )  &data ); 

  /*
   * Add event handlers to track the mouse and allow
   * the user to select a region with rubberband rectangle.
   */

   XtAddEventHandler ( canvas, ButtonPressMask, FALSE,
                       StartRubberBand, &data );
   XtAddEventHandler ( canvas, ButtonMotionMask, FALSE,
                       TrackRubberBand, &data );
   XtAddEventHandler ( canvas, ButtonReleaseMask, FALSE,
                       EndRubberBand, &data );

   XtRealizeWidget ( shell );
    
   WaitForWindow ( shell );
   HandleEvents ( shell );
   ReportTime ( "Up", TRUE );

   XtAppMainLoop ( app );
}

static void ShowPreferencesCallback ( Widget    w,  
                                      XtPointer clientData, 
                                      XtPointer callData ) 
{
    ImageData *data = ( ImageData * ) clientData;
    
   /* Call external function to display pereference panel. */

    ShowPreferences ( XtParent ( w ), data );
}

static void QuitCallback ( Widget    w,  
                           XtPointer clientData, 
                           XtPointer callData ) 
{
    exit ( 0 );
}

static void InitData ( Widget w, ImageData *data ) 
{
    XGCValues values;

    /*
     * Get the size of the drawing area.
     */
    
     XtVaGetValues ( w,  
                     XmNwidth,      &data->width, 
                     XmNheight,     &data->height, 
                     XmNbackground, &values.foreground,
                     NULL );

     data->canvas = w;
     data->depth       = 200;
     data->origin.real = -1.4;
     data->origin.imag = 1.0;
     data->range       = 2.0;
     data->maxDistance = 4.0;

    /*
     * Find out how many colors we have to work with, and  
     * create a default, writable, graphics context.
     */
    
     data->ncolors = 
         XDisplayCells ( XtDisplay ( w ), 
                         XDefaultScreen ( XtDisplay ( w ) ) );
    
     data->gc = 
             XCreateGC ( XtDisplay ( w ),
                         DefaultRootWindow ( XtDisplay ( w )),
                         NULL, NULL ); 
   /*
    * Create a second GC set to XOR mode to use in the 
    * rubberbanding functions that select a region.
    */

    values.function = GXxor;

    data->xorGC = XtGetGC ( w, 
                            GCForeground | GCFunction, 
                            &values );

    /*
     *  Initialize the pixmap to NULL.
     */
    
     data->pixmap = NULL;
}

static void StartRubberBand ( Widget    w,
                              XtPointer clientData,
                              XEvent   *event,
                              Boolean  *flag )
{
    lastX = startX = event->xbutton.x;
    lastY = startY = event->xbutton.y;
}
static void TrackRubberBand ( Widget    w, 
                              XtPointer clientData,
                              XEvent   *event,
                              Boolean  *flag )
{
    int height;    
    ImageData *data = ( ImageData* ) clientData;

   /*
    * If a non-zero sized rectangle has been previously drawn,
    * erase it by drawing again in XOR mode.
    */

    if ( lastX - startX > 0 || lastY - startY > 0 )
        XDrawRectangle ( XtDisplay ( w ), XtWindow ( w ),
                         data->xorGC,
                         startX, startY, 
                         lastX - startX, lastY - startY );
   /*
    * Update the last point. Force an aspect ratio that 
    * matches the shape of the window.
    */
        
    lastX  =  event->xmotion.x;

    height = data->height * ( lastX - startX ) / data->width;

    lastY  =  startY + height;

    if ( lastX < startX )
        lastX = startX;

    if ( lastY < startY )
        lastY = startY;    

   /*
    * Draw rectangle in XOR mode so it can be easily erased.
    */

    XDrawRectangle ( XtDisplay ( w ), XtWindow ( w ), data->xorGC,
                     startX, startY, 
                     lastX - startX, lastY - startY );
}
static void EndRubberBand ( Widget    w,
                            XtPointer clientData,
                            XEvent   *event,
                            Boolean   *flag )
{
    int height;
    ImageData *data = ( ImageData* ) clientData;

   /*
    * If a non-zero sized rectangle has been previously
    * drawn, erase it by drawing again in XOR mode.
    */

    if ( lastX - startX > 0 || lastY - startY > 0 )
        XDrawRectangle ( XtDisplay ( w ), XtWindow ( w ), 
                         data->xorGC, startX, startY,
                         lastX - startX, lastY - startY );

   /*
    * Update the last point. Force an aspect ratio that 
    * matches the shape of the window.
    */
        
    lastX  = event->xmotion.x;
    height = data->height * ( lastX - startX ) / data->width;

    lastY  =  startY + height;
   
   /*
    * Unless a non-zero sized region was selected, return.
    */

    if ( lastX <= startX || lastY <= startY )
        return;

   /*
    * Convert the pixel-based corrdinates to the real 
    * coordinates used to compute the fractal image.
    */

    data->origin.real += data->range * 
                        ( double ) startX / ( double ) data->width;

    data->origin.imag -= data->range *
                        ( double )  startY / ( double ) data->height;

    data->range = data->range * 
               ( double ) ( lastX - startX ) / ( double ) data->width;

   /*
    * Create a new image, based on the selected range and
    * origin. Keep the preference panel in sync.
    */

    CreateImage ( data );
    UpdatePreferences ( data );
}

extern Boolean CreateImageWorkProc ( XtPointer clientData );
    
void CreateImage ( ImageData *data ) 
{
    static XtWorkProcId id = NULL;
    Widget w = data->canvas;

    if ( id )
        XtRemoveWorkProc ( id );

    data->restart = TRUE;
    
    id = XtAppAddWorkProc ( XtWidgetToApplicationContext ( w),
                            CreateImageWorkProc, data );
}

Boolean CreateImageWorkProc ( XtPointer clientData )
{
    ImageData    *data = (ImageData*) clientData;
    Widget        w = data->canvas;
    static int    x, y, iteration;
    static double rangeByWidth, rangeByHeight, *xconstants;
    static int    height;
    static int    width;
    GC            gc     = data->gc;
    Pixmap        pixmap = data->pixmap;    
    static double maxDistance;
    static double origin_imag;
    static double origin_real;
    static int    ncolors;
    static int    depth;
    int           slice;
    static int    size;
    static int    newResolution;
    Pixel         black = BlackPixel ( XtDisplay ( w ), 0 );
    
    if ( data->restart )
    {
       /*
        * If a completely new fractal image is to be  
        * generated, start from the beginning, with a point 
        * size of 16 by 16 pixels. Reinitialize all local 
        * variables to the correct values.
        */

        newResolution = TRUE;
        size          = 16;
        height        = data->height / size;
        width         = data->width  / size;
        maxDistance   = data->maxDistance;
        origin_imag   = data->origin.imag;
        origin_real   = data->origin.real;    
        ncolors       = data->ncolors;
        depth         = data->depth;
        data->restart = FALSE;
    }

    if ( data->restart || newResolution )
    {
       /*
        * If an entire new fractal is being generated, or if 
        * the resolution of the image has changed,  
        * reinitialize those variables that have to change.
        * Clear the pixmap and reset the point buffers 
        * for the next round of drawing.
        */

        ReportTime ( "Starting fractal", FALSE );

        rangeByWidth  = data->range / ( double ) width;
        rangeByHeight = data->range / ( double ) height;
        y = 0;
    
        InitBuffer(); 
        
        XSetForeground ( XtDisplay ( w ), gc,
                         BlackPixelOfScreen ( XtScreen ( w ) ) );
  
        XFillRectangle ( XtDisplay ( w ), pixmap, gc, 0, 0, 
                         data->width,  data->height );

        xconstants = ( double* ) XtMalloc ( width *
                                            sizeof ( double ) );

        for ( x = 0; x < width; x++ ) 
            xconstants[x] = origin_real + ( double ) x *
                                rangeByWidth;
    }
    for ( slice = 0 ; slice < 2 && y < height; y++, slice++ )     
    {
        static double z_real, z_imag, k_real, k_imag;

        k_imag =  origin_imag - ( double ) y * rangeByHeight;
            
        for ( x = 0; x < width; x++ ) 
        {
            int      distance;
                
            z_real =  z_imag = 0.0;
            k_real =  xconstants[x];
            
            /*
             * Calculate z =  z * z + k  over and over.
             */
            
            for ( iteration = 0; 
                  iteration < depth;
                  iteration++ ) 
            {
                static double   real;
                
                real   = z_real;
                z_real = z_real * z_real - 
                                    z_imag * z_imag + k_real;
                z_imag = (real + real) * z_imag + k_imag;

                distance  =  (int) ( z_real * z_real + 
                                     z_imag * z_imag );     
                
               /*
                * If the z point has moved off the plane,  
                * set the current foreground color to  
                * the distance (cast to an int and modulo
                * the number of colors available), and 
                * draw a point in the window and the pixmap.
                */
                
                if ( distance >= maxDistance )
                {
                    Pixel color;

                    color = ( Pixel ) iteration % ncolors;
                    
                    BufferPoint ( w, pixmap, gc,
                                  ( int ) color, x, y, size );
                    break;
                }
            }

           /*
            * The image is no longer cleared to black because 
            * it would destroy the effect. However, parts of 
            * the area inside the Mandelbrot set will be 
            * drawn incorrectly at previous resolutions, so
            * we must draw each point inside the set as well.
            */

            if ( distance < maxDistance )
                BufferPoint ( w, pixmap, gc,
                              ( int ) black, x, y, size );
        }
    }
   if ( y == data->height )
   {
        newResolution = FALSE;               
        FlushBuffer ( w, pixmap, gc, size );
        HandleEvents ( data->canvas );
        ReportTime ( "Fractal Generation Complete", TRUE ); 
        return ( TRUE );
   }
   else if ( y == height )
   {
       newResolution = TRUE;
       
       FlushBuffer ( w, pixmap, gc, size );
       
       size   /= 4;
       height  = data->height / size;
       width   = data->width  / size;

       if ( width > data->width/ 2 || height > data->height / 2)
       {  
           width  = data->width;
           height = data->height;
           size   = 1;
       }

       HandleEvents ( data->canvas );
       ReportTime ( "Round Completed", TRUE );

       return ( FALSE );
    }
    else
    {
        newResolution = FALSE;        
        return ( FALSE );
    }        
}

#define MAXPOINTS 500
#define MAXCOLOR  256

typedef struct {
    XPoint  points[MAXCOLOR][MAXPOINTS];
    int     numPoints[MAXCOLOR];
} PointBuffer;

PointBuffer  buffer;

static void InitBuffer()
{
    int i;

    for ( i=0;i<MAXCOLOR;i++ )
        buffer.numPoints[i] = 0;
}

static void BufferPoint ( Widget w, Pixmap pix, 
                           GC gc,   Pixel color, 
                           int x,   int y, int size )
{
   /*
    * Dont allow more than MAXCOLOR colors.
    */

    color = color % MAXCOLOR;

    if ( buffer.numPoints[color] == MAXPOINTS )
    {
        XSetForeground ( XtDisplay (  w ), gc, color );

        Render ( w, pix, gc, 
                 buffer.points [ color ],
                 buffer.numPoints [ color ], size );

       /*
        * Reset the buffer.
        */
      
        buffer.numPoints[color] = 0;
     }
    
    /*
     * Store the point in the buffer according to its color.
     */
    
    buffer.points[color][buffer.numPoints[color]].x = x;
    buffer.points[color][buffer.numPoints[color]].y = y;
    buffer.numPoints[color] += 1;
}

static void RedisplayCallback ( Widget    w, 
                                XtPointer clientData, 
                                XtPointer callData ) 
{
    ImageData                   *data = ( ImageData * ) clientData;
    XmDrawingAreaCallbackStruct *cb = 
                            ( XmDrawingAreaCallbackStruct * ) callData;
    
    XExposeEvent  *event = ( XExposeEvent * ) cb->event;
    
   /*
    * Extract the exposed area from the event and copy
    * from the saved pixmap to the window.
    */
    
    XCopyArea ( XtDisplay ( w ), data->pixmap,
                XtWindow ( w ), data->gc, 
                event->x, event->y, 
                event->width, event->height, 
                event->x, event->y );
}

static void Render ( Widget  w, Pixmap pix, GC gc, 
                     XPoint *points, 
                     int     numPoints, int size )
{
    if ( size <= 1 )
    {
       /*
        * If the size of a point is 1 pixel, just use 
        * XDrawPoints to draw each point in the given buffer.
        */

        if ( pix )    
            XDrawPoints ( XtDisplay ( w ), pix, gc, 
                          points, numPoints,
                          CoordModeOrigin );

        if ( XtIsRealized ( w ) )
            XDrawPoints ( XtDisplay ( w ), XtWindow(w), gc, 
                          points, numPoints,
                          CoordModeOrigin );
    }
    else
    {
       /*
        * If the size of point is larger than one, convert  
        * the buffer to an array of XRectangle structures 
        * and draw filled rectangles for each point.
        */

        static XRectangle rect[MAXPOINTS];
        int i;
        
        for ( i = 0; i < numPoints; i++ )
        {
            rect[i].x      = points[i].x * size;
            rect[i].y      = points[i].y * size;
            rect[i].width  = size;
            rect[i].height = size;            
        }

        if ( pix )    
            XFillRectangles ( XtDisplay ( w ), pix, gc, 
                              rect, numPoints );

        if ( XtIsRealized ( w ) )
            XFillRectangles ( XtDisplay ( w ), XtWindow(w),
                              gc, rect, numPoints );
    }
}

static void FlushBuffer ( Widget w, Pixmap pix, GC gc,
                          int size )
{ 
    int i;
    
    /*
     * Check each buffer.
     */
    
    for ( i=0 ; i < MAXCOLOR; i++ )
    {
       /*
        * If there are any points in this buffer, display them
        * in the window and the pixmap.
        */
        
         if ( buffer.numPoints [ i ] )
         {
             XSetForeground ( XtDisplay ( w ), gc, i );

             Render ( w, pix, gc, buffer.points[i],
                      buffer.numPoints[i], size ); 

             buffer.numPoints[i] = 0;
         }    
    }
}

static void ResizeCallback ( Widget    w,
                             XtPointer clientData,
                             XtPointer callData ) 
{
    ImageData     *data = ( ImageData * ) clientData;

   /*
    *  Get the new window size.
    */
    
    XtVaGetValues ( w, 
                    XmNwidth,  &data->width,
                    XmNheight, &data->height,
                    NULL );
   /*
    * Clear the window, forcing an Expose event to be generated
    */
    
    if ( XtIsRealized ( w ) ) 
        XClearArea ( XtDisplay ( w ), XtWindow ( w ),
                     0, 0, 0, 0, TRUE );
    
   /*
    *  Free the old pixmap and create a new pixmap 
    *  the same size as the window.
    */
    
    if ( data->pixmap ) 
        XFreePixmap ( XtDisplay ( w ), data->pixmap );
    
    data->pixmap = 
           XCreatePixmap ( XtDisplay ( w ),
                    DefaultRootWindow ( XtDisplay ( w ) ),
                    data->width, data->height, 
                    DefaultDepthOfScreen ( XtScreen ( w ) ));
    
    XSetForeground ( XtDisplay ( w ), data->gc,
                     BlackPixelOfScreen ( XtScreen ( w ) ) );
    
    XFillRectangle ( XtDisplay ( w ), data->pixmap,
                     data->gc, 0, 0, 
                     data->width,  data->height );
    
   /*
    * Generate a new image.
    */
    
    CreateImage ( data );
}

