/********************************************************************
 *         This example code is from the book:
 *
 *           Motif Debuggin and Performance Tuning
 *            ISBN 0-13-147984-9
 *         by
 *           Douglas Young
 *           Prentice Hall, 1995
 *
 *         Copyright 1994 by Prentice Hall
 *         All Rights Reserved
 *
 *  Permission to use, copy, modify, and distribute this software for 
 *  any purpose except publication and without fee is hereby granted, provided 
 *  that the above copyright notice appear in all copies of the software.
 * *****************************************************************************/

/*****************************************************
 * browser.c: main dirview interface.
 *****************************************************/
#include <Xm/Xm.h>
#include <Xm/MainW.h> 
#include <Xm/RowColumn.h>
#include <Xm/Label.h> 
#include <unistd.h>
#include <stdio.h>
#include "globals.h"
#include "TimerTools.h"

/*
 * Externally-defined functions
 */

extern Widget CreateMenu ( Widget );
extern Widget CreateIcon ( char *, Widget, FileType , Widget );
extern void DisplayBusyCursor ( Widget w);
extern void HideBusyCursor( Widget w);

/*
 * label widget that displays current directory.
 */

static Widget currentDirLabel;
static char *currentDirectory = NULL;
static DIR *fd;
static Widget rowColumn;

static int count = 0;


void SetCurrentDirectory ( char *dir )
{
    char buf[300];

   /*
    * Change the program's sense of the working directory.
    * This allows the program to continue using relative
    * names for files and directories.
    */

    if ( dir )
        chdir ( dir );
  
   /*
    * The name of the currentDirectory is needed
    * when opening the directory to read entries. To elliminate
    * names like ../../././.., which could result from
    * browsing many directories, set the current directory
    * by calling getcwd().
    */

    if ( currentDirectory )
        XtFree ( currentDirectory );

    currentDirectory = getcwd ( NULL, 300 );
  
   /*
    * Display the current directory in the label
    */

    sprintf ( buf, "Current Directory: %s", 
              currentDirectory );
  
    XtVaSetValues ( currentDirLabel,
                    XtVaTypedArg,
                    XmNlabelString,  XmRString,
                    buf, strlen ( buf ) + 1,
                    NULL );
}

Boolean exclude ( const char *name )
{
   /*
    * Return TRUE if a file should not be displayed. This
    * simple test excludes files whose names end in ".o"
    */

    if ( strlen ( name ) > 2 &&
        !strcmp ( &name [ strlen ( name ) - 2 ], ".o" ) )
        return ( TRUE );
     return ( FALSE );
}

Boolean ReadDirWorkProc ( XtPointer clientData )
{
    struct dirent *entry = NULL;
    Widget     parent = (Widget) clientData;
    Widget     icon;
    WidgetList children;
    Cardinal   numChildren;

    DisplayBusyCursor ( XtParent ( parent ) );
  
    if ( !count )
    {
        if ( ( fd = opendir ( currentDirectory ) ) == NULL )
            return TRUE;
          
        count = 1;

        XtVaGetValues ( parent,
                        XmNchildren,    &children,
                        XmNnumChildren, &numChildren,
                        NULL );

        if ( numChildren )
        {
            count = 0;
          
            while ( count < numChildren &&
                    ( entry = readdir ( fd ) ) != NULL )
            {
                if ( !exclude ( entry->d_name ) )
                {
                    icon = CreateIcon ( entry->d_name,
                                      parent,
                                      GetType (entry->d_name ),
                                      children[count]);
                    count++;
               }

               if ( ( count %50 ) == 0 )
                   DisplayBusyCursor ( XtParent ( parent ) );
            }

            XtManageChildren ( children, count );
          
            if ( !entry )
            {
                int diff = numChildren - count;
              
                XtUnmanageChildren ( &children[count], diff );
              
                HideBusyCursor ( XtParent ( parent ) );  
                ReportTime ( "Cached Icons Created", TRUE );

                XtManageChildren ( children, count );
                XtManageChild ( parent );
              
                closedir ( fd );
              
                return ( TRUE );
            }
        }
    }
  
    while ( ( count++ % 50 ) != 0 &&
            ( entry = readdir ( fd ) ) != NULL )
    {
        if ( !exclude ( entry->d_name ) )
            icon = CreateIcon ( entry->d_name,
                                parent,
                                GetType ( entry->d_name ) , NULL );
    }

    if ( entry && count && count <= 51 )
    {
        XtVaGetValues ( parent, 
                        XmNchildren, &children,
                        XmNnumChildren, &numChildren,
                        NULL );
        XtManageChildren ( children, numChildren );
        XtManageChild ( parent );

        XmUpdateDisplay ( parent );
        ReportTime ( "Initial icons created", TRUE );

        XSetWindowBackgroundPixmap ( XtDisplay ( parent ),
                             XtWindow ( XtParent ( parent ) ),
                             None );
        XtUnmanageChild ( parent );      
    }
  
    if ( entry )
        return FALSE;
  
    XtVaGetValues ( parent,
                    XmNchildren, &children,
                    XmNnumChildren, &numChildren,
                    NULL );
    XtManageChildren ( children, numChildren );

    XSetWindowBackgroundPixmap ( XtDisplay(parent),
                              XtWindow ( XtParent ( parent ) ),
                              ParentRelative );  
    XtManageChild ( parent );
  
    HideBusyCursor ( XtParent ( parent ) );  
    ReportTime ( "Icons Created", TRUE );

    closedir ( fd );
  
    return ( TRUE );
}

Widget CreateBrowser ( Widget parent, char *dir )
{
    Widget mainWindow, commandPanel, menuBar;
    XtAppContext app = XtWidgetToApplicationContext ( parent );

    if ( !dir )
        dir = getcwd ( NULL, 300 );

    currentDirectory = XtNewString ( dir );
  
    if ( ( fd = opendir ( currentDirectory ) ) == NULL )
        return NULL;    
    
    mainWindow = XtVaCreateWidget ( "browser",
                        xmMainWindowWidgetClass,   parent,
                        XmNscrollingPolicy,        XmAUTOMATIC,
                        XmNscrollBarDisplayPolicy, XmSTATIC,
                        XmNcommandWindowLocation, 
                                      XmCOMMAND_ABOVE_WORKSPACE,
                        NULL );

    menuBar = CreateMenu ( mainWindow );

    rowColumn = XtVaCreateWidget ( "rowColumn",
                                    xmRowColumnWidgetClass,
                                    mainWindow,
                                    XmNpacking, XmPACK_COLUMN,
                                    XmNnumColumns, 8,
                                    NULL );

    XtAppAddWorkProc ( app,
                       ReadDirWorkProc, 
                       ( XtPointer ) rowColumn );

    commandPanel = XtVaCreateManagedWidget  ( "commandPanel",
                       xmRowColumnWidgetClass, mainWindow, 
                       XmNorientation,    XmHORIZONTAL,
                       XmNentryAlignment, XmALIGNMENT_CENTER,
                       XmNpacking,        XmPACK_TIGHT,
                       NULL );

    currentDirLabel = XtCreateManagedWidget  ( "label1",
                                           xmLabelWidgetClass,
                                           commandPanel, 
                                           NULL, 0);

    SetCurrentDirectory ( dir );

    XtVaSetValues ( mainWindow,
		    XmNcommandWindow, commandPanel,
		    XmNworkWindow, rowColumn,
		    NULL);        
    return ( mainWindow );
}



void BrowseDirectory ( char *dir )
{
    WidgetList children;
    Cardinal   numChildren;
    int        i;
    XtAppContext app =
                 XtWidgetToApplicationContext ( rowColumn );

    ReportTime ( "Browsing new directory", FALSE );

    if ( ( fd = opendir ( dir ) ) == NULL )
        return;

    SetCurrentDirectory ( ( char* ) dir );

   /*
    * Reset the work procedure by setting count  
    * back to zero, and reinstall the work proc.
    */

    count = 0;
  
    XtAppAddWorkProc ( app,ReadDirWorkProc,
                       ( XtPointer ) rowColumn );
}
