/* $XConsortium: LookupCmap.c,v 1.8 92/11/23 15:43:44 rws Exp $ 
 * 
 * Copyright 1989 by the Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided 
 * that the above copyright notice appear in all copies and that both that 
 * copyright notice and this permission notice appear in supporting 
 * documentation, and that the name of M.I.T. not be used in advertising
 * or publicity pertaining to distribution of the software without specific, 
 * written prior permission. M.I.T. makes no representations about the 
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * M.I.T. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL M.I.T.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Donna Converse, MIT X Consortium
 *                                                                            
 *                                                                            
 *                                                                            
 *  (c) Copyright Hewlett-Packard Company, 1993, Fort Collins, Colorado       
 *                                                                            
 *                            All Rights Reserved                             
 *                                                                            
 *  Permission to use, copy, modify, and distribute this software and its     
 *  documentation for any purpose and without fee is hereby granted,          
 *  provided that the above copyright notices appear in all copies and that   
 *  both the copyright notices and this permission notice appear in           
 *  supporting documentation, and that the name of Hewlett-Packard not be     
 *  used in advertising or publicity pertaining to distribution of the        
 *  software without specific, written prior permission.                      
 *                                                                            
 *  HEWLETT-PACKARD MAKES NO WARRANTY OF ANY KIND WITH REGARD TO THIS         
 *  SOFTWARE, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF        
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Hewlett-Packard    
 *  shall not be liable for errors contained herein or direct, indirect,      
 *  special, incidental or consequential damages in connection with the       
 *  furnishing, performance or use of this software.                          
 *
 * Modifications for PEX:  John Waitz, Hewlett-Packard Company
 *
 */

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/Xmu/StdCmap.h>
#include <X11/Xmu/StdCmapExt.h>
#ifdef PEX_SUPPORT /* [ */
#include <X11/PEX5/PEXlib.h>
#include <X11/Xmu/StdCmapPEX.h>
#endif /* ] PEX_SUPPORT */

extern char *malloc();
static Status lookup();
#ifdef PEX_SUPPORT /* [ */
static Status checkPEXColorApprox();
static Status createTemporaryWindow();
static Status destroyTemporaryWindow();
#endif /* ] PEX_SUPPORT */

/*
 * To create a standard colormap if one does not currently exist, or
 * replace the currently existing standard colormap, use 
 * XmuLookupStandardColormapExt().
 *
 * Given a screen, a visual, and a property, XmuLookupStandardColormapExt()
 * will determine the best allocation for the property under the specified
 * visual, and determine the whether to create a new colormap or to use
 * the default colormap of the screen.  It will call XmuStandardColormapExt()
 * to create the standard colormap.
 *
 * If replace is true, any previous definition of the property will be 
 * replaced.  If retain is true, the property and the colormap will be
 * made permanent for the duration of the server session.  However,
 * pre-existing property definitions which are not replaced cannot be made
 * permanent by a call to XmuLookupStandardColormapExt(); a request to retain 
 * resources pertains to newly created resources.
 *
 * Returns 0 on failure, non-zero on success.  A request to create a 
 * standard colormap upon a visual which cannot support such a map is
 * considered a failure.  An example of this would be requesting any
 * standard colormap property on a monochrome visual, or, requesting an
 * RGB_BEST_MAP on a PseudoColor Visual whose colormap size is 16.
 */

Status XmuLookupStandardColormapExt(dpy, screen, visualid, depth, property,
				 replace, retain, params)
    Display		*dpy;		/* specifies X server connection */
    int			screen; 	/* specifies screen of display */
    VisualID		visualid;	/* specifies the visual type */
    unsigned int	depth;		/* specifies  the visual type */
    Atom		property;	/* a standard colormap property */
    Bool		replace;	/* specifies whether to replace */
    Bool		retain;		/* specifies whether to retain */
    XmuExtendedParams 	*params;	/* controls and hints */
{
    Display		*odpy;		/* original display connection */
    XStandardColormap	*colormap;	
    XVisualInfo		vinfo_template, *vinfo;	/* visual */
    long		vinfo_mask;
    unsigned long	r_max, g_max, b_max;	/* allocation */
    int			count;	
    Colormap		cmap;			/* colormap ID */
    Status		status = 0;
    int			create_default = False;
#ifdef PEX_SUPPORT /* [ */
    int			PEX_required = False;
#endif /* ] PEX_SUPPORT */

    /*
	Decode the controls and hints supported in this procedure.
	Set flag for the PEX requirement, but don't set it unless 
	property is RGB_DEFAULT_MAP or RGB_BEST_MAP.
	Set flag for the create_default hint, that allows a backup
	strategy of creating a new Colormap in case the default Colormap
	is over-allocated.
    */
    if (params) {
	int i;

#ifdef PEX_SUPPORT /* [ */
	for (i=0; i < params->num_controls; i++) {
	    if ((params->controls[i].name == XMU_CONTROL_PEX_SUPPORTED) &&
		((property == XA_RGB_DEFAULT_MAP) ||
		(property == XA_RGB_BEST_MAP)))
		PEX_required = params->controls[i].value;
	}
#endif /* ] PEX_SUPPORT */

	for (i=0; i < params->num_hints; i++) {
	    if (params->hints[i].name == XMU_HINT_CREATE_DEFAULT)
		create_default = params->hints[i].value;
	}
    }

    /* Match the requested visual */

    vinfo_template.visualid = visualid;	
    vinfo_template.screen = screen;
    vinfo_template.depth = depth;
    vinfo_mask = VisualIDMask | VisualScreenMask | VisualDepthMask;
    if ((vinfo = XGetVisualInfo(dpy, vinfo_mask, &vinfo_template, &count)) ==
	NULL)
	return 0;

    /* Monochrome visuals have no standard maps */

    if (vinfo->colormap_size <= 2) {
	XFree((char *) vinfo);
	return 0;	
    }

#ifdef PEX_SUPPORT /* [ */
    /*
	If PEX is required, check support for the Visual.
    */
    if (PEX_required) {

	PEXExtensionInfo	*ext_info;
	PEXRenderingTarget	*targets;
	unsigned long		target_count;

	targets = NULL;
	target_count = 0;

	if (NULL == (ext_info = PEXGetExtensionInfo (dpy)))
	    return 0;

	if ((ext_info->major_version == 5) &&
	    (ext_info->minor_version >= 1)) {

	    if ((! PEXMatchRenderingTargets (dpy, 
					RootWindow(dpy, screen),
					depth, PEXWindowDrawable, vinfo->visual,
					1, &target_count, &targets)) ||
		(target_count == 0)) {
		XFree((char *) vinfo);
		return 0;
	    }

	    XFree (targets);
	}
    }
#endif /* ] PEX_SUPPORT */

    /* If the requested property already exists on this screen, and, 
     * if the replace flag has not been set to true, return success.
     * lookup() will remove a pre-existing map if replace is true.
     */

    if (lookup(dpy, screen, visualid, property, (XStandardColormap *) NULL,
	       replace) && !replace) {
	int status;
	status = 1;

#ifdef PEX_SUPPORT /* [ */
	/*
	    Check to see whether the existing property that was found
	    contains a color ramp supported by PEX.
	*/
	if (PEX_required) {
	    int count;
	    XStandardColormap *stdcmaps;

	    count = 0;
	    XGetRGBColormaps (dpy, RootWindow(dpy, screen), 
				&stdcmaps, &count, property);

	    if (count)
		status = checkPEXColorApprox (dpy, vinfo, 
					    stdcmaps, count, property);

	    XFree ((char *) stdcmaps);
	}
#endif /* ] PEX_SUPPORT */

	XFree((char *) vinfo);
	return status;
    }

    /* Determine the best allocation for this property under the requested
     * visualid and depth, and determine whether or not to use the default
     * colormap of the screen.
     */

    if (!XmuGetColormapAllocationExt(vinfo, property, params,
					&r_max, &g_max, &b_max)) {
	XFree((char *) vinfo);
	return 0;
    }

    cmap = (property == XA_RGB_DEFAULT_MAP &&
	    visualid == XVisualIDFromVisual(DefaultVisual(dpy, screen)))
	? DefaultColormap(dpy, screen) : None;

    /* If retaining resources, open a new connection to the same server */

    if (retain) {
	XSync (dpy, 0);
	odpy = dpy;
	if ((dpy = XOpenDisplay(XDisplayString(odpy))) == NULL) {
	    XFree((char *) vinfo);
	    return 0;
	}

#ifdef PEX_SUPPORT /* [ */
	/*
	    If PEX required, need to open PEXlib on the new connection.
	*/
	if (PEX_required) {
	    PEXExtensionInfo *ext_info;

	    if (PEXInitialize (dpy, &ext_info, 0, NULL)) {
		XFree((char *) vinfo);
		return 0;
	    }
	}
#endif /* ] PEX_SUPPORT */
    }

    /* Create the standard colormap */

    colormap = XmuStandardColormapExt(dpy, screen, visualid, depth, property,
				   cmap, r_max, g_max, b_max, params);

    if (create_default) {
	if ((colormap == NULL) &&
	    (property == XA_RGB_DEFAULT_MAP) &&
	    (visualid == XVisualIDFromVisual(DefaultVisual(dpy, screen)))) {

	    cmap = None;
	    colormap = XmuStandardColormapExt(dpy, screen, visualid, depth, 
				property, cmap, r_max, g_max, b_max, params);
	}
    }

    /* Set the standard colormap property */

    if (colormap) {
	XGrabServer(dpy);

	if (lookup(dpy, screen, visualid, property, colormap, replace) &&
	    !replace) {
	    /* Someone has defined the property since we last looked.
	     * Since we will not replace it, release our own resources.
	     * If this is the default map, our allocations will be freed 
	     * when this connection closes.
	     */
	    if (colormap->killid == ReleaseByFreeingColormap)
		XFreeColormap(dpy, colormap->colormap);
	}
	else if (retain) {
		XSetCloseDownMode(dpy, RetainPermanent);
	}
	XUngrabServer(dpy);
	XFree((char *) colormap);
	status = 1;
    }

    if (retain) 
	XCloseDisplay(dpy);
    XFree((char *) vinfo);
    return status;
}

/***************************************************************************/

/* Lookup a standard colormap property.  If the property is RGB_DEFAULT_MAP,
 * the visualid is used to determine whether the indicated standard colormap
 * exists.  If the map exists and replace is true, delete the resources used
 * by the map and remove the property.  Return true if the map exists,
 * or did exist and was deleted; return false if the map was not found.
 *
 * Note that this is not the way that a Status return is normally used.
 *
 * If new is not NULL, new points to an XStandardColormap structure which
 * describes a standard colormap of the specified property.  It will be made
 * a standard colormap of the screen if none already exists, or if replace 
 * is true.
 */

static Status lookup(dpy, screen, visualid, property, new, replace)
    Display		*dpy;		/* specifies display connection */
    int			screen;		/* specifies screen number */
    VisualID		visualid;	/* specifies visualid for std map */
    Atom		property;	/* specifies colormap property name */
    XStandardColormap	*new;		/* specifies a standard colormap */
    Bool		replace;	/* specifies whether to replace */
{
    register int	i;
    int			count;
    XStandardColormap	*stdcmaps, *s;
    Window		win = RootWindow(dpy, screen);

    /* The property does not already exist */

    if (! XGetRGBColormaps(dpy, win, &stdcmaps, &count, property)) {
	if (new)
	    XSetRGBColormaps(dpy, win, new, 1, property);
	return 0;
    }

    /* The property exists and is not describing the RGB_DEFAULT_MAP */

    if (property != XA_RGB_DEFAULT_MAP) {
	if (replace) {
	    XmuDeleteStandardColormap(dpy, screen, property);
	    if (new)
		XSetRGBColormaps(dpy, win, new, 1, property);
	}
	XFree((char *)stdcmaps);
	return 1;
    }

    /* The property exists and is RGB_DEFAULT_MAP */
    
    for (i=0, s=stdcmaps; (i < count) && (s->visualid != visualid); i++, s++)
	;

    /* No RGB_DEFAULT_MAP property matches the given visualid */

    if (i == count) {
	if (new) {
	    XStandardColormap	*m, *maps;

	    s = (XStandardColormap *) malloc((unsigned) ((count+1) * sizeof
					      (XStandardColormap)));

	    for (i = 0, m = s, maps = stdcmaps; i < count; i++, m++, maps++) {
		m->colormap   = maps->colormap;
		m->red_max    = maps->red_max;
		m->red_mult   = maps->red_mult;
		m->green_max  = maps->green_max;
		m->green_mult = maps->green_mult;
		m->blue_max   = maps->blue_max;
		m->blue_mult  = maps->blue_mult;
		m->base_pixel = maps->base_pixel;
		m->visualid   = maps->visualid;
		m->killid     = maps->killid;
	    }
	    m->colormap   = new->colormap;
	    m->red_max    = new->red_max;
	    m->red_mult   = new->red_mult;
	    m->green_max  = new->green_max;
	    m->green_mult = new->green_mult;
	    m->blue_max   = new->blue_max;
	    m->blue_mult  = new->blue_mult;
	    m->base_pixel = new->base_pixel;
	    m->visualid   = new->visualid;
	    m->killid     = new->killid;

	    XSetRGBColormaps(dpy, win, s, ++count, property);
	    free((char *) s);
	}
	XFree((char *) stdcmaps);
	return 0;
    }

    /* Found an RGB_DEFAULT_MAP property with a matching visualid */

    if (replace) {
	/* Free old resources first - we may need them, particularly in 
	 * the default colormap of the screen.  However, because of this,
	 * it is possible that we will destroy the old resource and fail 
	 * to create a new one if XmuStandardColormap() fails.
	 */

	if (count == 1) {
	    XmuDeleteStandardColormap(dpy, screen, property);
	    if (new)
		XSetRGBColormaps(dpy, win, new, 1, property);
	}
	else {
	    XStandardColormap	*map;

	    /* s still points to the matching standard colormap */

	    if (s->killid == ReleaseByFreeingColormap) {
		if ((s->colormap != None) &&
		    (s->colormap != DefaultColormap(dpy, screen)))
		    XFreeColormap(dpy, s->colormap);
	    }
	    else if (s->killid != None)
		XKillClient(dpy, s->killid);

	    map = (new) ? new : stdcmaps + --count;

	    s->colormap   = map->colormap;
	    s->red_max    = map->red_max;
	    s->red_mult   = map->red_mult;
	    s->green_max  = map->green_max;
	    s->green_mult = map->green_mult;
	    s->blue_max   = map->blue_max;
	    s->blue_mult  = map->blue_mult;
	    s->base_pixel = map->base_pixel;
	    s->visualid   = map->visualid;
	    s->killid     = map->killid;

	    XSetRGBColormaps(dpy, win, stdcmaps, count, property);
	}
    }
    XFree((char *) stdcmaps);
    return 1;
}

#ifdef PEX_SUPPORT /* [ */

static Status checkPEXColorApprox (dpy, vinfo, stdcmap_array, count, property)
    Display		*dpy;
    XVisualInfo		*vinfo;
    XStandardColormap	*stdcmap_array;
    int			count;
    Atom		property;
{
    int 		i;
    XStandardColormap 	*stdcmap;
    int			return_value;
    Window		window;
    int			inquiry_supported;
    PEXExtensionInfo	*ext_info;


    /*
	Get the property entry to use.
    */
    if (property != XA_RGB_DEFAULT_MAP)
	stdcmap = stdcmap_array;
    else {
	for (i=0, stdcmap=stdcmap_array; 
	    (i < count) && (stdcmap->visualid != vinfo->visualid); 
	    i++, stdcmap++) ;

	if (i == count)
	    /* unexpected */
	    return 0;
    }


    if (NULL == (ext_info = PEXGetExtensionInfo (dpy)))
	return 0;

    /*
	Create a temporary unmapped window in the Visual.
    */
    if (!createTemporaryWindow (dpy, vinfo, &window))
	return 0;

    inquiry_supported = False;
    if ((ext_info->major_version == 5) &&
	(ext_info->minor_version >= 1)) 
    {
	int		enum_types[1];
	unsigned long 	*enum_count;
	PEXEnumTypeDesc	*enum_data;
	int		enum_index;

	/*
	    Verify that the QueryColorApprox escape is supported
	    using PEXGetEnumTypeInfo.  If not, we just can't verify
	    support, so we'll assume the color ramp is supported.
	*/

	enum_types[0] = PEXETEscape;
	if (!PEXGetEnumTypeInfo (dpy, window, 1, enum_types, 
				PEXETIndex, &enum_count, &enum_data)) {

	    destroyTemporaryWindow (dpy, vinfo, window);
	    return 0;
	}
	else {

	    for (enum_index = 0; enum_index < *enum_count; enum_index++) {
		if ((short) enum_data[enum_index].index == 
		    (short) PEXETEscapeQueryColorApprox) {
		    inquiry_supported = True;
		    break;
		}
	    }
	    PEXFreeEnumInfo (1, enum_count, enum_data);
	}
    }

    if (inquiry_supported) {

	PEXEscapeQueryColorApproxData		query;
	PEXEscapeQueryColorApproxReplyData	*reply;
	unsigned long 				reply_length;

	/*
	    Load up the query structure from the property.
	    Currently assuming only PEXColorSpace is of interest.
	    Note dither flag is a hint, so it shouldn't affect result.
	*/

	query.capx.type = PEXColorSpace;
	query.capx.model = PEXColorApproxRGB;
	query.capx.dither = PEXOn;
	query.capx.base_pixel = stdcmap->base_pixel;
	query.capx.max1 = stdcmap->red_max;
	query.capx.max2 = stdcmap->green_max;
	query.capx.max3 = stdcmap->blue_max;
	query.capx.weight1 = 0.0;
	query.capx.weight2 = 0.0;
	query.capx.weight3 = 0.0;
	query.capx.mult1 = stdcmap->red_mult;
	query.capx.mult2 = stdcmap->green_mult;
	query.capx.mult3 = stdcmap->blue_mult;

	query.drawable = window;

	reply = (PEXEscapeQueryColorApproxReplyData *)
		PEXEscapeWithReply (dpy, PEXEscapeQueryColorApprox, 
				    sizeof(query), ((char *) &query), 
				    &reply_length);

	if (reply != NULL) {

	    return_value = (reply->capx_is_supported);
	    XFree ((char *) reply);
	}
	else {
	    destroyTemporaryWindow (dpy, vinfo, window);
	    return 0;
	}
    }
    else {
	/* Can't determine support or non-support. */
	return_value = 1;
    }


    /*
	Destroy the temporary window.
    */
    destroyTemporaryWindow (dpy, vinfo, window);

    return return_value;
}


static Status createTemporaryWindow (display, vinfo, window_return)
    Display	*display;
    XVisualInfo	*vinfo;
    Window	*window_return;
{
    unsigned long 		window_mask;
    XSetWindowAttributes	window_attrs;
    Colormap			cmap_id;


    /*
	Create a window using override-redirect.  Do not map it.
	If the Visual is the default, use the root window.
	Otherwise, create a colormap to use.
    */

    if (vinfo->visual == DefaultVisual (display, vinfo->screen)) {

	*window_return = RootWindow (display, vinfo->screen);
	return 1;
    }
    else {
	cmap_id = XCreateColormap (display, 
				    RootWindow(display, vinfo->screen),
				    vinfo->visual, AllocNone );
	if (cmap_id == None)
	    return 0;

	window_attrs.colormap = cmap_id;
	window_mask = CWColormap;

	window_attrs.override_redirect = True;
	window_mask |= CWOverrideRedirect;

	window_attrs.background_pixel = 0;
	window_mask |= CWBackPixel;

	window_attrs.border_pixel = 0;
	window_mask |= CWBorderPixel;

	*window_return = XCreateWindow (display, 
				RootWindow (display, vinfo->screen),
				10, 10, 1, 1, 0,
				vinfo->depth, 
				InputOutput,
				vinfo->visual,
				window_mask,
				&(window_attrs));

	if (*window_return == None)
	    return 0;
	else
	    return 1;
    }
} /* createTemporaryWindow */


static Status destroyTemporaryWindow (display, vinfo, window)
    Display	*display;
    XVisualInfo	*vinfo;
    Window	window;
{
    XWindowAttributes	window_attrs;

    /*
	If this window is not in the default Visual, 
	the Colormap and Window need to be freed.
    */

    if (vinfo->visual != DefaultVisual (display, vinfo->screen)) {

	if (XGetWindowAttributes (display, window, &window_attrs))
	    XFreeColormap (display, window_attrs.colormap);

	XDestroyWindow (display, window);
    }

    return 1;

} /* destroyTemporaryWindow */

#endif /* ] PEX_SUPPORT */
