/*
 * $XConsortium: xstdcmap.c,v 1.6 89/07/24 11:06:26 jim Exp $
 *
 * Copyright 1989 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising
 * or publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * M.I.T. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL M.I.T.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Donna Converse, MIT X Consortium
 *                                                                            
 *                                                                            
 *                                                                            
 *  (c) Copyright Hewlett-Packard Company, 1993, Fort Collins, Colorado       
 *                                                                            
 *                            All Rights Reserved                             
 *                                                                            
 *  Permission to use, copy, modify, and distribute this software and its     
 *  documentation for any purpose and without fee is hereby granted,          
 *  provided that the above copyright notices appear in all copies and that   
 *  both the copyright notices and this permission notice appear in           
 *  supporting documentation, and that the name of Hewlett-Packard not be     
 *  used in advertising or publicity pertaining to distribution of the        
 *  software without specific, written prior permission.                      
 *                                                                            
 *  HEWLETT-PACKARD MAKES NO WARRANTY OF ANY KIND WITH REGARD TO THIS         
 *  SOFTWARE, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF        
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Hewlett-Packard    
 *  shall not be liable for errors contained herein or direct, indirect,      
 *  special, incidental or consequential damages in connection with the       
 *  furnishing, performance or use of this software.                          
 *
 * Modifications for PEX:  John Waitz, Hewlett-Packard Company
 *
 */

#include <stdio.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/Xatom.h>
#include <X11/Xmu/StdCmap.h>
#include <X11/Xmu/StdCmapExt.h>
#ifdef PEX_SUPPORT /* [ */
#include <X11/PEX5/PEXlib.h>
#endif /* ] PEX_SUPPORT */

extern void exit();

#define REPLACE		1
#define DO_NOT_REPLACE  0
#define RETAIN		1
#define DO_NOT_RETAIN	0

static char		*display_name = NULL;
static char		*program_name = NULL;
static Bool		all = 0;
static Bool		help = 0;
static Bool		full_default = 0;
static Bool 		verbose = 0;
static int		ramp_hint = 0;
#ifdef PEX_SUPPORT /* [ */
static Bool		pex = 0;
#endif /* ] PEX_SUPPORT */
static Display		*dpy = NULL;
static XmuExtendedParams params = {
    0, NULL, 0, NULL
};

typedef struct
{
    Bool	create;
    Bool	delete;
    Atom	property;
    char	*name;
    char	*nickname;
} colormap_property;

static colormap_property propertyTable[]=
{
{0,	0,	XA_RGB_DEFAULT_MAP,	"RGB_DEFAULT_MAP",	"default"},
{0,	0,	XA_RGB_BEST_MAP,	"RGB_BEST_MAP",		"best"},
{0,	0,	XA_RGB_GRAY_MAP,	"RGB_GRAY_MAP",		"gray"},
{0,	0,	XA_RGB_RED_MAP,		"RGB_RED_MAP",		"red"},
{0,	0,	XA_RGB_GREEN_MAP,	"RGB_GREEN_MAP",	"green"},
{0,	0,	XA_RGB_BLUE_MAP,	"RGB_BLUE_MAP",		"blue"},
};
#define NPROPERTIES (sizeof propertyTable / sizeof propertyTable[0])

#define DEFAULT	0
#define BEST	1
#define GRAY	2
#define RED	3
#define GREEN	4
#define BLUE	5

static char	*usage_message[]=
{
"    -all               make all standard colormaps for the display",
"    -best              make the RGB_BEST_MAP",
"    -blue              make the RGB_BLUE_MAP",
"    -default           make the RGB_DEFAULT_MAP",
"    -delete name       remove a standard colormap",
"    -display dpy       X server to use",
"    -fullDefault       make entry for each Visual when making RGB_DEFAULT_MAP",
"    -gray              make the RGB_GRAY_MAP",
"    -green             make the RGB_GREEN_MAP",
#ifdef PEX_SUPPORT /* [ */
"    -pex               make ramps that are supported by PEX",
#endif /* ] PEX_SUPPORT */
"    -ramp ordering     hint for color ramps in PseudoColor/StaticColor",
"    -red               make the RGB_RED_MAP",
"    -verbose           turn on logging",
"",
NULL };

static XrmOptionDescRec optionTable[]=
{
{"-all",	".all",		XrmoptionNoArg,		(caddr_t) "on"},
{"-best",	".best",	XrmoptionNoArg,		(caddr_t) "on"},
{"-blue",	".blue",	XrmoptionNoArg,		(caddr_t) "on"},
{"-default",	".default",	XrmoptionNoArg,		(caddr_t) "on"},
{"-delete",	".delete",	XrmoptionSepArg,	(caddr_t) NULL},
{"-display",	".display", 	XrmoptionSepArg,	(caddr_t) NULL},
{"-fullDefault",".fullDefault",	XrmoptionNoArg,		(caddr_t) "on"},
{"-gray",	".gray",	XrmoptionNoArg,		(caddr_t) "on"},
{"-green",	".green",	XrmoptionNoArg,		(caddr_t) "on"},
{"-help",	".help",        XrmoptionNoArg,		(caddr_t) "on"},
#ifdef PEX_SUPPORT /* [ */
{"-pex",	".pex",         XrmoptionNoArg,		(caddr_t) "on"},
#endif /* ] PEX_SUPPORT */
{"-ramp",	".ramp",        XrmoptionSepArg,	(caddr_t) NULL},
{"-red",	".red",		XrmoptionNoArg,		(caddr_t) "on"},
{"-verbose",	".verbose",	XrmoptionNoArg,		(caddr_t) "on"},
};
#define NOPTIONS (sizeof optionTable / sizeof optionTable[0])

static int findRampHint (name)
    char *name;
{
    if (!strcmp(name,"rgb"))
	return RAMP_RGB;
    else if (!strcmp(name,"gbr"))
	return RAMP_GBR;
    else if (!strcmp(name,"brg"))
	return RAMP_BRG;
    else if (!strcmp(name,"rbg"))
	return RAMP_RBG;
    else if (!strcmp(name,"grb"))
	return RAMP_GRB;
    else if (!strcmp(name,"bgr"))
	return RAMP_BGR;
    else {
	usage(1);
    }
}

static void parse(argc, argv)
    int		argc;
    char	**argv;
{
    XrmDatabase		database = NULL;
    char		*type;
    XrmValue		value;
    char		option[512];

    if (argc == 1)
	usage(0);

    XrmInitialize();
    XrmParseCommand(&database, optionTable, NOPTIONS, program_name, &argc,
		    argv);
    if (--argc)
	usage(1);

    (void) sprintf(option, "%s%s", program_name, ".all");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value)) {
    	all++;
	full_default++;
    }

    (void) sprintf(option, "%s%s", program_name, ".best");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
    	propertyTable[BEST].create++;

    (void) sprintf(option, "%s%s", program_name, ".blue");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
	propertyTable[BLUE].create++;

    (void) sprintf(option, "%s%s", program_name, ".default");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
	propertyTable[DEFAULT].create++;

    (void) sprintf(option, "%s%s", program_name, ".delete");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value)) {
	register int i;
	for (i=0; i < NPROPERTIES; i++) 
	    if (strncmp((char *) value.addr, propertyTable[i].nickname,
			(int) value.size) == 0) {
		propertyTable[i].delete++;
		break;
	    }
    }
		
    (void) sprintf(option, "%s%s", program_name, ".display");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
	display_name = value.addr;

    (void) sprintf(option, "%s%s", program_name, ".fullDefault");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value)) {
	full_default++;
	propertyTable[DEFAULT].create++;
    }

    (void) sprintf(option, "%s%s", program_name, ".gray");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
	propertyTable[GRAY].create++;

    (void) sprintf(option, "%s%s", program_name, ".green");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
	propertyTable[GREEN].create++;

    (void) sprintf(option, "%s%s", program_name, ".help");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
	help++;

#ifdef PEX_SUPPORT /* [ */
    (void) sprintf(option, "%s%s", program_name, ".pex");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
    	pex++;
#endif /* ] PEX_SUPPORT */

    (void) sprintf(option, "%s%s", program_name, ".ramp");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
	ramp_hint = findRampHint(value.addr);

    (void) sprintf(option, "%s%s", program_name, ".red");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
	propertyTable[RED].create++;

    (void) sprintf(option, "%s%s", program_name, ".verbose");
    if (XrmGetResource(database, option, (char *) NULL, &type, &value))
	verbose++;
}

Exit(status)
    Status	status;
{
    if (dpy)
	XCloseDisplay(dpy);
    exit(status);
}

usage(status)
    Status		status;
{
    register char	**i;
    (void) fprintf(stderr, "usage:  %s [-options]\n\n", program_name);
    (void) fprintf(stderr, "where options include:\n");
    for (i = usage_message; *i != NULL; i++)
	(void) fprintf(stderr, "%s\n", *i);
    Exit(status);
}

/* Determine the visual of greatest depth in a given visual class.
 * If no such visual exists, return NULL.  
 */
static XVisualInfo *getDeepestVisual(visual_class, vinfo, nvisuals,
					targets, ntargets)
    int		visual_class;	/* specifies the desired visual class */
    XVisualInfo	*vinfo;		/* specifies all visuals for a screen */
    int		nvisuals;	/* specifies number of visuals in the list */
    char 	*targets;	/* PEX target list, NULL if PEX not required */
    int		ntargets;	/* specifes count of PEX targets */
{
    register int	i;
    unsigned int	maxdepth = 0;
    XVisualInfo		*v = NULL;
    
    for (i=0; i < nvisuals; i++, vinfo++) {

#ifdef PEX_SUPPORT /* [ */
	/*
	    Filter so only PEX-supported Visuals are considered.
	*/
	if (ntargets) {
	    int PEX_supported_on_Visual;
	    PEXRenderingTarget	*p_target;
	    int j;

	    PEX_supported_on_Visual = False;
	    for (j=0, p_target = (PEXRenderingTarget *) targets; 
		    j < ntargets; j++, p_target++) {

		if ((p_target->depth == vinfo->depth) &&
		   (XVisualIDFromVisual(p_target->visual) == vinfo->visualid)) {
		    PEX_supported_on_Visual = True;
		    break;
		}
	    }
	    if (!PEX_supported_on_Visual)
		continue;
	}
#endif /* ] PEX_SUPPORT */

	if (vinfo->class == visual_class && vinfo->depth > maxdepth)
	{
	    maxdepth = vinfo->depth;
	    v = vinfo;
	}
    }
    return(v);
}

/* Determine the ``best'' visual of the screen for a standard colormap
 * property.  Return NULL if no visual is appropriate.
 */
static XVisualInfo *getBestVisual(property, vinfo, nvisuals)
    Atom	property;	/* specifies the standard colormap */
    XVisualInfo *vinfo;		/* specifies all visuals of the screen */
    int		nvisuals;	/* specifies number of visuals of screen */
{	
    XVisualInfo	*v1 = NULL, *v2 = NULL;

    if (vinfo == NULL)		 /* unexpected: a screen with no visuals */
	return v1;

    v1 = getDeepestVisual(DirectColor, vinfo, nvisuals);
    v2 = getDeepestVisual(PseudoColor, vinfo, nvisuals);
    if (v2 && (!v1 || (v2->colormap_size >=
		       ((v1->red_mask | v1->green_mask | v1->blue_mask) + 1))))
	return v2;
    else if (v1)
	return v1;
    if (property == XA_RGB_BEST_MAP)
	if (((v1 = getDeepestVisual(TrueColor, vinfo, nvisuals)) != NULL) ||
	    ((v1 = getDeepestVisual(StaticColor, vinfo, nvisuals)) != NULL))
	    return v1;
    if (property == XA_RGB_GRAY_MAP)
	if (((v1 = getDeepestVisual(GrayScale, vinfo, nvisuals)) != NULL) ||
	    ((v1 = getDeepestVisual(StaticGray, vinfo, nvisuals)) != NULL))
	    return v1;
    return v1;

}


#ifdef PEX_SUPPORT /* [ */

#define	FREE_PEX_TARGETS 	if (ntargets) XFree (targets); \
				XFree ((char *) vinfo);

static XVisualInfo *getBestPEXVisual(property, vinfo, nvisuals)
    Atom	property;	/* specifies the standard colormap */
    XVisualInfo *vinfo;		/* specifies all visuals of the screen */
    int		nvisuals;	/* specifies number of visuals of screen */
{	
    XVisualInfo		*v0 = NULL, *v1 = NULL, *v2 = NULL, *ret_ptr = NULL;
    int			ncolors_v0 = 0, ncolors_v1 = 0, ncolors_v2 = 0;
    int			ret_colors = 0;
    unsigned long	ntargets = 0;
    char *		targets = NULL;
    PEXExtensionInfo	*ext_info;


    if (vinfo == NULL)		 /* unexpected: a screen with no visuals */
	return v1;

    if (NULL == (ext_info = PEXGetExtensionInfo (dpy))) {
	XFree ((char *) vinfo);
	return 0;
    }

    if ((ext_info->major_version == 5) &&
	(ext_info->minor_version >= 1)) {

	if (! PEXMatchRenderingTargets (dpy, 
				    RootWindow(dpy, vinfo->screen),
				    0, PEXWindowDrawable, NULL,
				    64, &ntargets, 
				    (PEXRenderingTarget**) &targets)
	  || (ntargets == 0)) {
	    XFree ((char *) vinfo);
	    return 0;
	}
    }

    if (v0 = getDeepestVisual(TrueColor, vinfo, nvisuals, targets, ntargets))
	ncolors_v0 = ((v0->red_mask | v0->green_mask | v0->blue_mask) + 1);

    if (v1 = getDeepestVisual(DirectColor, vinfo, nvisuals, targets, ntargets))
	ncolors_v1 = ((v1->red_mask | v1->green_mask | v1->blue_mask) + 1);

    if (v2 = getDeepestVisual(PseudoColor, vinfo, nvisuals, targets, ntargets))
	ncolors_v2 = v2->colormap_size;

    if (ncolors_v1 >= ncolors_v2) {
	ret_ptr = v1;
	ret_colors = ncolors_v1;
    }
    else {
	ret_ptr = v2;
	ret_colors = ncolors_v2;
    }

    if ((property == XA_RGB_BEST_MAP) || (property == XA_RGB_DEFAULT_MAP)) {
	if (ncolors_v0 >= ret_colors) {
	    ret_ptr = v0;
	    ret_colors = ncolors_v0;
	}
    }

    if (ret_ptr) {
	FREE_PEX_TARGETS
	return ret_ptr;
    }
    
    if ((property == XA_RGB_BEST_MAP) || (property == XA_RGB_DEFAULT_MAP)) {
	if ((v1 = getDeepestVisual(StaticColor, vinfo, nvisuals, 
				    targets, ntargets)) != NULL){
	    FREE_PEX_TARGETS
	    return v1;
	}
    }
    if (property == XA_RGB_GRAY_MAP) {
	if (((v1 = getDeepestVisual(GrayScale, vinfo, nvisuals, 
				    targets, ntargets)) != NULL) ||
	    ((v1 = getDeepestVisual(StaticGray, vinfo, nvisuals, 
				    targets, ntargets)) != NULL)){
	    FREE_PEX_TARGETS
	    return v1;
	}
    }
    FREE_PEX_TARGETS
    return v1;
}

#undef	FREE_PEX_TARGETS
#endif /* ] PEX_SUPPORT */


static char *visualStringFromClass(class)
    int	class;
{
    switch (class) {
      case PseudoColor: return "PseudoColor";
      case DirectColor: return "DirectColor";
      case GrayScale: return "GrayScale";
      case StaticColor: return "StaticColor";
      case TrueColor: return "TrueColor";
      case StaticGray: return "StaticGray";
    }
    return "unknown visual class";
}

static Status doFullDefault(screen, vinfo, nvisuals, replace, retain)
    int			screen;
    XVisualInfo		*vinfo;
    int			nvisuals;
    int			replace;
    int			retain;
{
    int 		j;
    XVisualInfo		*v;
    Status 		status, overall_status = 0;

    for (j=0, v=vinfo; j<nvisuals; j++, v++) {

	if ((verbose) & !all)
	    (void) fprintf(stderr,
		       "%s: making %s entry on a %s visual of depth %u.\n",
		       program_name, propertyTable[DEFAULT].name,
		       visualStringFromClass(v->class), v->depth);

	status = XmuLookupStandardColormapExt(dpy, screen, v->visualid,
				   v->depth,
				   propertyTable[DEFAULT].property,
				   replace, retain, &params);
	overall_status |= status;

	if ((verbose) & !all)
	    (void) fprintf(stderr,
		       "%s: %s standard colormap entry %s.\n", program_name,
		       propertyTable[DEFAULT].name, (status)
		       ? "was created or already exists"
		       : "cannot be defined");
    }

    return overall_status;
}

static int doIndividualColormaps()
{
    int			i, screen, nvisuals;
    Status		status;
    XVisualInfo		*vinfo = NULL, *v = NULL, template;
    
    /*
	Note:   This procedure only creates properties on the default
	        screen for the connection.   If PEX is required, it will
		return a failure status if PEX is not supported on the
		default screen.  That means that -pex with any of the 
		individual options (-best, -default, etc.) might fail, 
		but -pex -all might succeed since it's happy with PEX support
		on any one screen.
    */
    screen = DefaultScreen(dpy);
    template.screen = screen;
    vinfo = XGetVisualInfo(dpy, VisualScreenMask, &template, &nvisuals);

    /* check for individual standard colormap requests */
    for (i=0; i < NPROPERTIES; i++) {

	if (propertyTable[i].delete) {
	    XmuDeleteStandardColormap(dpy, screen, propertyTable[i].property);
	    if (verbose)
		fprintf(stderr, "%s: %s was deleted or did not exist.\n",
			program_name, propertyTable[i].name);
	}

	if (! propertyTable[i].create)	
	    continue;
	
	if ((propertyTable[i].property == XA_RGB_DEFAULT_MAP) && 
	    full_default)

	    status = doFullDefault(screen, vinfo, nvisuals,
				   DO_NOT_REPLACE, RETAIN);

	else {
	    /* which visual is best for this property? */
#ifdef PEX_SUPPORT /* [ */
	    if (pex)
		v = getBestPEXVisual(propertyTable[i].property, vinfo,nvisuals);
	    else
#endif /* ] PEX_SUPPORT */
		v = getBestVisual(propertyTable[i].property, vinfo, nvisuals);

	    if (v == NULL) {
		if (verbose)
		    (void) fprintf(stderr,
			   "%s: no visual appropriate for %s on screen %d.\n",
			    program_name, propertyTable[i].name, screen);
		continue;
	    }

	    if (verbose)
		(void) fprintf(stderr,
			       "%s: making %s on a %s visual of depth %u.\n",
			       program_name, propertyTable[i].name,
			       visualStringFromClass(v->class), v->depth);
	    
	    status = XmuLookupStandardColormapExt(dpy, screen, v->visualid,
					       v->depth,
					       propertyTable[i].property,
					       DO_NOT_REPLACE, RETAIN, 
					       &params);
	    if (verbose)
		(void) fprintf(stderr,
			       "%s: %s standard colormap %s.\n", program_name,
			       propertyTable[i].name, (status)
			       ? "was created or already exists"
			       : "cannot be defined");
	}

	if (!status)
	    break;
    }
    XFree((char *) vinfo);
    return status;
}

/* Bare bones standard colormap generation utility */
main(argc, argv)
    int		argc;
    char	**argv;
{
    Status	status = 0;
    XmuParam    param_array[3];
    int		param_index;

    if (program_name = rindex(*argv, '/'))
	program_name++;
    else
	program_name = *argv;

    parse(argc, argv);

    if ((dpy = XOpenDisplay(display_name)) == NULL) {
	(void) fprintf(stderr, "%s: cannot open display \"%s\".\n",
		       program_name, XDisplayName(display_name));
	exit(1);
    }

    if (help) {
	usage(0);
	Exit(0);
    }

/*
XSynchronize (dpy, 1);
*/

    /*
	Set up controls and hints.  When modifying this code, make sure
	all controls are set up before any hints.  Also, you may need to
	increase size of param_array, above.
    */

    param_index = 0;
    params.num_controls = 0;
    params.controls = &(param_array[param_index]);

#ifdef PEX_SUPPORT /* [ */
    if (pex) {
	PEXExtensionInfo *ext_info;
	char error_string[256];

	if (PEXInitialize (dpy, &ext_info, 255, error_string)) {
	    status = 0;
	    (void) fprintf(stderr,
			   "%s: cannot initialize PEX on \"%s\".\n",
			   program_name, XDisplayName(display_name));
	    error_string[255] = '\0';
	    (void) fprintf(stderr,
			   "%s: PEX error string is \"%s\".\n",
			   program_name, error_string);
	    exit(1);
	}

	params.num_controls++;
	param_array[param_index].name = XMU_CONTROL_PEX_SUPPORTED;
	param_array[param_index].value = True;
	param_index++;
    }
#endif /* ] PEX_SUPPORT */

    params.num_hints = 0;
    params.hints = &(param_array[param_index]);

    if (ramp_hint) {
	params.num_hints++;
	param_array[param_index].name = XMU_HINT_PSEUDO_RAMP;
	param_array[param_index].value = ramp_hint;
	param_index++;
    }

    if (full_default) {
	params.num_hints++;
	param_array[param_index].name = XMU_HINT_CREATE_DEFAULT;
	param_array[param_index].value = True;
	param_index++;
    }


    /*
	Now create the properties.
    */
    if (all) {

	if (verbose)
	    (void) fprintf(stderr,
			   "%s: making all appropriate standard colormaps...",
			   program_name);
	status = XmuAllStandardColormapsExt(dpy, &params);

	if (status && full_default) {

	    int			screen, nvisuals;
	    XVisualInfo		*vinfo = NULL, template;
	    int			overall_status = 1;

	    /* for each screen, determine all visuals of this server */
	    for (screen=0; screen < ScreenCount(dpy); screen++)
	    {
		template.screen = screen;
		vinfo = XGetVisualInfo(dpy, VisualScreenMask, &template, 
					&nvisuals);
		if (vinfo == NULL)
		    continue;

		status = doFullDefault(screen, vinfo, nvisuals, 
					REPLACE, RETAIN);
		overall_status &= status;

		XFree ((char *) vinfo);

		if (!status) break;
	    }
	    status = overall_status;
	}

	if (verbose)
	    (void) fprintf(stderr,
			   "\n%s!\n", (status) ? "success" : "failure");
    }
    else {
	status = doIndividualColormaps(&params);
	if (!status && verbose)
	    (void) fprintf(stderr, 
		    "Not all new colormap definitions will be retained.\n");
    }
    Exit((status == 0) ? 1 : 0);
}
