/* spline.c: Interpolation
** Spring 1993
**
** Written and Copyright (C) 1993, 1994 by Michael J. Gourlay
**
** NO WARRANTEES, EXPRESS OR IMPLIED.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "spline.h"
#include "my_malloc.h"

/* --------------------------------------------------------------- */

#define MAX(a,b)      ((a)>(b)?(a):(b))
#define MIN(a,b)      ((a)<(b)?(a):(b))
#define ABS(a)        ((a)>=0?(a):(-(a)))
#define NEAR(x1, x2)  (((x2)!=0.0) && (((x1)/(x2)) >= 0.999) && (((x1)/(x2))<1.001))

#ifndef FALSE
#define FALSE 0
#endif

/* --------------------------------------------------------------- */

/* derivative_hack: compute 1st derivative of x,y data (len entries)
**
** Written and Copyright (C) 1994 by Michael J. Gourlay
**
** NO WARRANTEES, EXPRESS OR IMPLIED.
**
** Mathematically, it's a hack to prevent overshooting knots, but
** maintain smoothness and it works more intuitively.  Besides, we all
** know that mathematicians are too worried about rigor, and the
** physicists end up creating the right math. -- MJG
*/
static void
derivative_hack(double *x, double *y, double *yd, int len)
{
  int indx;

  if(x[0]==x[1]) {
    yd[0] = 0.0; /* avoid division by zero */
  } else {
    yd[0] = (y[1]-y[0])/(x[1]-x[0]);
  }
  if(x[len-2] == x[len-1]) {
    yd[len-1] = 0.0; /* avoid division by zero */
  } else {
    yd[len-1] = (y[len-1]-y[len-2])/(x[len-1]-x[len-2]);
  }
  for(indx=1; indx<(len-1); indx++) {
    if(x[indx-1]==x[indx] || x[indx]==x[indx+1]) {
      yd[indx] = 0.0; /* avoid division by zero */
    } else {
      if( (y[indx-1] - y[indx]) * (y[indx+1] - y[indx]) >= 0.0) {
        /* There was a change in the sign of yd so force it zero */
        /* This will prevent the spline from overshooting this knot */
        yd[indx] = 0.0;
      } else {
        /* Set slope at this knot to slope between two adjacent knots */
        yd[indx] = (y[indx-1]-y[indx+1]) / (x[indx-1]-x[indx+1]);
      }
    }
  }
}

/* --------------------------------------------------------------- */
/* spline3_setup: set parameters for natural cubic spline
** input: x, y, n
** output: c, h
*/
void
spline3_setup(x, y, n, c, h)
double *x, *y; /* knots */
long   n;      /* number of knots */
double *c;     /* spline parameters */
double *h;     /* intervals: h[i]=x[i+1]-x[i] */
{
  double *u, *v;
  long i;

  /* only need u and v to index from 1 to n-1 */
  u=(double *)my_malloc(sizeof(double)*n, "spline3_setup");
  v=(double *)my_malloc(sizeof(double)*n, "spline3_setup");

  for(i=0; i<n; i++) {
    h[i]=x[i+1]-x[i];
  }

  for(i=1; i<n; i++) {
    v[i]=3.0/h[i]*(y[i+1]-y[i]) - 3.0/h[i-1]*(y[i]-y[i-1]);
    u[i]=2.0*(h[i]+h[i-1]);
  }

  c[0]=c[n]=0;
  for(i=n-1; i>0; i--) {
    c[i]=(v[i]-h[i]*c[i+1])/u[i];
  }

  free(u);
  free(v);
}

/* --------------------------------------------------------------- */
/* spline3_eval -- evaluate the natural cubic spline
** input : w, x, y, n, c, h
** output : s1, s2
** return: spline evaluation
**  if s1==NULL then s1 is not evaluated
**  if s2==NULL then s2 is not evaluated
*/
double
spline3_eval(w, x, y, n, c, h, s1, s2)
double w;   /* argument, point at which spline is evaluated */
double *x;  /* array of knot x values, in increasing order */
double *y;  /* array of knot y values */
long n;     /* number of x's and y's */
double *c;  /* array of parameters from spline3_setup */
double *h;  /* array of intervals between x's */
double *s1; /* spline first derivative */
double *s2; /* spline second derivative */
{
  double diff=0.0;
  double b, d;
  long i;

  /* find interval of spline to evaluate */
  for(i=n-1; (i>=0) && ((diff=(w-x[i]))<0.0); i--)
   ;

  /* calculate other spline parameters */
  /* here a=y[i] so it is not explicitly named a */
  b = (y[i+1]-y[i])/h[i] - h[i]/3.0*(2.0*c[i] + c[i+1]);
  d = (c[i+1]-c[i])/h[i];

  /* evaluate derivatives of spline */
  if(s1!=NULL) *s1 = b + diff*(2.0*c[i] + 3.0*d*diff);
  if(s2!=NULL) *s2 = 2.0*(c[i] + 3.0*d*diff);

  /* return spline value */
  return(y[i] + diff*(b + diff*(c[i] + diff*d)));
}


/* --------------------------------------------------------------- */
/* d_parabola -- returns the derivative of a parabola fit through 3 points */
double
d_parabola(x, xp0, xp1, xp2, yp0, yp1, yp2)
double x;          /* place on parabola where derivative is taken */
double xp0, xp1, xp2; /* points through which parabola passes */
double yp0, yp1, yp2;
{
  double dP=(  xp0*(yp1-yp2)*(2*x - xp0)
             + xp1*(yp2-yp0)*(2*x - xp1)
             + xp2*(yp0-yp1)*(2*x - xp2)) / ((xp0-xp1)*(xp0-xp2)*(xp2-xp1));

  return(dP);
}

/* --------------------------------------------------------------- */
/* hermite3_interp : cubic hermite interpolation
** MJG corrected 18jul94 -- was reading beyond bounds at last knot.
** input : w, x, y, d, n, f, find_d
** output : h1, h2
** return : evaluation of interpolation
*/
double
hermite3_interp(w, x, y, d, n, f, find_d, h1, h2)
double w;      /* evaluation point */
double *x, *y; /* arrays of knots */
double *d;     /* derivatives */
long n;        /* number of knots */
double (*f)(); /* derivative function, NULL=>approximate from knots */
int find_d;    /* find_derivative flag, 0=>use d, 1=>find d */
double *h1;    /* first derivative of spline, NULL=>ignore */
double *h2;    /* second derivative of spline, NULL=>ignore */
{
  double A, B;
  double h, h_2;
  double diff=0.0;
  double H;
  long si;

  /* find interval of spline to evaluate */
  for(si=n-2; (si>=0) && ((diff=(w-x[si]))<0.0); si--)
    ;

  /* h is the interval between knots */
  h = x[si+1]-x[si];
  h_2 = h*h;

  /* either the derivatives were provided or must be found */
  if(find_d) {
    /* must calculate derivatives */
    if(f!=NULL) {
      /* calculate the derivative */
      d[si]=(*f)(x[si]);
      d[si+1]=(*f)(x[si+1]);
    } else {
      /* approximate derivative using parabola fit */
      if(si==0) { /* at first knot */
        d[si]=d_parabola(x[si], x[si], x[si+1], x[si+2],
                              y[si], y[si+1], y[si+2]);
        d[si+1]=d_parabola(x[si+1], x[si], x[si+1], x[si+2],
                                  y[si], y[si+1], y[si+2]);
      } else if(si>=(n-2)) { /* at or near last knot */
        d[si]=d_parabola(x[si], x[si-1], x[si], x[si+1],
                              y[si-1], y[si], y[si+1]);
        d[si+1]=d_parabola(x[si+1], x[si-1], x[si], x[si+1],
                                  y[si-1], y[si], y[si+1]);
      } else { /* between first and 2nd to last knot */
        d[si]=d_parabola(x[si], x[si-1], x[si], x[si+1],
                              y[si-1], y[si], y[si+1]);
        d[si+1]=d_parabola(x[si+1], x[si], x[si+1], x[si+2],
                                  y[si], y[si+1], y[si+2]);
      }
    }
  }

  /* calculate interpolant parameters */
  A = (y[si+1] - y[si] - h*d[si])/h_2;
  B = (d[si+1] - d[si] -2.0*h*A)/h_2;

  /* evaluate spline derivatives */
  if(h1!=NULL) *h1 = d[si] + diff*(2.0*A + B*(diff + 2.0*(w-x[si+1])));
  if(h2!=NULL) *h2 = 2.0*A + 2.0*B*(2.0*diff + (w-x[si+1]));

  /* return the spline evaluation */
  H  =  y[si] + diff*(d[si] + diff*(A + (w-x[si+1])*B));

  return(H);
}

/* --------------------------------------------------------------- */
/* hermite3_array : cubic hermite interpolation for an array of points
** Uses derivative_hack to find derivatives.
** input : kx, ky, nk, sx, sy, ns (sy only need be allocated; contents ignored)
** output : sy (spline value at evaluation points sx)
*/
int
hermite3_array(kx, ky, nk, sx, sy, ns)
double *kx, *ky; /* arrays of knots */
long nk;         /* number of knots */
double *sx;      /* evaluation points array (input) */
double *sy;      /* evaluation values array (output) */
long   ns;       /* number of evaluation points */
{
  register long xi;
  double *kyd;

  if((kyd=(double*)my_calloc(nk, sizeof(double), "hermite3_array"))==NULL)
    return(1);

  /* Test bounds. */
  /* As of 18jul94, this test was triggering for cases
  ** where the bounds were nearly equal, but slightly out of range, in
  ** which case the spline should work anyway, which is why I let it run
  ** even if the spline abcissas are out of range.
  */
  if((sx[0] < kx[0]) || (sx[ns-1] > kx[nk-1])) {
    if(!NEAR(sx[ns-1], kx[nk-1])) {
      fprintf(stderr, "hermite3_array: out of range.\n");
      fprintf(stderr, "hermite3_array: %.20lg<%.20lg | %.20lg>%.20lg\n",
        sx[0], kx[0], sx[ns-1], kx[nk-1]);
    }
  }

  /* Find array of derivatives */
  derivative_hack(kx, ky, kyd, nk);

  /* Evaluate the spline */
  for(xi=0; xi<ns; xi++) {
    sy[xi]=hermite3_interp(sx[xi], kx, ky, kyd, nk, NULL, FALSE, NULL, NULL);
  }

  return(0);
}
