/*
 * Java-Gnome Bindings Library
 * 
 * Copyright 1998-2002 the Java-Gnome Team, all rights reserved.
 * 
 * The Java-Gnome Team Members: Jean Van Wyk <jeanvanwyk@iname.com> Jeffrey S.
 * Morgan <jeffrey.morgan@bristolwest.com> Dan Bornstein <danfuzz@milk.com> Tom
 * Ball <tom.ball@sun.com>
 * 
 * The Java-Gnome bindings library is free software distributed under the terms
 * of the GNU Library General Public License version 2.
 */

package org.gnu.glib;

import org.gnu.glib.GObject;
import org.gnu.gtk.event.GtkEventType;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * An EventMap is a Java-GNOME implementation class which encapsulates all
 * static information that maps GTK signals to Java-GNOME events. There should
 * be one private static EventMap for each Java-GNOME class which fires events.
 * 
 * Besides reducing the code needed to add event handling to a widget instance,
 * this mapping is needed by libglade to look up which event is associated with
 * which signal for a given class. It should not be used by client code,
 * however.
 * 
 * @author Tom Ball
 */
public class EventMap {

	private List events = new ArrayList();

	private class Entry {
		String signal;
		String method;
		GtkEventType eventType;
		Class listenerClass;
		Entry(String s, String m, GtkEventType t, Class l) {
			signal = s;
			method = m;
			eventType = t;
			listenerClass = l;
		}
	}

	public void addEvent(String signal, String method, GtkEventType eventType, Class listenerClass) {
		events.add(new Entry(signal, method, eventType, listenerClass));
	}

	/**
	 * Connect all event handlers to this event source.
	 */
	public void initialize(GObject source) {
		Iterator i = events.iterator();
		while (i.hasNext()) {
			Entry e = (Entry) i.next();
			source.addEventHandler(e.signal, e.method, source);
		}
	}

	/**
	 * Return the event listener class for a given signal.
	 */
	public Class getListenerClass(String signal) {
		Iterator i = events.iterator();
		while (i.hasNext()) {
			Entry e = (Entry) i.next();
			if (e.signal.equals(signal))
				return e.listenerClass;
		}
		return null;
	}

	/**
	 * Get the event type for a given signal.
	 */
	public GtkEventType getEventType(String signal) {
		Iterator i = events.iterator();
		while (i.hasNext()) {
			Entry e = (Entry) i.next();
			if (e.signal.equals(signal))
				return e.eventType;
		}
		return null;
	}
}
