/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import java.util.Vector;

import org.gnu.glib.EventMap;
import org.gnu.glib.GObject;
import org.gnu.glib.Type;
import org.gnu.gtk.event.TextViewEvent;
import org.gnu.gtk.event.TextViewListener;
import org.gnu.gtk.event.GtkEventType;

/**
 * <h2>Conceptual Overview</h2>
 *
 * <p>java-gnome has an extremely powerful framework for multiline text editing. The
 * primary objects involved in the process are {@link TextBuffer}, which represents
 * the text being edited, and {@link TextView}, a widget which can display a
 * TextBuffer. Each buffer can be displayed by any number of views.
 * 
 * <p>One of the important things to remember about text in java-gnome is that it's in
 * the UTF-8 encoding. This means that one character can be encoded as multiple
 * bytes. Character counts are usually referred to as offsets, while byte counts
 * are called indexes. If you confuse these two, things will work fine with
 * ASCII, but as soon as your buffer contains multibyte characters, bad things
 * will happen.
 *
 * <p>Text in a buffer can be marked with tags. A tag is an attribute that can
 * be applied to some range of text. For example, a tag might be called "bold"
 * and make the text inside the tag bold. However, the tag concept is more
 * general than that; tags don't have to affect appearance. They can instead
 * affect the behavior of mouse and key presses, "lock" a range of text so the
 * user can't edit it, or countless other things. A tag is represented by a
 * {@link TextTag} object. One TextTag can be applied to any number of text
 * ranges in any number of buffers.
 *
 * <p>Each tag is stored in a {@link TextTagTable}. A tag table defines a set of
 * tags that can be used together. Each buffer has one tag table associated with
 * it; only tags from that tag table can be used with the buffer. A single tag
 * table can be shared between multiple buffers, however.
 *
 * <p>Tags can have names, which is convenient sometimes (for example, you can
 * name your tag that makes things bold "bold"), but they can also be anonymous
 * (which is convenient if you're creating tags on-the-fly).
 *
 * <p>Most text manipulation is accomplished with iterators, represented by a
 * {@link TextIter}. An iterator represents a position between two characters in
 * the text buffer. Iterators are not valid indefinitely; whenever the buffer is
 * modified in a way that affects the number of characters in the buffer, all
 * outstanding iterators become invalid. (Note that deleting 5 characters and
 * then reinserting 5 still invalidates iterators, though you end up with the
 * same number of characters you pass through a state with a different number).
 *
 * <p>Because of this, iterators can't be used to preserve positions across
 * buffer modifications. To preserve a position, the {@link TextMark} object is
 * ideal. You can think of a mark as an invisible cursor or insertion point; it
 * floats in the buffer, saving a position. If the text surrounding the mark is
 * deleted, the mark remains in the position the text once occupied; if text is
 * inserted at the mark, the mark ends up either to the left or to the right of
 * the new text, depending on its gravity. The standard text cursor in
 * left-to-right languages is a mark with right gravity, because it stays to the
 * right of inserted text.
 *
 * <p>Like tags, marks can be either named or anonymous. There are two marks
 * built-in to {@link TextBuffer}; these are named "insert" and
 * "selection_bound" and refer to the insertion point and the boundary of the
 * selection which is not the insertion point, respectively. If no text is
 * selected, these two marks will be in the same position. You can manipulate
 * what is selected and where the cursor appears by moving these marks around.
 *
 * <p>Text buffers always contain at least one line, but may be empty (that is,
 * buffers can contain zero characters). The last line in the text buffer never
 * ends in a line separator (such as newline); the other lines in the buffer
 * always end in a line separator. Line separators count as characters when
 * computing character counts and character offsets. Note that some Unicode line
 * separators are represented with multiple bytes in UTF-8, and the
 * two-character sequence "\r\n" is also considered a line separator.
 */
public class TextView extends Container {

	/**
	 * The list of objects interested in this widget's events
	 */
	private Vector listeners;

	/**
	 * Constructs a new TextView. If you don't specify a buffer before using it,
	 * a default one will be created for you.
	 */
	public TextView() {
		super(gtk_text_view_new());
		initializeEventHandlers();
	}

	/**
	 * Creates a new TextView widget displaying the buffer buffer. One buffer
	 * can be shared among many widgets. 
	 * @param buffer Buffer to use
	 */
	public TextView(TextBuffer buffer) {
		super(gtk_text_view_new_with_buffer(buffer.getHandle()));
		initializeEventHandlers();
	}

	/**
	 * Construct a TextView from a handle to a native resource.
	 */
	public TextView(int handle) {
		super(handle);
		initializeEventHandlers();
	}

	/**
	 * Sets buffer as the buffer being displayed by the view.
	 * @param buffer The new buffer to display
	 */
	public void setBuffer(TextBuffer buffer) {
		gtk_text_view_set_buffer(getHandle(), buffer.getHandle());
	}

	/**
	 * Returns the buffer being used
	 * @return The buffer used in the widget.
	 */
	public TextBuffer getBuffer() {
		int hndl = gtk_text_view_get_buffer(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (TextBuffer)obj;
		return new TextBuffer(hndl);
	}
	/**
	 * Scrolls the view so that mark is on the screen in the position indicated
	 * by xAlign and yAlign. An alignment of 0.0 indicates left or top, 1.0
	 * indicates right or bottom, 0.5 means center. 
	 * @param mark A {@link TextMark}
	 * @param withinMargin Margin as a fraction of screen size
	 * @param useAlign Whether to use alignment arguments.
	 * @param xAlign Horizontal alignment of mark within visible area.
	 * @param yAlign : vertical alignment of mark within visible area
	 */
	public void scrollToMark(TextMark mark, double withinMargin, boolean useAlign, double xAlign, double yAlign) {
		gtk_text_view_scroll_to_mark(getHandle(), mark.getHandle(), withinMargin, useAlign, xAlign, yAlign);
	}

	/**
	 * Scrolls the view so that mark is on the screen. The
	 * text scrolls the minimal distance to get the mark onscreen, possibly not
	 * scrolling at all. The effective screen for purposes of this function is
	 * reduced by a margin of size withinMargin.
	 * @param mark A {@link TextMark}
	 * @param withinMargin Margin as a fraction of screen size
	 */
	public void scrollToMark(TextMark mark, double withinMargin) {
		gtk_text_view_scroll_to_mark(getHandle(), mark.getHandle(), withinMargin, false, 0.0, 0.0);
	}

	/**
	 * Scrolls text view so that iter is on the screen in the position indicated
	 * by xalign and yalign. An alignment of 0.0 indicates left or top, 1.0
	 * indicates right or bottom, 0.5 means center. 
	 * The effective screen for purposes of this function is
	 * reduced by a margin of size within_margin. NOTE: This function uses the
	 * currently-computed height of the lines in the text buffer. Note that line
	 * heights are computed in an idle handler; so this function may not have
	 * the desired effect if it's called before the height computations. To
	 * avoid oddness, consider using gtk_text_view_scroll_to_mark() which saves
	 * a point to be scrolled to after line validation.
	 * @param iter a TextIter
	 * @param withinMargin Margin as a [0.0,0.5) fraction of screen size.
	 * @param xAlign : horizontal alignment of mark within visible area.
	 * @param yAlign : vertical alignment of mark within visible area
	 * @return TRUE if scrolling occurred
	 */
	public boolean scrollToIter(TextIter iter, double withinMargin, double xAlign, double yAlign) {
		return gtk_text_view_scroll_to_iter(getHandle(), iter.getHandle(), withinMargin, true, xAlign, yAlign);
	}

	/**
	 * Scrolls text view so that iter is on the screen.
	 * The effective screen for purposes of this function is
	 * reduced by a margin of size within_margin. NOTE: This function uses the
	 * currently-computed height of the lines in the text buffer. Note that line
	 * heights are computed in an idle handler; so this function may not have
	 * the desired effect if it's called before the height computations. To
	 * avoid oddness, consider using gtk_text_view_scroll_to_mark() which saves
	 * a point to be scrolled to after line validation.
	 * @param iter a TextIter
	 * @param withinMargin Margin as a [0.0,0.5) fraction of screen size.
	 * @return TRUE if scrolling occurred
	 */
	public boolean scrollToIter(TextIter iter, double withinMargin) {
		return gtk_text_view_scroll_to_iter(getHandle(), iter.getHandle(), withinMargin, false, 0, 0);
	}

	/**
	 * Moves the cursor to the currently visible region of the buffer, it it
	 * isn't there already.
	 * @return  TRUE if the cursor had to be moved.
	 */
	public boolean moveCursorOnScreen() {
		return gtk_text_view_place_cursor_onscreen(getHandle());
	}

	/**
	 * Adds a child widget in the text buffer, at the given anchor.
	 * @param child A Widget
	 * @param anchor A TextChildAnchor in the TextBuffer for his view
	 */
	public void addChild(Widget child, TextChildAnchor anchor) {
		gtk_text_view_add_child_at_anchor(getHandle(), child.getHandle(), anchor.getHandle());
	}

	/**
	 * Sets the line wrapping for the view.
	 * @param wrapMode A WrapMode
	 */
	public void setWrapMode(WrapMode wrapMode) {
		gtk_text_view_set_wrap_mode(getHandle(), wrapMode.getValue());
	}

	/**
	 * Sets the default editability of the GtkTextView. You can override this
	 * default setting with tags in the buffer, using the "editable" attribute
	 * of tags.  
	 * @param setting Whether it's editable
	 */
	public void setEditable(boolean setting) {
		gtk_text_view_set_editable(getHandle(), setting);
	}

	/**
	 * Returns the default editability of the GtkTextView. Tags in the buffer
	 * may override this setting for some ranges of text.
	 * @return true if the widget is editable.
	 */
	public boolean getEditable() {
		return gtk_text_view_get_editable(getHandle());
	}

	/**
	 * Toggles whether the insertion point is displayed. A buffer with no
	 * editable text probably shouldn't have a visible cursor, so you may want
	 * to turn the cursor off.
	 * @param setting Whether to show the insertion cursor
	 */
	public void setCursorVisible(boolean setting) {
		gtk_text_view_set_cursor_visible(getHandle(), setting);
	}

	/**
	 * Sets the default number of blank pixels above paragraphs in text_view.
	 * Tags in the buffer for text_view may override the defaults.
	 * @param pixelsAboveLines Pixels above paragraphs
	 */
	public void setPixelsAboveLines(int pixelsAboveLines) {
		gtk_text_view_set_pixels_above_lines(getHandle(), pixelsAboveLines);
	}

	/**
	 * Sets the default number of pixels of blank space to put below paragraphs
	 * in text_view. May be overridden by tags applied to text_view's buffer.
	 * @param pixelsBelowLines Pixels below paragraphs
	 */
	public void setPixelsBelowLine(int pixelsBelowLines) {
		gtk_text_view_set_pixels_below_lines(getHandle(), pixelsBelowLines);
	}

	/**
	 * Sets the default number of pixels of blank space to leave between
	 * display/wrapped lines within a paragraph. May be overridden by tags in
	 * text_view's buffer.
	 * @param pixelsInsideWrap Default number of pixels between wrapped lines
	 */
	public void setPixelsInsideWrap(int pixelsInsideWrap) {
		gtk_text_view_set_pixels_inside_wrap(getHandle(), pixelsInsideWrap);
	}

	/**
	 * Sets the default justification of text in text_view. Tags in the view's
	 * buffer may override the default.  
	 * @param justification The justification to use.
	 */
	public void setJustification(Justification justification) {
		gtk_text_view_set_justification(getHandle(), justification.getValue());
	}

	/**
	 * Sets the default left margin for text in text_view. Tags in the buffer
	 * may override the default.
	 * @param leftMargin Size of left margin, in pixels
	 */
	public void setLeftMargin(int leftMargin) {
		gtk_text_view_set_left_margin(getHandle(), leftMargin);
	}

	/**
	 * Sets the default right margin for text in text_view. Tags in the buffer
	 * may override the default.
	 * @param rightMargin Size of right margin, in pixels
	 */
	public void setRightMargin(int rightMargin) {
		gtk_text_view_set_right_margin(getHandle(), rightMargin);
	}

	/**
	 * Sets the default indentation for paragraphs in text_view. Tags in the
	 * buffer may override the default.
	 * @param indent Indentation in pixels
	 */
	public void setIndent(int indent) {
		gtk_text_view_set_indent(getHandle(), indent);
	}

	/**
	 * Sets the default tab stops for paragraphs in text_view. Tags in the
	 * buffer may override the default.
	 * @param tabStops  tabs as a PangoTabArray
	 */
	public void setTabs(org.gnu.pango.TabArray tabStops) {
		gtk_text_view_set_tabs(getHandle(), tabStops.getHandle());
	}

	/**
	 * Obtains a copy of the default text attributes. These are the attributes
	 * used for text unless a tag overrides them.
	 * 
	 * @return The text attributes being used
	 */
	public TextAttributes getDefaultAttributes() {
		return new TextAttributes(gtk_text_view_get_default_attributes(getHandle()));
	}

	/**
	 * Sets the behavior of the text widget when the Tab key is pressed. If
	 * TRUE, a tab character is inserted. If is FALSE, the keyboard focus is
	 * moved to the next widget in the focus chain.
	 *
	 * @param acceptsTab TRUE if pressing the Tab key should insert a tab character, FALSE, if pressing the Tab key should move the keyboard focus
	 *
	 * @since 2.4
	 */
	public void setAcceptsTab( boolean acceptsTab ){
		gtk_text_view_set_accepts_tab( getHandle(), acceptsTab );
	}

	/**
	 * Returns whether pressing the Tab key inserts a tab characters.
	 *
	 * @see #setAcceptsTab(boolean)
	 *
	 * @since 2.4
	 */
	public boolean getAcceptsTab(){
		return gtk_text_view_get_accepts_tab   (getHandle());
	}

	/**
	 * Changes the overwrite mode (whether text is overwritten)
	 *
	 * @param overwrite  TRUE to turn on overwrite mode, FALSE to turn it off
	 * 
	 * @since 2.4
	 */
	public void setOverwrite( boolean overwrite ){
		gtk_text_view_set_overwrite ( getHandle(), overwrite );
	}

	/**
	 * Returns whether the TextView is in overwrite mode or not.
	 * @see #getOverwrite(boolean)
	 * @since 2.4
	 */
	public boolean getOverwrite(){
		return gtk_text_view_get_overwrite (getHandle());
	}

	/**
	 * Initialize this widget to receive its' own events so it can
	 * propogate them to the listeners.
	 */
	protected void initializeEventHandlers() {
		if (!eventsInitialized) {
			super.initializeEventHandlers();
			evtMap.initialize(this);
			eventsInitialized = true;
		}
	}

	public Class getEventListenerClass(String signal) {
		Class cls = evtMap.getEventListenerClass(signal);
		if (cls == null) cls = super.getEventListenerClass(signal);
		return cls;
	}

	public GtkEventType getEventType(String signal) {
		GtkEventType et = evtMap.getEventType(signal);
		if (et == null) et = super.getEventType(signal);
		return et;
	}

	private static EventMap evtMap = new EventMap();
	static {
		addEvents(evtMap);
	}

	/**
	* Implementation method to build an EventMap for this widget class.
	* Not useful (or supported) for application use.
	*/
	private static void addEvents(EventMap anEvtMap) {
		anEvtMap.addEvent(
			"copy_clipboard",
			"handleCopyClipboard",
			TextViewEvent.Type.COPY_CLIPBOARD,
			TextViewListener.class);
		anEvtMap.addEvent(
			"cut_clipboard",
			"handleCutClipboard",
			TextViewEvent.Type.CUT_CLIPBOARD,
			TextViewListener.class);
		anEvtMap.addEvent(
			"paste_clipboard",
			"handlePasteClipboard",
			TextViewEvent.Type.PASTE_CLIPBOARD,
			TextViewListener.class);
		anEvtMap.addEvent(
			"delete_from_cursor",
			"handleDeleteFromCursor",
			TextViewEvent.Type.DELETE_FROM_CURSOR,
			TextViewListener.class);
		anEvtMap.addEvent(
			"insert_at_cursor",
			"handleInsertAtCursor",
			TextViewEvent.Type.INSERT_AT_CURSOR,
			TextViewListener.class);
		anEvtMap.addEvent("move_cursor", "handleMoveCursor", TextViewEvent.Type.MOVE_CURSOR, TextViewListener.class);
		anEvtMap.addEvent("move_focus", "handleMoveFocus", TextViewEvent.Type.MOVE_FOCUS, TextViewListener.class);
		anEvtMap.addEvent(
			"page_horizontally",
			"handlePageHorizontally",
			TextViewEvent.Type.PAGE_HORIZONTALLY,
			TextViewListener.class);
		anEvtMap.addEvent(
			"populate_popup",
			"handlePopulatePopup",
			TextViewEvent.Type.POPULATE_POPUP,
			TextViewListener.class);
		anEvtMap.addEvent("set_anchor", "handleSetAnchor", TextViewEvent.Type.SET_ANCHOR, TextViewListener.class);
		anEvtMap.addEvent(
			"set_scroll_adjustments",
			"handleSetScrollAdjustments",
			TextViewEvent.Type.SET_SCROLL_ADJUSTMENTS,
			TextViewListener.class);
		anEvtMap.addEvent(
			"toggle_overwrite",
			"handleToggleOverwrite",
			TextViewEvent.Type.TOGGLE_OVERWRITE,
			TextViewListener.class);
	}

	/****************************************
	 * TextView event handling.
	 ****************************************/
	/**
	 * Register an object to receive text view event notification.
	 * 
	 * @param listener The object that has implemented the 
	 * TextViewListener interface that is to receive the text view events.
	 */
	public void addListener(TextViewListener listener) {
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(listeners, listener);
		if (i == -1) {
			if (null == listeners) {
				listeners = new Vector();
			}
			listeners.addElement(listener);
		}
	}

	/**
	 * Unregister an object that was receiving text view event notification.
	 * 
	 * @param listener The object that is to no longer receive
	 * text view events.
	 */
	public void removeListener(TextViewListener listener) {
		int i = findListener(listeners, listener);
		if (i > -1) {
			listeners.remove(i);
		}
		if (0 == listeners.size()) {
			listeners = null;
		}
	}

	protected void fireTextViewEvent(TextViewEvent event) {
		if (null == listeners) {
			return;
		}
		int size = listeners.size();
		int i = 0;
		while (i < size) {
			TextViewListener tvc = (TextViewListener)listeners.elementAt(i);
			tvc.textViewEvent(event);
			i++;
		}
	}

	private void handleCopyClipboard() {
		fireTextViewEvent(new TextViewEvent(this, TextViewEvent.Type.COPY_CLIPBOARD));
	}

	private void handleCutClipboard() {
		fireTextViewEvent(new TextViewEvent(this, TextViewEvent.Type.CUT_CLIPBOARD));
	}

	private void handlePasteClipboard() {
		fireTextViewEvent(new TextViewEvent(this, TextViewEvent.Type.PASTE_CLIPBOARD));
	}

	private void handleInsertAtCursor(String str) {
		fireTextViewEvent(new TextViewEvent(this, TextViewEvent.Type.INSERT_AT_CURSOR));
	}

	private void handleDeleteFromCursor(int deleteType, int count) {
		fireTextViewEvent(new TextViewEvent(this, TextViewEvent.Type.DELETE_FROM_CURSOR));
	}

	private void handleMoveCursor(int step, int count, boolean extendedSelection) {
		TextViewEvent tve = new TextViewEvent(this, TextViewEvent.Type.MOVE_CURSOR);
		tve.setMovementStep(MovementStep.intern(step));
		tve.setCount(count);
		fireTextViewEvent(tve);
	}

	private void handleMoveFocus(int direction) {
		fireTextViewEvent(new TextViewEvent(this, TextViewEvent.Type.MOVE_FOCUS));
	}

	private void handlePageHorizontally(int count, boolean extendedSelection) {
		fireTextViewEvent(new TextViewEvent(this, TextViewEvent.Type.PAGE_HORIZONTALLY));
	}

	private void handlePopulatePopup(int menu) {
		TextViewEvent tve = new TextViewEvent(this, TextViewEvent.Type.POPULATE_POPUP);
		GObject obj = retrieveGObject(menu);
		if (null != obj)
			tve.setMenu((Menu)obj);
		else
			tve.setMenu(new Menu(menu));
		fireTextViewEvent(tve);
	}

	private void handleSetScrollAdjustments(int hajd, int vadj) {
		fireTextViewEvent(new TextViewEvent(this, TextViewEvent.Type.SET_SCROLL_ADJUSTMENTS));
	}

	private void handleSetAnchor() {
		fireTextViewEvent(new TextViewEvent(this, TextViewEvent.Type.SET_ANCHOR));
	}

	private void handleToggleOverwrite() {
		fireTextViewEvent(new TextViewEvent(this, TextViewEvent.Type.TOGGLE_OVERWRITE));
	}

	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_text_view_get_type());
	}

	protected native static final void gtk_text_view_set_accepts_tab   (int view, boolean setting);
	protected native static final boolean gtk_text_view_get_accepts_tab (int textview);
	protected native static final void gtk_text_view_set_overwrite (int view, boolean setting );
	protected native static final boolean gtk_text_view_get_overwrite( int view );

	/****************************************
	 * BEGINNING OF GENERATED CODE
	 ****************************************/
	native static final protected int gtk_text_view_get_type();
	native static final protected int gtk_text_view_new();
	native static final protected int gtk_text_view_new_with_buffer(int buffer);
	native static final protected void gtk_text_view_set_buffer(int textView, int buffer);
	native static final protected int gtk_text_view_get_buffer(int textView);
	native static final protected boolean gtk_text_view_scroll_to_iter(
		int textView,
		int iter,
		double withinMargin,
		boolean useAlign,
		double xalign,
		double yalign);
	native static final protected void gtk_text_view_scroll_to_mark(
		int textView,
		int mark,
		double withinMargin,
		boolean useAlign,
		double xalign,
		double yalign);
	native static final protected void gtk_text_view_scroll_mark_onscreen(int textView, int mark);
	native static final protected boolean gtk_text_view_move_mark_onscreen(int textView, int mark);
	native static final protected boolean gtk_text_view_place_cursor_onscreen(int textView);
	native static final protected void gtk_text_view_get_visible_rect(int textView, int visibleRect);
	native static final protected void gtk_text_view_set_cursor_visible(int textView, boolean setting);
	native static final protected boolean gtk_text_view_get_cursor_visible(int textView);
	native static final protected void gtk_text_view_get_iter_location(int textView, int iter, int location);
	native static final protected void gtk_text_view_get_iter_at_location(int textView, int iter, int x, int y);
	native static final protected void gtk_text_view_get_line_yrange(int textView, int iter, int[] y, int[] height);
	native static final protected void gtk_text_view_get_line_at_y(int textView, int targetIter, int y, int[] lneTop);
	native static final protected void gtk_text_view_buffer_to_window_coords(
		int textView,
		int win,
		int bufferX,
		int bufferY,
		int[] windowX,
		int[] windowY);
	native static final protected void gtk_text_view_window_to_buffer_coords(
		int textView,
		int win,
		int windowX,
		int windowY,
		int[] bufferX,
		int[] bufferY);
	native static final protected int gtk_text_view_get_window(int textView, int win);
	native static final protected int gtk_text_view_get_window_type(int textView, int window);
	native static final protected void gtk_text_view_set_border_window_size(int textView, int type, int size);
	native static final protected int gtk_text_view_get_border_window_size(int textView, int type);
	native static final protected boolean gtk_text_view_forward_display_line(int textView, int iter);
	native static final protected boolean gtk_text_view_backward_display_line(int textView, int iter);
	native static final protected boolean gtk_text_view_forward_display_line_end(int textView, int iter);
	native static final protected boolean gtk_text_view_backward_display_line_start(int textView, int iter);
	native static final protected boolean gtk_text_view_starts_display_line(int textView, int iter);
	native static final protected boolean gtk_text_view_move_visually(int textView, int iter, int count);
	native static final protected void gtk_text_view_add_child_at_anchor(int textView, int child, int anchor);
	native static final protected void gtk_text_view_add_child_in_window(
		int textView,
		int child,
		int whichWindow,
		int xpos,
		int ypos);
	native static final protected void gtk_text_view_move_child(int textView, int child, int xpos, int ypos);
	native static final protected void gtk_text_view_set_wrap_mode(int textView, int wrapMode);
	native static final protected int gtk_text_view_get_wrap_mode(int textView);
	native static final protected void gtk_text_view_set_editable(int textView, boolean setting);
	native static final protected boolean gtk_text_view_get_editable(int textView);
	native static final protected void gtk_text_view_set_pixels_above_lines(int textView, int pixelsAboveLines);
	native static final protected int gtk_text_view_get_pixels_above_lines(int textView);
	native static final protected void gtk_text_view_set_pixels_below_lines(int textView, int pixelsBelowLines);
	native static final protected int gtk_text_view_get_pixels_below_lines(int textView);
	native static final protected void gtk_text_view_set_pixels_inside_wrap(int textView, int pixelsInsideWrap);
	native static final protected int gtk_text_view_get_pixels_inside_wrap(int textView);
	native static final protected void gtk_text_view_set_justification(int textview, int justification);
	native static final protected int gtk_text_view_get_justification(int textView);
	native static final protected void gtk_text_view_set_left_margin(int textView, int leftMargin);
	native static final protected int gtk_text_view_get_left_margin(int textView);
	native static final protected void gtk_text_view_set_right_margin(int textView, int rightMargin);
	native static final protected int gtk_text_view_get_right_margin(int textView);
	native static final protected void gtk_text_view_set_indent(int textView, int indent);
	native static final protected int gtk_text_view_get_indent(int textView);
	native static final protected void gtk_text_view_set_tabs(int textView, int tabs);
	native static final protected int gtk_text_view_get_tabs(int textView);
	native static final protected int gtk_text_view_get_default_attributes(int textView);
	/****************************************
	 * END OF GENERATED CODE
	 ****************************************/
}
