/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */


package org.gnu.gnomevte;

import java.io.File;
import java.io.FileNotFoundException;

import org.gnu.gdk.Color;
import org.gnu.gdk.Pixbuf;
import org.gnu.glib.Handle;
import org.gnu.pango.FontDescription;

/**
 * VteTerminal widget bindings implementation.
 */
public class Terminal extends org.gnu.gtk.Widget {
	
	private int pid_t;
	
	/**
	 * Used in the bindings development.
	 * You should not use this constructor
	 * Use <code>Terminal ()</code> or <code>Terminal (command, args, workingDir)</code> instead.
	 */
	protected Terminal (Handle handle){
		super (handle);
	}

	/**
	 * Creates a new Termina widget.
	 * This constructor launchs a shell inside de widget.
	 */
	public Terminal (){
		super (vte_terminal_new ());
	}

	/**
	 * Creates a new Terminal widget and executes <code>command</code> inside it.
	 *
	 * @param command Command launched inside the terminal.
	 * @param args arguments passed to the <code>command</code>.
	 * @param workingDir working directory passed to the process.
	 */
	public Terminal (String command, String args[], String workingDir){
		this();
		pid_t = forkCommand (command, args, workingDir, true, true, true);
	}

	/**
	 * Creates an new Terminal widget and launches a shell inside it.
	 *
	 * @return The widget with a shell in it.
	 */
	public static Terminal terminalAndShell (){
		Terminal t = new Terminal (null, null, null);
		return t;
	}
		
	/**
	 * Forks the <code>command</code> and show it inside the widget.
	 * 
	 * @param command program to launch in a new process.
	 * @param args The argument list available to the executed program.
	 * @param directory working directory available to the executed program.
	 * FIXME: investigate the meaning of laslog, utmp, wtmp. 
	 * @return Integer representing the process id.
	 */
	public int forkCommand (String command, String args[], String directory, boolean lastlog, boolean utmp, boolean wtmp){
		if(args == null)
			args = new String[]{};
		return vte_terminal_fork_command (getHandle(), command, args, directory, lastlog, utmp, wtmp);
	}

	/** 
	 * Send data to the terminal to display to handle in some way. 
	 * If data is null, it will be ignored.
	 * 
	 * @param data Sent to the terminal.
	 */
	public void feed (String data){
		if (data != null)
			vte_terminal_feed (getHandle(), data, data.length ());
	}
	
	/**	
	 * Send data to the terminal's forked command to handle in some way.
	 * If data is null, it will be ignored.
	 * 
	 * @param data Sent to the terminal.
	 */
	public void feedChild (String data){
		if (data != null){
			vte_terminal_feed_child (getHandle(), data, data.length ());
		}
	}

	/**
	 * Copy currently selected text to the clipboard.
	 */
	public void copyClipboard (){
		vte_terminal_copy_clipboard (getHandle());
	}

	/**
	 * Paste clipboard text from the clipboard to the terminal.
	 */
	public void pasteClipboard (){
		vte_terminal_paste_clipboard (getHandle());
	}

	/*
	 * TODO: Test!!!
	 */	 
	public void copyPrimary (){
		vte_terminal_copy_primary(getHandle());
	}

	/*
	 * TODO: Test!!!
	 */
	public void pastePrimary (){
		vte_terminal_paste_primary(getHandle());
	}

	/**
	 * Set the terminal's size.
	 *
	 * @param columns The terminal's width.
	 * @param rows The terminal's height.
	 */
	public void setSize (int columns, int rows){
		vte_terminal_set_size (getHandle(), columns, rows);
	}
	
	/**
	 * Set the terminal's audible bell.
	 *
	 * @param audible  If true, the terminal emits a Bip to attract users attention.
	 */
	public void setAudibleBell (boolean audible){
		vte_terminal_set_audible_bell (getHandle(), audible);
	}

	/**
	 *  Get the terminal's audible bell state.
	 *
	 *  @return The audible bell state.
	 */
	public boolean isBellAudible (){
		return vte_terminal_get_audible_bell (getHandle());
	}

	/**
	 * Set the terminal's visible bell state.
	 * 
	 * @param visible  If true, the terminal blinks to attract users attention.
	 */
	public void setVisibleBell (boolean visible){
		vte_terminal_set_visible_bell (getHandle(), visible);
	}

	/**
	 *  Get the terminal's audible bell state.
	 *
	 *  @return The visual bell state
	 */
	public boolean isBellVisible (){
		return vte_terminal_get_visible_bell (getHandle());
	}

	/**
	 * If true, scrolls the widget down following the output.
	 *
	 * @param scrollOnOutput If true, scroll.
	 */
	public void setScrollOnOutput (boolean scrollOnOutput){
		vte_terminal_set_scroll_on_output (getHandle(), scrollOnOutput);
	}

	/**
	 * If true, scrolls the widget down when pressing following the keyboard press.
	 *
	 * @param scrollOnKeystroke If true, scroll.
	 */
	public void setScrollOnKeystroke (boolean scrollOnKeystroke){
		vte_terminal_set_scroll_on_keystroke (getHandle(), scrollOnKeystroke);
	}

	/**
	 * Restores the default colors.
	 * 
	 */
	public void setDefaultColors() {
		vte_terminal_set_default_colors(getHandle());
	}
	 
	/**
	 * Set the <code>file</code> as the background image of the terminal.
	 *
	 * @param file File to set as background.
	 * 
	 * @throws FileNotFoundException if the image doesn't exist.
	 */
	public void setBackgroudImage(String file) throws FileNotFoundException{
		File f = new File(file);
		if (f.exists())
			vte_terminal_set_background_image_file(getHandle(), file);
		else
			throw new FileNotFoundException();
	}
		
	/**
	 * Sets the background image for the widget.  Text which would otherwise 
	 * be drawn using the default background color will instead be drawn 
	 * over the specified image. If necessary, the image will be tiled to 
	 * cover the widget's entire visible area.
	 * 
	 * @param image
	 */
	public void setBackgroundImage(Pixbuf image) {
		vte_terminal_set_background_image(getHandle(), image.getHandle());
	}
		
	/**
	 * Set whether or not the cursor blinks.
	 *
	 * @param blinks If true, blinks.
	 */
	public void setCursorBlinks(boolean blinks){
		vte_terminal_set_cursor_blinks(getHandle(), blinks);
	}
	
	/**
	 * Set the number of scrollback lines.
	 *
	 * @param lines The number of lines to save in the buffer.
	 */
	public void setScrollbackLines(int lines){
		vte_terminal_set_scrollback_lines(getHandle(), lines);
	}
		
	/**
	 * Sets the terminal's background saturation level.
	 *
	 * @param saturation The saturation level.
	 */
	public void setBackgroundSaturation (int saturation){
		vte_terminal_set_background_saturation (getHandle(), saturation);
	}

	/**
	 * Sets the terminal backgroud transparent or not.
	 *
	 * @param transparent Transparent if <code>true</code>.
	 */
	public void setBackgroundTransparent (boolean transparent){
		vte_terminal_set_background_transparent (getHandle(), transparent);
	}
	
	/**
	 * Sets the background color for text which does not have a specific 
	 * background color assigned. Only has effect when no background image 
	 * is set and when the terminal is not transparent.
	 * @param color
	 */
	public void setBackgroudColor(Color color) {
		vte_terminal_set_color_background(getHandle(), color.getHandle());
	}	

	/**
	 * Sets the foreground color used to draw normal text
	 * @param color
	 */
	public void setForegroundColor(Color color) {
		vte_terminal_set_color_foreground(getHandle(), color.getHandle());
	}

	/**
	 * The terminal widget uses a 28-color model comprised of the default 
	 * foreground and background colors, the bold foreground color, the 
	 * dim foreground color, an eight color palette, and bold versions of 
	 * the eight color palette, and a dim version of the the eight color 
	 * palette.
	 * 
	 * size must be either 0, 8, 16, or 24. If foreground is null and 
	 * size is greater than 0, the new foreground color is taken from 
	 * palette[7]. If background is null and size is greater than 0, 
	 * the new background color is taken from palette[0]. If size is 
	 * 8 or 16, the third (dim) and possibly second (bold) 8-color 
	 * palette is extrapolated from the new background color and the 
	 * items in palette.
	 * 
	 * @param fgcolor
	 * @param bgcolor
	 * @param palett
	 * @param size
	 */
	public void setColors(Color fgcolor, Color bgcolor, Color palett, int size) {
		vte_terminal_set_colors(getHandle(), fgcolor.getHandle(), 
							bgcolor.getHandle(), palett.getHandle(), size);
	}

	/**
	 * Sets the color used to draw bold text in the default foreground color.
	 * @param color
	 */
	public void setColorBold(Color color) {
		vte_terminal_set_color_bold(getHandle(), color.getHandle());
	}
	
	/**
	 * Sets the color used to draw dim text in the default foreground color.
	 * 
	 * @param color
	 */
	public void setColorDim(Color color) {
		vte_terminal_set_color_dim(getHandle(), color.getHandle());
	}

	/**
	 * Sets the font used for rendering all text displayed by the terminal. 
	 * The terminal will immediately attempt to load the desired font, 
	 * retrieve its metrics, and attempts to resize itself to keep the same 
	 * number of rows and columns.
	 */
	public void setFont(FontDescription font) {
		vte_terminal_set_font(getHandle(), font.getHandle());
	}

	/**
	 * Queries the terminal for information about the fonts which will be 
	 * used to draw text in the terminal.
	 * @return FontDescription
	 */
	public FontDescription getFont() {
		return new FontDescription(vte_terminal_get_font(getHandle()));
	}

	/**
	 * Controls whether or not the terminal will attempt to draw bold 
	 * text by repainting text with a different offset.
	 * @param allowBold
	 */
	public void setAllowBold(boolean allowBold) {
		vte_terminal_set_allow_bold(getHandle(), allowBold);
	}
	
	/**
	 * Checks whether or not the terminal will attempt to draw bold text 
	 * by repainting text with a one-pixel offset.
	 * @return boolean
	 */
	public boolean getAllowBold() {
		return vte_terminal_get_allow_bold(getHandle());
	}
	
	/**
	 * Resets as much of the terminal's internal state as possible, 
	 * discarding any unprocessed input data, resetting character 
	 * attributes, cursor state, national character set state, status 
	 * line, terminal modes (insert/delete), selection state, and encoding.
	 * 
	 * @param full true to reset tabstops
	 * @param clearHistory true to empty the terminal's scrollback buffer
	 */
	public void reset(boolean full, boolean clearHistory) {
		vte_terminal_reset(getHandle(), full, clearHistory);
	}
	
	/**
	 * Sets what type of terminal the widget attempts to emulate by scanning 
	 * for control sequences defined in the system's termcap file. Unless 
	 * you are interested in this feature, always use "xterm".
	 * @param emulation
	 */
	public void setEmulation(String emulation) {
		vte_terminal_set_emulation(getHandle(), emulation);
	}
	
	/**
	 * Queries the terminal for its current emulation, as last set by a 
	 * call to setEmulation().
	 * @return String
	 */
	public String getEmulation() {
		return vte_terminal_get_emulation(getHandle());
	}
	
	/**
	 * Changes the encoding the terminal will expect data from the child 
	 * to be encoded with. For certain terminal types, applications 
	 * executing in the terminal can change the encoding. The default 
	 * encoding is defined by the application's locale settings.
	 * @param codeset
	 */
	public void setEncoding(String codeset) {
		vte_terminal_set_encoding(getHandle(), codeset);
	}
	
	/**
	 * Determines the name of the encoding in which the terminal expects 
	 * data to be encoded.
	 * @return String
	 */
	public String getEncoding() {
		return vte_terminal_get_encoding(getHandle());
	}
	
	static {
		System.loadLibrary ("vtejni-0.11");
	}

	/*-------- NATIVE METHODS -----------*/
	static native Handle vte_terminal_new();
	static native int vte_terminal_fork_command(Handle handle,
				String command, String[] argv,
				String directory,
				boolean lastlog,
				boolean utmp,
				boolean wtmp);

	/* Send data to the terminal to display, or to the terminal's forked command
	to handle in some way.  If it's 'cat', they should be the same. */
	static native void vte_terminal_feed(Handle handle, String data, int length);
	static native void vte_terminal_feed_child (Handle handle,
			     String data, long length);

	/* Copy currently-selected text to the clipboard, or from the clipboard to
 	* the terminal. */
	static native void vte_terminal_copy_clipboard (Handle handle);
	static native void vte_terminal_paste_clipboard (Handle handle);
	static native void vte_terminal_copy_primary(Handle handle);
	static native void vte_terminal_paste_primary(Handle handle);

	/* Set the terminal's size. */
	static native void vte_terminal_set_size(Handle handle, int columns, int rows);

	/* Set various one-off settings. */
	static native void vte_terminal_set_audible_bell(Handle handle, boolean is_audible);
	static native boolean vte_terminal_get_audible_bell(Handle handle);
	static native void vte_terminal_set_visible_bell(Handle handle, boolean is_visible);
	static native boolean vte_terminal_get_visible_bell(Handle handle);
	static native void vte_terminal_set_scroll_on_output(Handle handle, boolean scroll);
	static native void vte_terminal_set_scroll_on_keystroke(Handle handle, boolean scroll);

	/* Set the color scheme. */
	static native void vte_terminal_set_color_dim(Handle handle, Handle gdkColor);
	static native void vte_terminal_set_color_bold(Handle handle, Handle gdkColor);
	static native void vte_terminal_set_color_foreground(Handle handle, Handle gdkColor);
	static native void vte_terminal_set_color_background(Handle handle, Handle gdkColor);
	
	// foreground, background and palette are gdk.Color
	static native void vte_terminal_set_colors(Handle handle, Handle foreground, Handle background, Handle palette, int palette_size);
	static native void vte_terminal_set_default_colors(Handle handle);

	/* Background effects. */

	//image is a GdkPixbuf
	static native void vte_terminal_set_background_image(Handle handle, Handle image);
	static native void vte_terminal_set_background_image_file(Handle handle, String path);
	static native void vte_terminal_set_background_saturation(Handle handle, double saturation);
	static native void vte_terminal_set_background_transparent(Handle handle, boolean transparent);

	/* Set whether or not the cursor blinks. */
	static native void vte_terminal_set_cursor_blinks(Handle handle, boolean blink);

	/* Set the number of scrollback lines, above or at an internal minimum. */
	static native void vte_terminal_set_scrollback_lines(Handle handle, int lines);

	/* Append the input method menu items to a given shell. */
	// menushell: GtkMenuShell
//	static native void vte_terminal_im_append_menuitems(Handle handle, Handle menushell);

	/* Set or retrieve the current font. */
	// font_desc: PangoFontDescription
	static native void vte_terminal_set_font(Handle handle, Handle font_desc);
//	static native void vte_terminal_set_font_from_string(Handle handle, byte[] name);
	
	//return: PangoFontDescription.
	static native Handle vte_terminal_get_font(Handle handle);
//	static native boolean vte_terminal_get_using_xft(Handle handle);
	static native void vte_terminal_set_allow_bold(Handle handle, boolean allow_bold);
	static native boolean vte_terminal_get_allow_bold(Handle handle);

	/* Check if the terminal is the current selection owner. */
//	static native boolean vte_terminal_get_has_selection(Handle handle);

	/* Set the list of word chars, optionally using hyphens to specify ranges
 	* (to get a hyphen, place it first), and check if a character is in the
 	* range. */
//	static native void vte_terminal_set_word_chars(Handle handle, byte[] spec);
//	static native boolean vte_terminal_is_word_char(Handle handle, char c);

	/* Set what happens when the user strikes backspace or delete. */
	// binding VteTerminalEraseBinding
//	static native void vte_terminal_set_backspace_binding(Handle handle, int binding);
	// binding VteTerminalEraseBinding
//	static native void vte_terminal_set_delete_binding(Handle handle, int binding);

	/* Manipulate the autohide setting. */
//	static native void vte_terminal_set_mouse_autohide(Handle handle, boolean setting);
//	static native boolean vte_terminal_get_mouse_autohide(Handle handle);

	/* Reset the terminal, optionally clearing the tab stops and line history. */
	static native void vte_terminal_reset(Handle handle, boolean full, boolean clear_history);

	/* Read the contents of the terminal, using a callback function to determine
 	* if a particular location on the screen (0-based) is interesting enough to
 	* include.  Each byte in the returned string will have a corresponding
 	* struct vte_char_attributes in the passed GArray, if the array was not NULL.
 	* Note that it will have one entry per byte, not per character, so indexes
 	* should match up exactly. */
	/*static byte[] vte_terminal_get_text(Handle handle,
			    boolean(*is_selected)(Handle handle,
						   int column,
						   int row,
						   gpointer data),
			    gpointer data,
			    GArray *attributes);
	static byte[] vte_terminal_get_text_range(Handle handle,
				  int start_row, int start_col,
				  int end_row, int end_col,
				  boolean(*is_selected)(Handle handle,
							 int column,
							 int row,
							 gpointer data),
				  gpointer data,
				  GArray *attributes);*/
//	static native void vte_terminal_get_cursor_position(Handle handle, int[] column, int[] row);

	/* Display string matching:  clear all matching expressions. */
//	static native void vte_terminal_match_clear_all(Handle handle);

	/* Add a matching expression, returning the tag the widget assigns to that
	 * expression. */
//	static native int vte_terminal_match_add(Handle handle, byte[] match);
	/* Remove a matching expression by tag. */
//	static native void vte_terminal_match_remove(Handle handle, int tag);

	/* Check if a given cell on the screen contains part of a matched string.  If
 	* it does, return the string, and store the match tag in the optional tag
 	* argument. */
//	static native String vte_terminal_match_check(Handle handle, int column, int row, int[] tag);

	/* Set the emulation type.  Most of the time you won't need this. */
	static native void vte_terminal_set_emulation(Handle handle, String emulation);
	static native String vte_terminal_get_emulation(Handle handle);

	/* Set the character encoding.  Most of the time you won't need this. */
	static native void vte_terminal_set_encoding(Handle handle, String codeset);
	static native String vte_terminal_get_encoding(Handle handle);

	/* Get the contents of the status line. */
//	static native String vte_terminal_get_status_line(Handle handle);

	/* Get the padding the widget is using. */
//	static native void vte_terminal_get_padding(Handle handle, int[] xpad, int[] ypad);


	/* Accessors for bindings. */
	//return: GtkAdjustment
//	static native Handle vte_terminal_get_adjustment(Handle handle);
//	static native long vte_terminal_get_char_width(Handle handle);
//	static native long vte_terminal_get_char_height(Handle handle);
//	static native long vte_terminal_get_char_descent(Handle handle);
//	static native long vte_terminal_get_char_ascent(Handle handle);
//	static native long vte_terminal_get_row_count(Handle handle);
//	static native long vte_terminal_get_column_count(Handle handle);
//	static native String vte_terminal_get_window_title(Handle handle);
//	static native String vte_terminal_get_icon_title(Handle handle);
}
