/* control.c -- create and handle the control panel			*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include "xdatplot.h"

#ifdef	__STDC__
static	Widget	make_zoom(Widget);
static	Widget	make_button(String, Widget, XtPointer, XtCallbackProc);
static	Widget	make_toggle(String, Widget, XtCallbackProc);
static	void	t_zoom_field_cb(Widget, XtPointer, XmAnyCallbackStruct *);
static	void	t_zoom_in_cb(Widget, double *, XmPushButtonCallbackStruct *);
static	void	t_zoom_out_cb(Widget, double *, XmPushButtonCallbackStruct *);
static	void	t_full_scale_cb(Widget, XtPointer,
				XmPushButtonCallbackStruct *);
static	void	v_zoom_field_cb(Widget, XtPointer, XmAnyCallbackStruct *);
static	void	v_zoom_in_cb(Widget, double *, XmPushButtonCallbackStruct *);
static	void	v_zoom_out_cb(Widget, double *, XmPushButtonCallbackStruct *);
static	void	v_full_scale_cb(Widget, XtPointer,
				XmPushButtonCallbackStruct *);
static	void	autoscale_cb(Widget, XtPointer, XmPushButtonCallbackStruct *);
static	void	auto_always_cb(Widget, XtPointer,
			       XmToggleButtonCallbackStruct *);
static	Widget	make_cursor(Widget);
static	Widget	make_comment(Widget);
static	void	t_field_cb(Widget, XtPointer, XmAnyCallbackStruct *);
#else	/* __STDC__ */
static	Widget	make_zoom();
static	Widget	make_button();
static	Widget	make_toggle();
static	void	t_zoom_field_cb();
static	void	t_zoom_in_cb();
static	void	t_zoom_out_cb();
static	void	t_full_scale_cb();
static	void	v_zoom_field_cb();
static	void	v_zoom_in_cb();
static	void	v_zoom_out_cb();
static	void	v_full_scale_cb();
static	void	autoscale_cb();
static	void	auto_always_cb();
static	Widget	make_cursor();
static	Widget	make_comment();
static	void	t_field_cb();
#endif	/* __STDC__ */

static	Widget	t_zoom_field = NO_WIDGET;
static	Widget	t_zoom_in_button = NO_WIDGET;
static	Widget	t_zoom_way_in_button = NO_WIDGET;
static	Widget	t_zoom_out_button = NO_WIDGET;
static	Widget	t_full_scale_button = NO_WIDGET;
static	Widget	v_zoom_field = NO_WIDGET;
static	Widget	v_zoom_in_button = NO_WIDGET;
static	Widget	v_zoom_way_in_button = NO_WIDGET;
static	Widget	v_zoom_out_button = NO_WIDGET;
static	Widget	v_full_scale_button = NO_WIDGET;
static	Widget	autoscale_button = NO_WIDGET;
static	Widget	auto_always_toggle = NO_WIDGET;
static	Widget	comment_text = NO_WIDGET;

Widget
create_control(main_w)
Widget		main_w;
{
    Widget		control;
    Widget		zoom_form;
    Widget		cursor_form;
    Widget		comment_form;
    Dimension		wf, hf, w, h;

    control = XtVaCreateWidget("control",
	xmFormWidgetClass, main_w,
	XmNresizable, True,
    NULL);
    zoom_form = make_zoom(control);
    XtVaSetValues(zoom_form,
	XmNtopAttachment, XmATTACH_FORM,
	XmNleftAttachment, XmATTACH_FORM,
	XmNbottomAttachment, XmATTACH_FORM,
    NULL);
    cursor_form = make_cursor(control);
    XtVaSetValues(cursor_form,
	XmNtopAttachment, XmATTACH_FORM,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, zoom_form,
	XmNbottomAttachment, XmATTACH_FORM,
    NULL);
    comment_form = make_comment(control);
    XtVaSetValues(comment_form,
	XmNtopAttachment, XmATTACH_FORM,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, cursor_form,
	XmNbottomAttachment, XmATTACH_FORM,
    NULL);
    GetSize(zoom_form, wf, hf);
    GetSize(cursor_form, w, h);
    wf += w;
    hf = max(hf, h);
    GetSize(comment_form, w, h);
    wf += w;
    hf = max(hf, h);
    SetSize(control, wf, hf);
    if (!app_data.no_control_panel) XtManageChild(control);
    return(control);
}

void
set_auto_always(val)
Bool		val;
{
    XtVaSetValues(auto_always_toggle, XmNset, val, NULL);
}

void
t_is_full_scale(val)
Bool		val;
{
    XtSetSensitive(t_zoom_out_button, !val);
    XtSetSensitive(t_full_scale_button, !val);
}

void
v_is_full_scale(val)
Bool		val;
{
    XtSetSensitive(v_zoom_out_button, !val);
    XtSetSensitive(v_full_scale_button, !val);
}

static Widget
make_zoom(control)
Widget		control;
{
    static double	zoom_fac = ZOOMFAC; /* passed as callback data	*/
    static double	big_zoom_fac = BIGZOOMFAC;

    Widget		zoom_form;
    Widget		t_zoom_frame;
    Widget		t_zoom_form;
    Widget		t_zoom_label;
    Widget		v_zoom_frame;
    Widget		v_zoom_form;
    Widget		v_zoom_label;
    Dimension		w1, h1, w2, h2;
    Dimension		wf, hf;

    zoom_form = XtVaCreateWidget("zoomForm",
	xmFormWidgetClass, control,
    NULL);
    /*
     * Time scale zoom subpanel:
     */
    t_zoom_frame = XtVaCreateManagedWidget("tZoomFrame",
	xmFrameWidgetClass, zoom_form,
	XmNshadowType, XmSHADOW_ETCHED_IN,
	XmNtopAttachment, XmATTACH_FORM,
	XmNleftAttachment, XmATTACH_FORM,
	XmNbottomAttachment, XmATTACH_FORM,
    NULL);
    t_zoom_form = XtVaCreateWidget("tZoomForm",
	xmFormWidgetClass, t_zoom_frame,
    NULL);
    t_zoom_field = XtVaCreateManagedWidget("tZoomField",
	xmTextFieldWidgetClass, t_zoom_form,
	XmNleftAttachment, XmATTACH_FORM,
	XmNbottomAttachment, XmATTACH_FORM,
    NULL);
    AddCallback(t_zoom_field, XmNactivateCallback, t_zoom_field_cb, NULL);
    t_zoom_label = XtVaCreateManagedWidget("tZoomLabel",
	xmLabelWidgetClass, t_zoom_form,
	XmNalignment, XmALIGNMENT_BEGINNING,
	XmNtopAttachment, XmATTACH_FORM,
	XmNbottomAttachment, XmATTACH_WIDGET,
	XmNbottomWidget, t_zoom_field,
	XmNleftAttachment, XmATTACH_FORM,
    NULL);
    t_zoom_out_button = make_button("tZoomOutButton", t_zoom_form,
				    (XtPointer) &zoom_fac,
				    (XtCallbackProc) t_zoom_out_cb);
    XtVaSetValues(t_zoom_out_button,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, t_zoom_field,
	XmNbottomAttachment, XmATTACH_FORM,
    NULL);
    t_zoom_in_button = make_button("tZoomInButton", t_zoom_form,
				   (XtPointer) &zoom_fac,
				   (XtCallbackProc) t_zoom_in_cb);
    XtVaSetValues(t_zoom_in_button,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, t_zoom_field,
	XmNbottomAttachment, XmATTACH_WIDGET,
	XmNbottomWidget, t_zoom_out_button,
	XmNrightAttachment, XmATTACH_OPPOSITE_WIDGET,
	XmNrightWidget, t_zoom_out_button,
    NULL);
    t_full_scale_button = make_button("tFullScale", t_zoom_form,
				      NULL, (XtCallbackProc) t_full_scale_cb);
    XtVaSetValues(t_full_scale_button,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, t_zoom_out_button,
	XmNbottomAttachment, XmATTACH_FORM,
	XmNrightAttachment, XmATTACH_FORM,
    NULL);
    t_zoom_way_in_button = make_button("tZoomWayInButton", t_zoom_form,
				       (XtPointer) &big_zoom_fac,
				       (XtCallbackProc) t_zoom_in_cb);
    XtVaSetValues(t_zoom_way_in_button,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, t_zoom_out_button,
	XmNbottomAttachment, XmATTACH_WIDGET,
	XmNbottomWidget, t_full_scale_button,
	XmNrightAttachment, XmATTACH_FORM,
    NULL);
    GetSize(t_zoom_field, w1, h1);
    GetSize(t_zoom_label, w2, h2);
    wf = max(w1, w2);
    hf = h1 + h2;
    GetSize(t_full_scale_button, w1, h1);
    GetSize(t_zoom_way_in_button, w2, h2);
    wf += max(w1, w2);
    hf = max(hf, h1 + h2);
    GetSize(t_zoom_out_button, w1, h1);
    GetSize(t_zoom_in_button, w2, h2);
    wf += max(w1, w2);
    hf = max(hf, h1 + h2);
    SetSize(t_zoom_form, wf, hf);
    XtManageChild(t_zoom_form);
    XtVaGetValues(t_zoom_frame,
	XmNmarginWidth, &w1,
	XmNmarginHeight, &h1,
	XmNshadowThickness, &h2,
    NULL);
    SetSize(t_zoom_frame, wf + 2*w1 + 2*h2, hf + 2*h1 + 2*h2);
    /*
     * Voltage scale zoom subpanel:
     */
    v_zoom_frame = XtVaCreateManagedWidget("vZoomFrame",
	xmFrameWidgetClass, zoom_form,
	XmNshadowType, XmSHADOW_ETCHED_IN,
	XmNtopAttachment, XmATTACH_FORM,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, t_zoom_frame,
	XmNbottomAttachment, XmATTACH_FORM,
    NULL);
    v_zoom_form = XtVaCreateWidget("vZoomForm",
	xmFormWidgetClass, v_zoom_frame,
    NULL);
    v_zoom_field = XtVaCreateManagedWidget("vZoomField",
	xmTextFieldWidgetClass, v_zoom_form,
	XmNleftAttachment, XmATTACH_FORM,
	XmNbottomAttachment, XmATTACH_FORM,
    NULL);
    AddCallback(v_zoom_field, XmNactivateCallback, v_zoom_field_cb, NULL);
    v_zoom_label = XtVaCreateManagedWidget("vZoomLabel",
	xmLabelWidgetClass, v_zoom_form,
	XmNalignment, XmALIGNMENT_BEGINNING,
	XmNtopAttachment, XmATTACH_FORM,
	XmNbottomAttachment, XmATTACH_WIDGET,
	XmNbottomWidget, v_zoom_field,
	XmNleftAttachment, XmATTACH_FORM,
    NULL);
    v_zoom_out_button = make_button("vZoomOutButton", v_zoom_form,
				    (XtPointer) &zoom_fac,
				    (XtCallbackProc) v_zoom_out_cb);
    XtVaSetValues(v_zoom_out_button,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, v_zoom_field,
	XmNbottomAttachment, XmATTACH_FORM,
    NULL);
    v_zoom_in_button = make_button("tZoomInButton", v_zoom_form,
				   (XtPointer) &zoom_fac,
				   (XtCallbackProc) v_zoom_in_cb);
    XtVaSetValues(v_zoom_in_button,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, v_zoom_field,
	XmNbottomAttachment, XmATTACH_WIDGET,
	XmNbottomWidget, v_zoom_out_button,
	XmNrightAttachment, XmATTACH_OPPOSITE_WIDGET,
	XmNrightWidget, v_zoom_out_button,
    NULL);
    v_full_scale_button = make_button("tFullScale", v_zoom_form,
				      NULL,
				      (XtCallbackProc) v_full_scale_cb);
    XtVaSetValues(v_full_scale_button,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, v_zoom_out_button,
	XmNbottomAttachment, XmATTACH_FORM,
    NULL);
    v_zoom_way_in_button = make_button("tZoomWayInButton", v_zoom_form,
				       (XtPointer) &big_zoom_fac,
				       (XtCallbackProc) v_zoom_in_cb);
    XtVaSetValues(v_zoom_way_in_button,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, v_zoom_out_button,
	XmNbottomAttachment, XmATTACH_WIDGET,
	XmNbottomWidget, v_full_scale_button,
    NULL);
    autoscale_button = make_button("autoscaleButton", v_zoom_form,
				   NULL, (XtCallbackProc) autoscale_cb);
    XtVaSetValues(autoscale_button,
	XmNbottomAttachment, XmATTACH_FORM,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, v_full_scale_button,
	XmNrightAttachment, XmATTACH_FORM,
    NULL);
    auto_always_toggle = make_toggle("autoscaleAlwaysToggle", v_zoom_form,
				     (XtCallbackProc) auto_always_cb);
    XtVaSetValues(auto_always_toggle,
	XmNbottomAttachment, XmATTACH_WIDGET,
	XmNbottomWidget, autoscale_button,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, v_full_scale_button,
	XmNrightAttachment, XmATTACH_FORM,
    NULL);
    set_auto_always(AUTOSCALE_ALWAYS);
    GetSize(v_zoom_field, w1, h1);
    GetSize(v_zoom_label, w2, h2);
    wf = max(w1, w2);
    hf = h1 + h2;
    GetSize(v_zoom_out_button, w1, h1);
    GetSize(v_zoom_in_button, w2, h2);
    wf += max(w1, w2);
    hf = max(hf, h1 + h2);
    GetSize(v_full_scale_button, w1, h1);
    GetSize(v_zoom_way_in_button, w2, h2);
    wf += max(w1, w2);
    hf = max(hf, h1 + h2);
    GetSize(autoscale_button, w1, h1);
    GetSize(auto_always_toggle, w2, h2);
    wf += max(w1, w2);
    hf = max(hf, h1 + h2);
    SetSize(v_zoom_form, wf, hf);
    XtManageChild(v_zoom_form);
    XtVaGetValues(v_zoom_frame,
	XmNmarginWidth, &w1,
	XmNmarginHeight, &h1,
	XmNshadowThickness, &h2,
    NULL);
    SetSize(v_zoom_frame, wf + 2*w1 + 2*h2, hf + 2*h1 + 2*h2);
    /*
     * finish up
     */
    GetSize(t_zoom_frame, w1, h1);
    GetSize(v_zoom_frame, w2, h2);
    wf = w1 + w2;
    hf = max(h1, h2);
    SetSize(zoom_form, wf, hf);
    XtManageChild(zoom_form);
    return(zoom_form);
}

static Widget
make_button(name, parent, data, callback)
String		name;
Widget		parent;
XtPointer	data;
XtCallbackProc	callback;
{
    Widget		button;

    button = XtVaCreateManagedWidget(name,
        xmPushButtonWidgetClass, parent,
    NULL);
    if (NULL != callback) {
	AddCallback(button, XmNactivateCallback, callback, data);
    }
    return(button);
}

static Widget
make_toggle(name, parent, callback)
String		name;
Widget		parent;
XtCallbackProc	callback;
{
    Widget		toggle;

    toggle = XtVaCreateManagedWidget(name,
        xmToggleButtonWidgetClass, parent,
	XmNindicatorType, XmN_OF_MANY,
    NULL);
    if (NULL != callback) {
	AddCallback(toggle, XmNvalueChangedCallback, callback, NULL);
    }
    return(toggle);
}

static void
t_zoom_field_cb(field, data, cbs)
Widget			field;
XtPointer		data;
XmAnyCallbackStruct	*cbs;
{
    String		s = XmTextFieldGetString(field);
    String		t;
    int			c;
    double		zoomfac;

    dprintf("t_zoom_field_cb\n");
    zoomfac = strtod(s, &t);
    while(isspace(*t)) t++;
    c = *t;
    Free(s);
    if ('\0' != c) Bellret;
    if (0.0 >= zoomfac) Bellret;
    zoomfac /= t_to_Dim_sf;
    if (zoomfac > 1.0) t_zoom_in(zoomfac);
    else if (zoomfac < 1.0) t_zoom_out(1.0/zoomfac);
    else {}
}

static void
t_zoom_in_cb(widget, zoom, cbs)
Widget				widget;
double				*zoom;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("t_zoom_in_cb\n");
    t_zoom_in(*zoom);
}

void
t_zoom_in(zoom)
double		zoom;
{
    double		tc;		/* time to center		*/
    TIME		otl = app_data.tl;
    TIME		otr = app_data.tr;
    double		w;

    if (2 >= TR - app_data.tl) return;
    if (IS_CURSOR && (CURSOR >= app_data.tl) && (CURSOR < TR)) tc = CURSOR;
    else { tc = (otl + otr); tc /= 2.0; }
    w = (otr - otl); w /= 2.0; w/= zoom;
    app_data.tl = floor(tc - w + 0.5);
    app_data.tlfrac = 0.0;
    TR = floor(app_data.tl + 2.0*w + 0.5);
    TR = max(TR, app_data.tl+2);
    FORCE_RANGE(app_data.tl, TR, TMIN, TMAX);
    if ((otl == app_data.tl) && (otr == app_data.tr)) return;
    if (AUTOSCALE_ALWAYS) set_auto_scale();
    adjust_sb();
    replot_whole();
}

static void
t_zoom_out_cb(widget, zoom, cbs)
Widget				widget;
double				*zoom;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("t_zoom_out_cb\n");
    t_zoom_out(*zoom);
}

void
t_zoom_out(zoom)
double		zoom;
{
    TIME		ow;		/* old width			*/
    TIME		otl = app_data.tl;
    TIME		otr = app_data.tr;
    double		w;
    double		tc;		/* time to center		*/

    ow = otr - otl;
    if (IS_CURSOR && (CURSOR >= TL) && (CURSOR < TR)) tc = CURSOR;
    else { tc = (otl + otr); tc /= 2.0; }
    w = ow; w /= 2.0; w *= zoom;
    app_data.tl = floor(tc - w + 0.5);
    app_data.tlfrac = 0.0;
    TR = floor(app_data.tl + 2.0*w + 0.5);
    if (TR - app_data.tl <= ow) TR = app_data.tl + ow + 1;
    FORCE_RANGE(app_data.tl, TR, TMIN, TMAX);
    if (TR - app_data.tl >= TMAX - TMIN) {
	t_full_scale();
	return;
    }
    if (AUTOSCALE_ALWAYS) set_auto_scale();
    adjust_sb();
    replot_whole();
}

static void
t_full_scale_cb(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("t_full_scale_cb\n");
    t_full_scale();
}

void
t_full_scale()
{
    app_data.tl = TMIN;
    app_data.tlfrac = 0.0;
    TR = TMAX;
    if (AUTOSCALE_ALWAYS) set_auto_scale();
    adjust_sb();
    replot_whole();
}

static void
v_zoom_field_cb(field, data, cbs)
Widget			field;
XtPointer		data;
XmAnyCallbackStruct	*cbs;
{
    String		s = XmTextFieldGetString(field);
    String		t;
    int			c;
    double		zoomfac;

    dprintf("v_zoom_field_cb\n");
    zoomfac = strtod(s, &t);
    while(isspace(*t)) t++;
    c = *t;
    Free(s);
    if ('\0' != c) Bellret;
    if (0.0 >= zoomfac) Bellret;
    zoomfac /= v_to_Dim_sf;
    if (zoomfac > 1.0) v_zoom_in(zoomfac);
    else if (zoomfac < 1.0) v_zoom_out(1.0/zoomfac);
    else {}
}

static void
v_zoom_in_cb(widget, zoom, cbs)
Widget				widget;
double				*zoom;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("v_zoom_in_cb\n");
    v_zoom_in(*zoom);
}

void
v_zoom_in(zoom)
double		zoom;
{
    FILTER_VAL		vc = (VB + VT)/2;

    if (2 >= VT - VB) return;
    VB = vc - (vc - VB) / zoom;
    VT = vc + (VT - vc) / zoom;
    VT = max(VT, VB+2);
    adjust_sb();
    replot_whole();
}

static void
v_zoom_out_cb(widget, zoom, cbs)
Widget				widget;
double				*zoom;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("v_zoom_out_cb\n");
    v_zoom_out(*zoom);
}

void
v_zoom_out(zoom)
double		zoom;
{
    FILTER_VAL		ow = VT - VB;
    FILTER_VAL		vc = (VB + VT)/2;

    VB = vc - (vc - VB) * zoom;
    VT = vc + (VT - vc) * zoom;
    if (VT - VB <= ow) VT = VB + ow + 1;
    FORCE_RANGE(VB, VT, VMIN, VMAX);
    if (VT - VB >= VMAX - VMIN) {
	v_full_scale();
	return;
    }
    adjust_sb();
    replot_whole();
}

static void
v_full_scale_cb(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("v_full_scale_cb\n");
    v_full_scale();
}

void
v_full_scale()
{
    VB = VMIN;
    VT = VMAX;
    adjust_sb();
    replot_whole();
}

void
set_zoom_fields()
{
    char		lbuf[LLEN];

    sprintf(lbuf, "%.15g", (double) t_to_Dim_sf);
    XtVaSetValues(t_zoom_field, XmNvalue, lbuf, NULL);
    sprintf(lbuf, "%.15g", (double) v_to_Dim_sf);
    XtVaSetValues(v_zoom_field, XmNvalue, lbuf, NULL);
}

static void
autoscale_cb(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmPushButtonCallbackStruct	*cbs;
{
    dprintf("autoscale_cb\n");
    if (set_auto_scale()) replot_whole();
}

static void
auto_always_cb(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmToggleButtonCallbackStruct	*cbs;
{
    dprintf("auto_always_cb\n");
    AUTOSCALE_ALWAYS = cbs->set;
}

static Widget
make_cursor(control)
Widget		control;
{
    XmString		xms;
    Widget		cursor_form;
    Widget		cursor_frame;
    Widget		t_label;
    Widget		v_label;
    Dimension		w1, h1, w2, h2, wf, hf;

    cursor_frame = XtVaCreateManagedWidget("cursorFrame",
	xmFrameWidgetClass, control,
	XmNshadowType, XmSHADOW_ETCHED_IN,
    NULL);
    cursor_form = XtVaCreateWidget("cursorForm",
	xmFormWidgetClass, cursor_frame,
    NULL);
    t_field = XtVaCreateManagedWidget("tField",
	xmTextFieldWidgetClass, cursor_form,
	XmNrightAttachment, XmATTACH_FORM,
    NULL);
    AddCallback(t_field, XmNactivateCallback, t_field_cb, NULL);
    t_label = XtVaCreateManagedWidget("tLabel",
	xmLabelWidgetClass, cursor_form,
	XmNalignment, XmALIGNMENT_BEGINNING,
	XmNleftAttachment, XmATTACH_FORM,
	XmNrightAttachment, XmATTACH_WIDGET,
	XmNrightWidget, t_field,
	XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET,
	XmNtopWidget, t_field,
	XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
	XmNbottomWidget, t_field,
    NULL);
    v_field = XtVaCreateManagedWidget("vField",
	xmTextFieldWidgetClass, cursor_form,
	XmNtopAttachment, XmATTACH_WIDGET,
	XmNtopWidget, t_field,
	XmNrightAttachment, XmATTACH_FORM,
    NULL);
    v_label = XtVaCreateManagedWidget("vLabel",
	xmLabelWidgetClass, cursor_form,
	XmNalignment, XmALIGNMENT_BEGINNING,
	XmNleftAttachment, XmATTACH_FORM,
	XmNrightAttachment, XmATTACH_WIDGET,
	XmNrightWidget, v_field,
	XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET,
	XmNtopWidget, v_field,
	XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
	XmNbottomWidget, v_field,
    NULL);
    GetSize(t_label, w1, h1);
    GetSize(v_label, w2, h2);
    wf = max(w1, w2);
    hf = h1 + h2;
    GetSize(t_field, w1, h1);
    GetSize(v_field, w2, h2);
    wf += max(w1, w2);
    hf = max(hf, h1 + h2);
    SetSize(cursor_form, wf, hf);
    XtManageChild(cursor_form);
    XtVaGetValues(cursor_frame,
	XmNmarginWidth, &w1,
	XmNmarginHeight, &h1,
	XmNshadowThickness, &h2,
    NULL);
    SetSize(cursor_frame, wf + 2*w1 + 2*h2, hf + 2*h1 + 2*h2);
    return(cursor_frame);
}

static	void
t_field_cb(widget, data, cbs)
Widget			widget;
XtPointer		data;
XmAnyCallbackStruct	*cbs;
{
    double		dval;
    String		val;
    String		s;
    TIME		t;

    XtVaGetValues(t_field, XmNvalue, &val, NULL);
    dval = strtod(val, &s);
    while(isspace(*s)) s++;
    if ('\0' != *s) {
	PU_error("Incorrect format for t value", "mainw.html#CONTROL");
	return;
    }
    t = Map_t_to_TIME(dval);
    t = max(TMIN, min(TMAX-1, t));
    set_cursor(t);
}

static Widget
make_comment(control)
Widget		control;
{
    Widget		comment_frame;
    Dimension		w1, h1, w2, h2, wf, hf;

    comment_frame = XtVaCreateManagedWidget("commentFrame",
	xmFrameWidgetClass, control,
	XmNshadowType, XmSHADOW_ETCHED_IN,
    NULL);
    comment_text = XmCreateScrolledText(comment_frame, "commentText", NULL, 0);
    XtManageChild(comment_text);
    GetSize(comment_text, wf, hf);
    XtVaGetValues(comment_frame,
	XmNmarginWidth, &w1,
	XmNmarginHeight, &h1,
	XmNshadowThickness, &h2,
    NULL);
    SetSize(comment_frame, wf + 2*w1 + 2*h2, hf + 2*h1 + 2*h2);
    return(comment_frame);
}

void
set_comment(s)
String		s;
{
    if (NO_WIDGET != comment_text)
	XmTextSetString(comment_text, s);
}

/* returned string must be freed by caller with XtFree			*/
String
get_comment()
{
    return(
	(NO_WIDGET == comment_text) ?
	"" :
	XmTextGetString(comment_text)
    );
}
