/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.gnu.gtk;

import java.util.Vector;

import org.gnu.glib.EventMap;
import org.gnu.gtk.event.ComboBoxEvent;
import org.gnu.gtk.event.ComboBoxListener;
import org.gnu.gtk.event.GtkEventType;

/**
 */
public class ComboBox extends Bin {
	
	public ComboBox() {
		if (-1 == handle) {
			handle = gtk_combo_box_new();
			initializeEventHandlers();
		}
	}
	
	public ComboBox(TreeModel model) {
		handle = gtk_combo_box_new_with_model(model.getHandle());
		initializeEventHandlers();
	}
	
	public void setWrapWidth(int width) {
		gtk_combo_box_set_wrap_width(handle, width);
	}
	
	public void setRowSpanColumn(int rowSpan) {
		gtk_combo_box_set_row_span_column(handle, rowSpan);
	}
	
	public void setColumnSpanColumn(int columnSpan) {
		gtk_combo_box_set_column_span_column(handle, columnSpan);
	}
	
	public int getActive() {
		return gtk_combo_box_get_active(handle);
	}
	
	public void setActive(int index) {
		gtk_combo_box_set_active(handle, index);
	}
	
	public TreeIter getActiveIter() {
		int [] iter = new int[1];
		boolean ret = gtk_combo_box_get_active_iter(handle, iter);
		if (ret)
			return new TreeIter(iter[0]);
		return null;
	}
	
	public void setActiveIter(TreeIter iter) {
		gtk_combo_box_set_active_iter(handle, iter.getHandle());
	}
	
	public TreeModel getModel() {
		return new TreeModel(gtk_combo_box_get_model(handle));
	}
	
	public void setModel(TreeModel model) {
		gtk_combo_box_set_model(handle, model.getHandle());
	}
	
	public void appendText(String text) {
		gtk_combo_box_append_text(handle, text);
	}
	
	public void insertText(int position, String text) {
		gtk_combo_box_insert_text(handle, position, text);
	}
	
	public void prependText(String text) {
		gtk_combo_box_prepend_text(handle, text);
	}
	
	public void gtk_combo_box_remove_text(int position) {
		gtk_combo_box_remove_text(handle, position);
	}

	/* **************************************
	 * EVENT LISTENERS
	 ****************************************/

	/**
	 * Listeners for handling dialog events
	 */
	private Vector cbListeners = null;

	/**
	 * Register an object to handle dialog events.
	 * @see ComboBoxListener
	 */
	public void addListener(ComboBoxListener listener) {
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(cbListeners, listener);
		if (i == -1) {
			if (null == cbListeners) {
				cbListeners = new Vector();
			}
			cbListeners.addElement(listener);
		}
	}
	
	/**
	 * Removes a listener
	 * @see #addListener(ComboBoxListener)
	 */
	public void removeListener(ComboBoxListener listener) {
		int i = findListener(cbListeners, listener);
		if (i > -1) {
			cbListeners.remove(i);
		}
		if (0 == cbListeners.size()) {
			cbListeners = null;
		}
	}

	protected void fireComboBoxEvent(ComboBoxEvent event) {
		if (null == cbListeners) {
			return;
		}
		int size = cbListeners.size();
		int i = 0;
		while (i < size) {
			ComboBoxListener cbl = (ComboBoxListener)cbListeners.elementAt(i);
			cbl.comboBoxEvent(event);
			i++;
		}
	}

	private void handleChanged() {
		fireComboBoxEvent(new ComboBoxEvent(this, ComboBoxEvent.Type.CHANGED));
	}

	protected void initializeEventHandlers() {
		super.initializeEventHandlers();
		evtMap.initialize(this);
	}

	public Class getEventListenerClass(String signal) {
		return evtMap.getListenerClass(signal);
	}

	public GtkEventType getEventType(String signal) {
		return evtMap.getEventType(signal);
	}

	private static EventMap evtMap = new EventMap();
	static {
		addEvents(evtMap);
	}

	/**
	 * Implementation method to build an EventMap for this widget class.
	 * Not useful (or supported) for application use.
	 */
	private static void addEvents(EventMap anEvtMap) {
		anEvtMap.addEvent("changed", "handleChanged", ComboBoxEvent.Type.CHANGED, ComboBoxListener.class);
	}

	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int gtk_combo_box_get_type ();
	native static final protected int gtk_combo_box_new();
	native static final protected int gtk_combo_box_new_with_model(int model);
	native static final protected void gtk_combo_box_set_wrap_width(int combo, int width);
	native static final protected void gtk_combo_box_set_row_span_column(int box, int rowSpan);
	native static final protected void gtk_combo_box_set_column_span_column(int box, int columnSpan);
	native static final protected int gtk_combo_box_get_active(int combo);
	native static final protected void gtk_combo_box_set_active(int combo, int index);
	native static final protected boolean gtk_combo_box_get_active_iter(int combo, int[] iter);
	native static final protected void gtk_combo_box_set_active_iter(int combo, int iter);
	native static final protected void gtk_combo_box_set_model(int combo, int model);
	native static final protected int gtk_combo_box_get_model(int combo);
	native static final protected int gtk_combo_box_new_text();
	native static final protected void gtk_combo_box_append_text(int combo, String text);
	native static final protected void gtk_combo_box_insert_text(int combo, int position, String text);
	native static final protected void gtk_combo_box_prepend_text(int combo, String text);
	native static final protected void gtk_combo_box_remove_text(int combo, int position);
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
}
