/******************************************************************************
 JXDocument.cc

	Maintains some information and a window that displays it.

	Derived classes must override OKToClose so they can save data
	or otherwise let the user clean things up before closing.  (Clean up
	that doesn't involve the user should be done in the destructor.)

	OKToDeactivate() is automatically wired to OKToClose() on the assumption
	that the user can't tell the difference (and shouldn't have to, either!)
	between closing a document and merely hiding its window.

	Derived classes must also override the following function:

		NeedsSave
			Returns kTrue if the data needs to be saved.

		SafetySave
			Save the data in a temporary file, in case the program crashes.
			No error reporting is allowed because one possibility is
			that the GUI itself has died.

		GetMenuIcon
			Return a small icon for the document.  This is used when building
			a menu of all the documents that are open.  JXFileDocument
			implements this to return JXGetPlainFileIcon().

			The result is JXPM instead of JXImage because JXImage is
			restricted to a single display.

	Derived classes may want to override the following functions:

		GetName
			Return a name for the document.  This is used when building
			a menu of all the documents that are open, so it should be
			unique (though it doesn't have to be).  The default implementation
			is to return the title of our window.  JXFileDocument overrides
			this to return the name of its file.

		NeedDocument
			Return kTrue if you need the given document to stay open.
			The document dependency graph must be acyclic.

	BASE CLASS = JXWindowDirector

	Copyright  1996-98 by John Lindal. All rights reserved.

 ******************************************************************************/

#include <JXDocument.h>
#include <JXDocumentManager.h>
#include <JXWindow.h>
#include <jXGlobals.h>
#include <JString.h>
#include <jAssert.h>

/******************************************************************************
 Constructor

 ******************************************************************************/

JXDocument::JXDocument
	(
	JXDirector* supervisor
	)
	:
	JXWindowDirector(supervisor)
{
	(JXGetDocumentManager())->DocumentCreated(this);
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JXDocument::~JXDocument()
{
	(JXGetDocumentManager())->DocumentDeleted(this);
}

/******************************************************************************
 Close

	Returns kTrue if the user ok's it and nobody needs us.

 ******************************************************************************/

JBoolean
JXDocument::Close()
{
	if (!OKToClose())
		{
		return kFalse;
		}

	if ((JXGetDocumentManager())->OKToCloseDocument(this))
		{
		return JXWindowDirector::Close();
		}
	else
		{
		if (NeedsSave())
			{
			DiscardChanges();
			}
		Deactivate();
		return JNegate(IsActive());
		}
}

/******************************************************************************
 Deactivate (virtual)

	The user can't tell the difference between hiding and closing.

 ******************************************************************************/

JBoolean
JXDocument::Deactivate()
{
	if (JXWindowDirector::Deactivate())
		{
		if (NeedsSave())
			{
			DiscardChanges();	// OKToDeactivate() returned kTrue
			}
		return kTrue;
		}
	else
		{
		return kFalse;
		}
}

/******************************************************************************
 OKToDeactivate (virtual protected)

	The user can't tell the difference between hiding and closing.

 ******************************************************************************/

JBoolean
JXDocument::OKToDeactivate()
{
	return OKToClose();
}

/******************************************************************************
 GetName (virtual)

	Returns a name for the document.  The default is to use the window title.

 ******************************************************************************/

JString
JXDocument::GetName()
	const
{
	return (GetWindow())->GetTitle();
}

/******************************************************************************
 NeedDocument (virtual)

	Derived classes should override if they require that other documents
	must stay open.

	*** The dependency graph must be acyclic!

 ******************************************************************************/

JBoolean
JXDocument::NeedDocument
	(
	JXDocument* doc
	)
	const
{
	return kFalse;
}

/******************************************************************************
 Cast to JXFileDocument*

	Not inline because they are virtual

 ******************************************************************************/

JXFileDocument*
JXDocument::CastToJXFileDocument()
{
	return NULL;
}

const JXFileDocument*
JXDocument::CastToJXFileDocument()
	const
{
	return NULL;
}
