/******************************************************************************
 JXEPSPrintSetupDialog.cc

	Clients must call BuildWindow() after constructing the object.
	Derived classes must override BuildWindow() and call SetObjects()
	at the end of their implementation.

	BASE CLASS = JXDialogDirector

	Copyright  1997-99 by John Lindal. All rights reserved.

 ******************************************************************************/

#include <JXEPSPrintSetupDialog.h>
#include <JXEPSPrinter.h>
#include <JXWindow.h>
#include <JXTextButton.h>
#include <JXStaticText.h>
#include <JXTextCheckbox.h>
#include <JXFileNameDisplay.h>
#include <jXGlobals.h>
#include <JString.h>
#include <jFileUtil.h>
#include <jDirUtil.h>
#include <jAssert.h>

/******************************************************************************
 Constructor

 ******************************************************************************/

JXEPSPrintSetupDialog::JXEPSPrintSetupDialog()
	:
	JXDialogDirector(JXGetApplication(), kTrue)
{
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JXEPSPrintSetupDialog::~JXEPSPrintSetupDialog()
{
}

/******************************************************************************
 BuildWindow

 ******************************************************************************/

void
JXEPSPrintSetupDialog::BuildWindow
	(
	const JCharacter*	fileName,
	const JBoolean		printPreview,
	const JBoolean		bw
	)
{
// begin JXLayout

    JXWindow* window = new JXWindow(this, 370,130, "");
    assert( window != NULL );
    SetWindow(window);

    itsChooseFileButton =
        new JXTextButton("Choose file", window,
                    JXWidget::kFixedLeft, JXWidget::kFixedTop, 20,20, 80,20);
    assert( itsChooseFileButton != NULL );
    itsChooseFileButton->SetShortcuts("#O");

    JXTextButton* okButton =
        new JXTextButton("Print", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 229,99, 72,22);
    assert( okButton != NULL );
    okButton->SetShortcuts("^M");

    JXTextButton* cancelButton =
        new JXTextButton("Cancel", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 80,100, 70,20);
    assert( cancelButton != NULL );

    itsBWCheckbox =
        new JXTextCheckbox("Black & White output", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 190,60, 150,20);
    assert( itsBWCheckbox != NULL );
    itsBWCheckbox->SetShortcuts("#B");

    itsPreviewCheckbox =
        new JXTextCheckbox("Include preview", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 30,60, 130,20);
    assert( itsPreviewCheckbox != NULL );
    itsPreviewCheckbox->SetShortcuts("#P");

    itsFileNameDisplay =
        new JXFileNameDisplay(fileName, window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 100,20, 250,20);
    assert( itsFileNameDisplay != NULL );

// end JXLayout

	SetObjects(okButton, cancelButton, itsFileNameDisplay, itsChooseFileButton,
			   itsPreviewCheckbox, printPreview, itsBWCheckbox, bw);
}

/******************************************************************************
 SetObjects (protected)

 ******************************************************************************/

void
JXEPSPrintSetupDialog::SetObjects
	(
	JXTextButton*		okButton,
	JXTextButton*		cancelButton,
	JXFileNameDisplay*	fileNameDisplay,
	JXTextButton*		chooseFileButton,
	JXTextCheckbox*		previewCheckbox,
	const JBoolean		printPreview,
	JXTextCheckbox*		bwCheckbox,
	const JBoolean		bw
	)
{
	itsPrintButton      = okButton;
	itsFileNameDisplay  = fileNameDisplay;
	itsChooseFileButton = chooseFileButton;
	itsPreviewCheckbox  = previewCheckbox;
	itsBWCheckbox       = bwCheckbox;

	(itsChooseFileButton->GetWindow())->SetTitle("Print Setup");
	SetButtons(okButton, cancelButton);

	ListenTo(itsChooseFileButton);

	itsPreviewCheckbox->SetState(printPreview);
	itsBWCheckbox->SetState(bw);

	if ((itsFileNameDisplay->GetText()).IsEmpty())
		{
		ChooseDestinationFile();
		}
}

/******************************************************************************
 OKToDeactivate (virtual protected)

 ******************************************************************************/

JBoolean
JXEPSPrintSetupDialog::OKToDeactivate()
{
	if (!JXDialogDirector::OKToDeactivate())
		{
		return kFalse;
		}
	else if (Cancelled())
		{
		return kTrue;
		}

	const JString& fullName = itsFileNameDisplay->GetText();
	if (fullName.IsEmpty())
		{
		(JGetUserNotification())->ReportError(
			"Please specify a destination file.");
		return kFalse;
		}

	JString path, fileName;
	JSplitPathAndName(fullName, &path, &fileName);
	const JBoolean fileExists = JFileExists(fullName);
	if (!JDirectoryExists(path))
		{
		(JGetUserNotification())->ReportError(
			"The specified directory does not exist.");
		return kFalse;
		}
	else if (!fileExists && !JDirectoryWritable(path))
		{
		(JGetUserNotification())->ReportError(
			"The file cannot be created because the you do not have write access "
			"to the specified directory.");
		return kFalse;
		}
	else if (fileExists && !JFileWritable(fullName))
		{
		(JGetUserNotification())->ReportError(
			"You do not have write access to the specified file.");
		return kFalse;
		}

	return kTrue;
}

/******************************************************************************
 Receive (protected)

 ******************************************************************************/

void
JXEPSPrintSetupDialog::Receive
	(
	JBroadcaster*	sender,
	const Message&	message
	)
{
	if (sender == itsChooseFileButton && message.Is(JXButton::kPushed))
		{
		ChooseDestinationFile();
		}
	else
		{
		JXDialogDirector::Receive(sender, message);
		}
}

/******************************************************************************
 ChooseDestinationFile (private)

 ******************************************************************************/

void
JXEPSPrintSetupDialog::ChooseDestinationFile()
{
	JString fileName;
	if ((JGetChooseSaveFile())->SaveFile("Save EPS file as:", NULL,
										 itsFileNameDisplay->GetText(), &fileName))
		{
		itsFileNameDisplay->SetText(fileName);
		itsPrintButton->Activate();
		}
	else if ((itsFileNameDisplay->GetText()).IsEmpty())
		{
		itsPrintButton->Deactivate();
		}
}

/******************************************************************************
 SetParameters

 ******************************************************************************/

void
JXEPSPrintSetupDialog::SetParameters
	(
	JXEPSPrinter* p
	)
	const
{
	p->SetOutputFileName(itsFileNameDisplay->GetText());
	p->PrintPreview(itsPreviewCheckbox->IsChecked());
	p->PSPrintBlackWhite(itsBWCheckbox->IsChecked());
}
