/******************************************************************************
 JXPartition.cc

	Base class for JX for a set of compartments arranged horizontally or vertically.

	This is not a JXScrollableWidget because adjustable partitions would
	be pointless in that case.  One only needs partitions when there is
	a limited amount of space available.

	This class implements CreateCompartmentObject() and DeleteCompartmentObject().
	Derived classes must implement GetTotalSize(), SetCompartmentSizes(),
	and the following routines:

		CreateCompartment
			Create a JXWidgetSet to be used as a compartment object.

	BASE CLASS = JXWidgetSet, JPartition

	Copyright  1996 by John Lindal. All rights reserved.

 ******************************************************************************/

#include <JXPartition.h>
#include <jAssert.h>

/******************************************************************************
 Constructor (protected)

	*** derived class must create compartments

 ******************************************************************************/

JXPartition::JXPartition
	(
	const JArray<JCoordinate>&	sizes,
	const JIndex				elasticIndex,
	const JArray<JCoordinate>&	minSizes,
	JXContainer*				enclosure,
	const HSizingOption			hSizing,
	const VSizingOption			vSizing,
	const JCoordinate			x,
	const JCoordinate			y,
	const JCoordinate			w,
	const JCoordinate			h
	)
	:
	JXWidgetSet(enclosure, hSizing, vSizing, x,y, w,h),
	JPartition(sizes, elasticIndex, minSizes)
{
	itsCompartments = new JPtrArray<JXContainer>;
	assert( itsCompartments != NULL );
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JXPartition::~JXPartition()
{
	delete itsCompartments;		// objects deleted by JXContainer
}

/******************************************************************************
 GetCompartmentIndex

	Returns kTrue if the given JXContainer is a compartment.

 ******************************************************************************/

JBoolean
JXPartition::GetCompartmentIndex
	(
	const JXContainer*	compartment,
	JIndex*				index
	)
	const
{
	*index = 0;

	const JSize compartmentCount = GetCompartmentCount();
	for (JIndex i=1; i<=compartmentCount; i++)
		{
		if (compartment == GetCompartment(i))
			{
			*index = i;
			return kTrue;
			}
		}

	return kFalse;
}

/******************************************************************************
 InsertCompartment

	Adjusts the adjacent compartments to make space and then creates a new
	compartment.  Returns kFalse if there isn't enough space available for
	at least minSize.

 ******************************************************************************/

JBoolean
JXPartition::InsertCompartment
	(
	const JIndex		index,
	const JCoordinate	reqSize,
	const JCoordinate	minSize,
	JXContainer**		newCompartment
	)
{
	if (JPartition::InsertCompartment(index, reqSize, minSize))
		{
		*newCompartment = GetCompartment(index);
		return kTrue;
		}
	else
		{
		*newCompartment = NULL;
		return kFalse;
		}
}

/******************************************************************************
 CreateCompartmentObject (virtual protected)

 ******************************************************************************/

void
JXPartition::CreateCompartmentObject
	(
	const JIndex		index,
	const JCoordinate	position,
	const JCoordinate	size
	)
{
	JXContainer* newCompartment = CreateCompartment(position, size);
	itsCompartments->InsertAtIndex(index, newCompartment);
}

/******************************************************************************
 DeleteCompartmentObject (virtual protected)

 ******************************************************************************/

void
JXPartition::DeleteCompartmentObject
	(
	const JIndex index
	)
{
	itsCompartments->DeleteElement(index);
}

/******************************************************************************
 ApertureResized (virtual protected)

	Adjust the sizes of the compartments.

 ******************************************************************************/

void
JXPartition::ApertureResized
	(
	const JCoordinate dw,
	const JCoordinate dh
	)
{
	JXWidgetSet::ApertureResized(dw,dh);
	PTBoundsChanged();
}
