/******************************************************************************
 JXScrollableWidget.h

	Interface for the JXScrollableWidget class

	Copyright  1996 by John Lindal. All rights reserved.

 ******************************************************************************/

#ifndef _H_JXScrollableWidget
#define _H_JXScrollableWidget

#include <JXWidget.h>

class JXScrollbar;
class JXScrollbarSet;
class JXAdjustScrollbarTask;

class JXScrollableWidget : public JXWidget
{
	friend class JXAdjustScrollbarTask;

public:

	JXScrollableWidget(JXScrollbarSet* scrollbarSet,
					   JXContainer* enclosure,
					   const HSizingOption hSizing, const VSizingOption vSizing,
					   const JCoordinate x, const JCoordinate y,
					   const JCoordinate w, const JCoordinate h);

	virtual ~JXScrollableWidget();

	void		AlwaysShowScrollbars(const JBoolean alwaysShow = kTrue);
	JBoolean	GetScrollbars(JXScrollbar** hScrollbar, JXScrollbar** vScrollbar) const;
	void		UpdateScrollbars();

	void		ReadScrollSetup(istream& input);
	static void	SkipScrollSetup(istream& input);
	void		WriteScrollSetup(ostream& output) const;

	virtual void	HandleKeyPress(const int key, const JXKeyModifiers& modifiers);

protected:

	JBoolean	ScrollForDrag(const JPoint& pt);
	JBoolean	ScrollForWheel(const JXMouseButton button,
							   const JXKeyModifiers& modifiers);

	void	SetHorizStepSize(const JCoordinate hStep);
	void	SetHorizPageStepContext(const JCoordinate hPageContext);
	void	SetVertStepSize(const JCoordinate vStep);
	void	SetVertPageStepContext(const JCoordinate vPageContext);

	virtual void	DrawBorder(JXWindowPainter& p, const JRect& frame);

	virtual void	HandleMouseDown(const JPoint& pt, const JXMouseButton button,
									const JSize clickCount,
									const JXButtonStates& buttonStates,
									const JXKeyModifiers& modifiers);

	virtual void	BoundsMoved(const JCoordinate dx, const JCoordinate dy);
	virtual void	BoundsResized(const JCoordinate dw, const JCoordinate dh);
	virtual void	ApertureMoved(const JCoordinate dx, const JCoordinate dy);
	virtual void	ApertureResized(const JCoordinate dw, const JCoordinate dh);

	virtual void	Receive(JBroadcaster* sender, const Message& message);

private:

	JXScrollbarSet*	itsScrollbarSet;		// can be NULL
	JBoolean		itsAlwaysShowScrollFlag;
	JBoolean		itsWindowFrozenFlag;
	JBoolean		itsAdjustingFlag;		// if should ignore scrollbar messages

	JXAdjustScrollbarTask*	itsAdjustScrollbarTask;

	// If these are positive, they override the defaults.

	JCoordinate	itsHStepSize;
	JCoordinate	itsVStepSize;

	// If these are non-negative, they override the defaults.

	JCoordinate	itsHPageStepContext;
	JCoordinate	itsVPageStepContext;

private:

	void	NeedAdjustScrollbars();
	void	AdjustScrollbars();

	// not allowed

	JXScrollableWidget(const JXScrollableWidget& source);
	const JXScrollableWidget& operator=(const JXScrollableWidget& source);
};

/******************************************************************************
 AlwaysShowScrollbars

 ******************************************************************************/

inline void
JXScrollableWidget::AlwaysShowScrollbars
	(
	const JBoolean alwaysShow
	)
{
	itsAlwaysShowScrollFlag = alwaysShow;
	AdjustScrollbars();
}

/******************************************************************************
 Step sizes

 ******************************************************************************/

inline void
JXScrollableWidget::SetHorizStepSize
	(
	const JCoordinate hStep
	)
{
	itsHStepSize = hStep;
	AdjustScrollbars();
}

inline void
JXScrollableWidget::SetHorizPageStepContext
	(
	const JCoordinate hPageContext
	)
{
	itsHPageStepContext = hPageContext;
	AdjustScrollbars();
}

inline void
JXScrollableWidget::SetVertStepSize
	(
	const JCoordinate vStep
	)
{
	itsVStepSize = vStep;
	AdjustScrollbars();
}

inline void
JXScrollableWidget::SetVertPageStepContext
	(
	const JCoordinate vPageContext
	)
{
	itsVPageStepContext = vPageContext;
	AdjustScrollbars();
}

#endif
