/******************************************************************************
 jXGlobals.cc

	Access to JX global objects and factories.

	Copyright  1997 John Lindal. All rights reserved.

 ******************************************************************************/

#include <jXGlobals.h>
#include <JXUserNotification.h>
#include <JXChooseSaveFile.h>
#include <JXCreatePG.h>
#include <JXGetCurrFontMgr.h>
#include <JXGetCurrColormap.h>
#include <JXHelpManager.h>
#include <JXDocumentManager.h>
#include <JXMDIServer.h>
#include <JXDisplay.h>
#include <JXAssert.h>
#include <X11/Xlib.h>
#include <jAssert.h>

static JXApplication*		theApplication    = NULL;	// owns itself
static JXChooseSaveFile*	theChooseSaveFile = NULL;	// owned by JCore
static JXAssert*			theAssertHandler  = NULL;	// owned by JCore

// we own these

static JXHelpManager*		theHelpManager    = NULL;	// can be NULL
static JXDocumentManager*	theDocManager     = NULL;	// can be NULL
static JXMDIServer*			theMDIServer      = NULL;	// can be NULL

/******************************************************************************
 JXCreateGlobals

 ******************************************************************************/

void
JXCreateGlobals
	(
	JXApplication* app
	)
{
	theApplication = app;

	theAssertHandler = new JXAssert;
	assert( theAssertHandler != NULL );

	JXUserNotification* un = new JXUserNotification;
	assert( un != NULL );

	theChooseSaveFile = new JXChooseSaveFile;
	assert( theChooseSaveFile != NULL );

	JInitCore(theAssertHandler, un, theChooseSaveFile,
			  new JXCreatePG, new JXGetCurrFontMgr, new JXGetCurrColormap,
			  "Helvetica", "Symbol", "Courier");

	XSetErrorHandler(JXDisplay::JXErrorHandler);
	XSetIOErrorHandler(JXApplication::JXIOErrorHandler);
}

/******************************************************************************
 JXCloseHelpManager

	Called before closing JXDisplays.

 ******************************************************************************/

void
JXCloseHelpManager()
{
	if (theHelpManager != NULL)
		{
		const JBoolean ok = theHelpManager->Close();
		assert( ok );

		theHelpManager = NULL;
		}
}

/******************************************************************************
 JXDeleteGlobals1

	Called after closing JXDisplays, before deleting idle and urgent tasks.

 ******************************************************************************/

void
JXDeleteGlobals1()
{
	delete theDocManager;
	theDocManager = NULL;

	delete theMDIServer;
	theMDIServer = NULL;
}

/******************************************************************************
 JXDeleteGlobals2

	Called after deleting idle and urgent tasks.

 ******************************************************************************/

void
JXDeleteGlobals2()
{
	theChooseSaveFile = NULL;
	theAssertHandler  = NULL;
	theApplication    = NULL;

	JDeleteGlobals();
}

/******************************************************************************
 JXGetApplication

 ******************************************************************************/

JXApplication*
JXGetApplication()
{
	assert( theApplication != NULL );
	return theApplication;
}

// called by JXDisplay

JBoolean
JXGetApplication
	(
	JXApplication** app
	)
{
	*app = theApplication;
	return JConvertToBoolean( theApplication != NULL );
}

/******************************************************************************
 JXGetChooseSaveFile

 ******************************************************************************/

JXChooseSaveFile*
JXGetChooseSaveFile()
{
	assert( theChooseSaveFile != NULL );
	return theChooseSaveFile;
}

/******************************************************************************
 JXGetAssertHandler

 ******************************************************************************/

JXAssert*
JXGetAssertHandler()
{
	assert( theAssertHandler != NULL );
	return theAssertHandler;
}

/******************************************************************************
 JXGetDocumentManager

	This function calls assert() if JXSetDocumentManager() has not been called.
	We do this because JXDocuments cannot be used otherwise.

 ******************************************************************************/

JXDocumentManager*
JXGetDocumentManager()
{
	assert( theDocManager != NULL );
	return theDocManager;
}

// called by JXApplication, JXDisplay

JBoolean
JXGetDocumentManager
	(
	JXDocumentManager** docManager
	)
{
	*docManager = theDocManager;
	return JConvertToBoolean( theDocManager != NULL );
}

/******************************************************************************
 JXSetDocumentManager

	The application code must call this exactly once if it uses JXDocument.
	We don't create JXDocumentManager automatically because one may need
	a derived class.

	After calling this function, we own the object and will delete it when
	the program terminates.

 ******************************************************************************/

void
JXSetDocumentManager
	(
	JXDocumentManager* docManager
	)
{
	assert( theDocManager == NULL && docManager != NULL );
	theDocManager = docManager;
}

/******************************************************************************
 JXCreateDefaultDocumentManager

	This is provided for convenience for all the programs that do
	not need a derived class of JXDocumentManager.

 ******************************************************************************/

void
JXCreateDefaultDocumentManager
	(
	const JXDocumentManager::ShortcutStyle	style,
	const JBoolean							useShortcutZero
	)
{
	assert( theDocManager == NULL );
	JXDocumentManager* docManager = new JXDocumentManager(style, useShortcutZero);
	assert( docManager != NULL );
	// constructor calls JXSetDocumentManager()
}

/******************************************************************************
 JXGetHelpManager

	This function calls assert() if JXInitHelp() has not been called.
	We do this because the help system cannot be used otherwise.

 ******************************************************************************/

JXHelpManager*
JXGetHelpManager()
{
	assert( theHelpManager != NULL );
	return theHelpManager;
}

/******************************************************************************
 JXInitHelp

	Create JXHelpManager.  Refer to the JXHelpManager documentation
	for RegisterSection() for information about the arguments to this
	function.

	tocSectionName can be NULL.

 ******************************************************************************/

void
JXInitHelp
	(
	const JCharacter*			tocSectionName,
	const JSize					sectionCount,
	const JCharacter*			sectionName[],
	const JCharacter*			sectionTitle[],
	const JCharacter*			sectionText[],
	const JXMenu::Style			menuStyle
	)
{
	assert( theHelpManager == NULL );
	assert( sectionCount > 0 );

	theHelpManager = new JXHelpManager(menuStyle);
	assert( theHelpManager != NULL );

	theHelpManager->SetTOCSectionName(tocSectionName);

	for (JIndex i=0; i<sectionCount; i++)
		{
		theHelpManager->RegisterSection(sectionName[i], sectionTitle[i],
										sectionText[i]);
		}
}

/******************************************************************************
 JXGetMDIServer

 ******************************************************************************/

JBoolean
JXGetMDIServer
	(
	JXMDIServer** server
	)
{
	*server = theMDIServer;
	return JConvertToBoolean( theMDIServer != NULL );
}

/******************************************************************************
 JXSetMDIServer

	The application code must call this exactly once if it supports MDI.

	After calling this function, we own the object and will delete it when
	the program terminates.

 ******************************************************************************/

void
JXSetMDIServer
	(
	JXMDIServer* server
	)
{
	assert( theMDIServer == NULL && server != NULL );
	theMDIServer = server;
}

/******************************************************************************
 JXGetHelveticaFontName

 ******************************************************************************/

const JCharacter*
JXGetHelveticaFontName()
{
	return "Helvetica";
}

/******************************************************************************
 JXGetTimesFontName

 ******************************************************************************/

const JCharacter*
JXGetTimesFontName()
{
	return "Times";
}

/******************************************************************************
 JXGetCourierFontName

 ******************************************************************************/

const JCharacter*
JXGetCourierFontName()
{
	return "Courier";
}

/******************************************************************************
 JXGetSymbolFontName

 ******************************************************************************/

const JCharacter*
JXGetSymbolFontName()
{
	return "Symbol";
}
