/*
 * Copyright (c) 2001, Gerald L. Gay
 *
 * You may distribute this source code under the terms of the license
 * in the COPYING file.
 *
 */

#include <Xarm/XarmInput.h>
#include <Xarm/AppContext.h>
#include <Xarm/PushB.h>
#include <Xarm/Form.h>
#include <Xarm/Separator.h>
#include <Xarm/Callback.h>
#include <Xarm/MessageB.h>
#include <string.h>

#ifdef XARM_XBAE_HAS_INPUT

_XtString fallbacks[] = {
    "*banner*fontList: -*-times-bold-r-normal--14-*-*-*-*-*-*-*",
    NULL
};

class Application : public AppContext
{
private:

    Input         *ssnField;
    Input         *phoneField;
    Label         *ssnLabel;
    Label         *phoneLabel;
    Label         *banner;
    PushButton    *okButton;
    Form          *myForm;
    Separator     *sep1;
    Separator     *sep2;
    MessageDialog *msg;

    bool           isSSNValid;
    bool           isPhoneValid;

    void exit(Widget, XtPointer, XtPointer);
    void onExpose(Widget, XtPointer, XEvent *, Boolean *);
    void onOK(Widget, XtPointer, XtPointer);
    void validate(Widget, XtPointer, XtPointer);

public:

    Application(char *app_class, int &argc_in_out, char **argv_in_out);
};

int main(int argc, char *argv[])
{
   // initialize application
   Application app("XarmInput", argc, argv);

   // realize application
   app.realize();

   // start the main loop
   app.mainLoop();

   // NOTE: mainLoop() *WILL* return!

   return EXIT_SUCCESS;
}

Application::Application(char *app_class,int &ac_io, char **av_io) :
    AppContext(app_class, NULL, 0, ac_io, av_io, fallbacks)
{

    isSSNValid = isPhoneValid = false;
    msg = NULL;

    // install window manager message handler(s)
    Atom proto = addWMProtocol(XA_WM_DELETE_WINDOW);
    addWMProtocolCallback(this, &Application::exit, this, proto);

    title("Xbae Input Demo");

    // install window(s) into the application window
    myForm = new Form("myForm", *this);

    XarmArg args;

    // A Banner Label at the top...
    args.reset();
    args(XmNtopAttachment,   XmATTACH_FORM)
        (XmNtopOffset,       5)
        (XmNleftAttachment,  XmATTACH_FORM)
        (XmNleftOffset,      5)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     5);

    banner = new Label(*myForm, args, "banner");
    banner->labelString("Xbae Input Widget Demo");

    args.reset();
    args(XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       banner)
        (XmNtopOffset,       5)
        (XmNleftAttachment,  XmATTACH_FORM)
        (XmNleftOffset,      2)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     2);

    sep1 = new Separator(*myForm, args, "sep1");

    // An input field that accepts only valid SSN's
    args.reset();
    args(XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       sep1)
        (XmNtopOffset,       5)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     5)
        (XmNautoFill,        True)
        (XmNpattern,         "ddd-dd-dddd");

    ssnField = new Input(*myForm, args, "ssnField");

    addCallback(this, &Application::validate,
                *ssnField, XmNvalidateCallback);

    args.reset();
    args(XmNtopAttachment,    XmATTACH_OPPOSITE_WIDGET)
        (XmNtopWidget,        ssnField)
        (XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET)
        (XmNbottomWidget,     ssnField)
        (XmNrightAttachment,  XmATTACH_WIDGET)
        (XmNrightWidget,      ssnField)
        (XmNrightOffset,      5)
        (XmNleftAttachment,   XmATTACH_FORM)
        (XmNleftOffset,       5);

    ssnLabel = new Label(*myForm, args, "ssnLabel");
    ssnLabel->labelString("Enter a valid US Social Security #:");

    args.reset();
    args(XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       ssnField)
        (XmNtopOffset,       10)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     5)
        (XmNautoFill,        True)
        (XmNpattern,         "(ddd) ddd-dddd");

    phoneField = new Input(*myForm, args, "phoneField");

    addCallback(this, &Application::validate,
                *phoneField, XmNvalidateCallback);

    args.reset();
    args(XmNtopAttachment,    XmATTACH_OPPOSITE_WIDGET)
        (XmNtopWidget,        phoneField)
        (XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET)
        (XmNbottomWidget,     phoneField)
        (XmNrightAttachment,  XmATTACH_WIDGET)
        (XmNrightWidget,      phoneField)
        (XmNrightOffset,      5)
        (XmNleftAttachment,   XmATTACH_FORM)
        (XmNleftOffset,       5);

    phoneLabel = new Label(*myForm, args, "ssnLabel");
    phoneLabel->labelString("Enter a valid 10-digit US Phone #:");

    args.reset();
    args(XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       phoneField)
        (XmNtopOffset,       5)
        (XmNleftAttachment,  XmATTACH_FORM)
        (XmNleftOffset,      2)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     2);

    sep2 = new Separator(*myForm, args, "sep2");

    args.reset();
    args(XmNtopAttachment,    XmATTACH_WIDGET)
        (XmNtopWidget,        sep2)
        (XmNtopOffset,        5)
        (XmNleftAttachment,   XmATTACH_FORM)
        (XmNbottomAttachment, XmATTACH_FORM)
        (XmNbottomOffset,     5);

    okButton = new PushButton(*myForm, args, "okButton");
    okButton->labelString("  OK  ");
    myForm->defaultButton(*okButton);

    addCallback(this, &Application::onOK, *okButton, XmNactivateCallback);
    
    addEventHandler(this, &Application::onExpose,
                    *myForm, ExposureMask, false);

    myForm->manage();
}

void Application::exit(Widget, XtPointer, XtPointer)
{
    quit();
}

void Application::validate(Widget w, XtPointer, XtPointer cdata)
{

    //
    // This validate function is necessary to keep the Input
    // widget happy when losing focus. It always beeps at
    // me if I don't have a validate function defined.
    //
    // Also, we use it to remember the last state of the text
    // automagically without checking when the user clicks OK.
    // We don't have to check the actual data, since the widget
    // will do that for us. All we have to do is make sure that
    // the length is right.

    XbaeInputValidateCallbackStruct *vcs =
        (XbaeInputValidateCallbackStruct *)cdata;

    int len = strlen(vcs->value);

    if (w == ssnField->widget()) {
        if (len == 11) isSSNValid = true;
        else           isSSNValid = false;
    } else {
        if (len == 14) isPhoneValid = true;
        else           isPhoneValid = false;
    }

    vcs->doit = True;
}

void Application::onExpose(Widget, XtPointer, XEvent *, Boolean *)
{

    // Another thing that always seems to be difficult:
    // I want my button to be centered horizontally on the
    // form. But I don't want to fix the size of the window
    // or the size of the button because I might want to change
    // fonts or label strings or whatever. So what I do is wait
    // for an expose event, which means Xt is done calculating
    // everything, and figure out for myself where the button
    // needs to be. If anyone knows an easier way to do this,
    // please let me know!

    // We don't want the dimensions of the button to change
    // so on the first expose event, we get them and remember
    // them.

    static int buttonWidth  = -1;
    static int buttonHeight = -1;

    if (buttonWidth == -1) {
        buttonWidth  = okButton->width();
        buttonHeight = okButton->height();
        // processTraversal() needs to be called after
        // everyone has been managed. This is a convenient
        // place to stick it!
        ssnField->processTraversal();
    }

    int oldLeftOff  = okButton->leftOffset();
    int formWidth   = myForm->width();

    int newLeftOff  = (formWidth - buttonWidth) / 2;

    if (newLeftOff != oldLeftOff) {
        okButton->leftOffset(newLeftOff);
        okButton->width(buttonWidth);
    }

    // Do the same thing for the top and bottom, except that
    // the position calculation is based on the separator and
    // the bottom of the form. We start out with a spacing of 5.
    // This is OK because the height can fluctuate without any
    // unpleasant visual effects.

    int totalHeight = myForm->height() - sep2->y() - sep2->height();
    int oldTopOff   = okButton->topOffset();

    int newTopOff   = (totalHeight - buttonHeight) / 2;

    if (newTopOff != oldTopOff) {
        okButton->topOffset(newTopOff);
        okButton->bottomOffset(newTopOff);
        okButton->height(buttonHeight);
    }
}

void Application::onOK(Widget w, XtPointer u, XtPointer c)
{
    if (isSSNValid && isPhoneValid) {
        exit(w, u, c);
        return;
    }

    char *reason;

    if (isSSNValid && !isPhoneValid)
        reason = "The Phone Number is invalid.\n"
                 "It should be of the form:\n"
                 "(NNN) NNN-NNNN";
    else if (!isSSNValid && isPhoneValid)
        reason = "The Social Security Number is invalid.\n"
                 "It should be of the form:\n"
                 "NNN-NN-NNNN";
    else
        reason = "Both fields are invalid.\n"
                 "SSN should be like: NNN-NN-NNNN\n"
                 "and Phone should be: (NNN) NNN-NNNN";

    if (msg == NULL) {
        msg = new MessageDialog(*this, "");
        msg->dialogTitle("Data Validation Error");
        XtUnmanageChild(msg->getChild(XmDIALOG_CANCEL_BUTTON));
        XtUnmanageChild(msg->getChild(XmDIALOG_HELP_BUTTON));
    }

    msg->messageString(reason);
    msg->manage();
}

#else

#ifdef XARM_HAS_NEWHEADERS
#include <iostream>
#else
#include <iostream.h>
#endif

#ifdef XARM_HAS_NAMESPACES
using namespace std;
#endif

int main()
{
    cout << "Sorry, your version of Xarm wasn't built with Xbae Input support." << endl;

    return 0;
}

#endif
